﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace wintree7
{
    class WinTreeCreator
    {
        private WinTreeSettings settings = new WinTreeSettings();
        int totalDirectoryCount = 0;
        int totalFileCount = 0;
        long totalFileSize = 0;
        string vertical;
        string branch;
        string tip;
        string space;

        public WinTreeCreator(DirectoryInfo targetDirectory, FileInfo saveFile)
        {
            try
            {
                if (settings.Style.boarderWidthType == WinTreeSettingsDisplayStyle.BoaderWidthType.Full)
                {
                    vertical = settings.Style.fullWidth.vertical;
                    branch = settings.Style.fullWidth.branch;
                    tip = settings.Style.fullWidth.tip;
                    space = settings.Style.fullWidth.whiteSpace;
                }
                else
                {
                    vertical = settings.Style.halfWidth.vertical;
                    branch = settings.Style.halfWidth.branch;
                    tip = settings.Style.halfWidth.tip;
                    space = settings.Style.halfWidth.whiteSpace;
                }

                using (TextWriter tw = saveFile.CreateText())
                {
                    ExportFileHeader(targetDirectory, tw);
                    CreateDirectoryLists(targetDirectory, 0, "", tw);
                    ExportFileFooter(targetDirectory, tw);
                }
            }
            catch (Exception err)
            {
                System.Windows.Forms.MessageBox.Show(
                    err.Message, "wintree7",
                    System.Windows.Forms.MessageBoxButtons.OK, 
                    System.Windows.Forms.MessageBoxIcon.Error);
            }
        }

        private void ExportFileHeader(DirectoryInfo targetDirectory, TextWriter writer)
        {
            writer.WriteLine(
                "WinTree7 Ver "
                + System.Diagnostics.FileVersionInfo.GetVersionInfo(System.Reflection.Assembly.GetExecutingAssembly().Location).FileVersion
                + "                                       " /// @todo 桁数に合うようにスペースを追加する
                + DateTime.Now.ToString());

            DriveInfo targetDrive = new DriveInfo(targetDirectory.Root.Name);
            System.Management.ManagementObject mo = new System.Management.ManagementObject("Win32_LogicalDisk=\"" + targetDrive.Name.Trim('\\') + "\"");

            writer.WriteLine(
                "ドライブ " + targetDrive.Name
                + " のボリュームラベルは" + (targetDrive.VolumeLabel == "" ? "ありません。" : " " + targetDrive.VolumeLabel)
                + " (" + mo.Properties["VolumeSerialNumber"].Value.ToString().Substring(0, 4) 
                + "-" + mo.Properties["VolumeSerialNumber"].Value.ToString().Substring(4, 4) + ")"
                );
            writer.WriteLine(
                "フォルダは "
                + targetDirectory.FullName
                );
            writer.WriteLine(); //空行
            writer.WriteLine(
                "ファイル／フォルダ名"
                + "                         " /// @todo 桁数に合うようにスペースを追加する
                + "バイト数 属性   更新日付 更新時刻"
                );
            writer.WriteLine(new string(' ', settings.Style.colomnLimit / 2).Replace(" ", settings.Style.fullWidth.rule));
        }

        private void ExportFileFooter(DirectoryInfo targetDirectory, TextWriter writer)
        {
            writer.WriteLine(new string(' ', settings.Style.colomnLimit / 2).Replace(" ", settings.Style.fullWidth.rule));

            string format;
            if (totalDirectoryCount == 0 && totalFileCount == 0)
            {
                format = "ファイルはありません。";
            }
            else if (totalDirectoryCount != 0 && totalFileCount == 0)
            {
                format = "{0} 個のフォルダがあります。";
            }
            else if (totalDirectoryCount == 0 && totalFileCount != 0)
            {
                format = "{1} 個 {2:N0} Byte のファイルがあります。";
            }
            else
            {
                format = "{0} 個のフォルダと {1} 個 {2:N0} Byte のファイルがあります。";
            }
            writer.WriteLine(format, totalDirectoryCount, totalFileCount, totalFileSize);

            DriveInfo targetDrive = new DriveInfo(targetDirectory.Root.Name);
            float remainSpacePercent = (float)(targetDrive.TotalFreeSpace) / (float)(targetDrive.TotalSize);
            System.Globalization.NumberFormatInfo nfmt = new System.Globalization.NumberFormatInfo();
            nfmt.PercentDecimalDigits = (remainSpacePercent >= 0.10 ? 1 : 2);
            nfmt.PercentSymbol = "％";

            writer.WriteLine("ディスク空き容量は {0:N0} Byte ({1}) あります。", targetDrive.TotalFreeSpace, remainSpacePercent.ToString("P", nfmt));
        }

        private void CreateDirectoryLists(DirectoryInfo currentDirectory, int depth, string parentRule, TextWriter writer)
        {
            // 自分自身を出力
            string fmt = ((depth != 0) ? parentRule.Replace(branch, vertical) + branch : "");
            writer.WriteLine(fmt + "{0}", currentDirectory.Name);

            if (settings.Other.searchRecursive)
            {
                DirectoryInfo[] subDirectory = currentDirectory.GetDirectories();
                Array.Sort<DirectoryInfo>(subDirectory, (a, b) => a.Name.CompareTo(b.Name)); //FAT32対策にソート

                foreach (DirectoryInfo sub in subDirectory)
                {
                    CreateDirectoryLists(sub, depth + 1, fmt, writer);
                }
                totalDirectoryCount += subDirectory.Length;
            }

            CreateFileList(currentDirectory, depth + 1, fmt, writer);
        }

        private void CreateFileList(DirectoryInfo currentDirectory, int depth, string parentRule, TextWriter writer)
        {
            FileInfo[] files = currentDirectory.GetFiles();
            switch (settings.Order.orderType)
            {
                case WinTreeSettingsDisplayOrder.DisplayOrderType.FileName:
                    if (settings.Order.orderByFileNameAscending)
                        Array.Sort<FileInfo>(files, (a, b) => a.Name.CompareTo(b.Name));
                    else
                        Array.Sort<FileInfo>(files, (a, b) => b.Name.CompareTo(a.Name));
                    break;
                case WinTreeSettingsDisplayOrder.DisplayOrderType.Extention:
                    break;
                case WinTreeSettingsDisplayOrder.DisplayOrderType.Size:
                    if (settings.Order.orderByFileSizeSmallToLarge)
                        Array.Sort<FileInfo>(files, (a, b) => (int)(a.Length - b.Length));
                    else
                        Array.Sort<FileInfo>(files, (a, b) => (int)(b.Length - a.Length));
                    break;
                case WinTreeSettingsDisplayOrder.DisplayOrderType.CreateDate:
                    if (settings.Order.orderByFileNewToOld)
                        Array.Sort<FileInfo>(files, (a, b) => a.CreationTime.CompareTo(b.CreationTime));
                    else
                        Array.Sort<FileInfo>(files, (a, b) => b.CreationTime.CompareTo(a.CreationTime));
                    break;
                case WinTreeSettingsDisplayOrder.DisplayOrderType.UpdateDate:
                    if (settings.Order.orderByFileNewToOld)
                        Array.Sort<FileInfo>(files, (a, b) => a.LastWriteTime.CompareTo(b.LastWriteTime));
                    else
                        Array.Sort<FileInfo>(files, (a, b) => b.LastWriteTime.CompareTo(a.LastWriteTime));
                    break;
            }

            int fCount = 0;
            foreach (FileInfo fi in files)
            {
                string fmt = parentRule.Replace(branch, vertical).Replace(tip, space) + ((fCount != files.Length - 1) ? branch : tip);
                writer.WriteLine(fmt + "{0}", fi.Name);
                totalFileSize += fi.Length;
                fCount++;
            }

            totalFileCount += files.Length;
        }
    }
}
