/* File: cmd5.c */

/* Purpose: Spell/Prayer commands */

/*
 * Copyright (c) 1989 James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research, and
 * not for profit purposes provided that this copyright and statement are
 * included in all such copies.
 */

#include "angband.h"

#define tval2realm(A) ((A) - TV_LIFE_BOOK + 1)

static const char *spell_tips[MAX_REALM][32] =
{
#ifdef JP
	{
		"᤯μٰʥ󥹥Τ롣",
		"Ϥ򾯤롣",
		"֡̿ΨAC˥ܡʥ롣",
		"ݤ",
		"Ȥ餷ƤϰϤΤʵפ뤯롣",
		"᤯Ƥ櫤ȳʤΤ롣",
		"Ϥٲ롣",
		"ʢˤ롣",
		"ġ󥹥˸̤롣",
		"ƥˤä夤롣",

		"Ϥ˲ۯ۰֤롣",
		"ʤϤġٰʥ󥹥Ф礭ʥ᡼Ϳ뤬ɤʥ󥹥ˤϸ̤ʤ",
		"դϷΤ᤯櫡⡢ʡƤθ󥹥Τ롣",
		"֡ꡢ䵤Ф롣ˤѤ롣",
		"󥹥1Τ̥λ롣񹳤̵",
		"ٰʥ󥹥ιɤХꥢĥ롣",
		"ˤƶϤʲʸǡۯ۰֤롣",
		"볦Ƥμٰʥ󥹥ƥݡȤ롣񹳤̵",
		"εġ",
		"ʬΤ뾲ξˡ󥹥̤ȴ꾤줿ꤹ뤳ȤǤʤʤ롼",

		"֡ҡʬˤʤ롣",
		"ƥˤäϤʼ롣",
		"֡ŷ⡢ꡢ䵤ǤФ롣ˤѤ롣",
		"볦Ƥμٰʥ󥹥˥᡼Ϳ롣",
		"볦ƤΥ󥹥̥λ롣񹳤̵",
		"볦μٰ¸ߤ礭ʥ᡼ͿϤǡݡۯ۰֡롣",
		"ʡ롣ϤʼΤäˤ񹳤롣ƥեȤʡ褦ȤƼԤ롣",
		"٤ƤΥơȷиͤ롣",
		"ʬ濴Ȥεȯ롣ˡγΤʵפ˾Ȥ餷󥸥⤹٤ƤΥƥΤ롣",
		"ƥλǽϤΤ롣",
		"Ƕμˡǡۯ۰֤롣",
		"ܤ󥹥ʤ᡼Ϳ볦Υ󥹥˥᡼®ۯ۰𡢶ݡ̲Ϳ롣Ϥݤֶβ®롣",
	},
	{
		"夤ˡġ",
		"᤯Ƥθ󥹥Τ롣",
		"ΥΥƥݡȤ򤹤롣",
		"᤯Ƥ櫤Τ롣",
		"Ȥ餷ƤϰϤΤʵפ뤯롣",
		"ľƤ櫤˲롣",
		"ΥΥƥݡȤ򤹤롣",
		"ʢˤʤ롣",
		"ˡεġ",
		"ϾˤȤϥ󥸥κǿء󥸥ˤȤϾؤȰư롣",

		"ɤϤƾˤ롣",
		"/ˡνŶ䤹ŶΥåɤνŶ֤򸺤餹",
		"դϷΤ롣",
		"ƥ1ļ̤롣٥뤬⤤ȥƥǽϤΤ뤳ȤǤ롣",
		"֡ƥѥǽϤ롣",
		"ϤΥӡġ",
		"֡®롣",
		"󥹥1ΤƥݡȤ롣񹳤̵",
		"åȤȯͤ롣",
		"ꤷʸΥ󥹥򸽺ߤγä롣񹳤̵",

		"󥹥λĤϡϡԡɤΤ롣",
		"󥹥1μʬζ᤯˰󤻤롣",
		"ֻ˾夫γ˥ƥݡȤ롣",
		"θġ",
		"᤯ƤΥ󥹥櫡⡢ʡƥƥΤ롣",
		"դΥƥࡢ󥹥Ϸ˲롣",
		"γΤʵפ˾Ȥ餷󥸥⤹٤ƤΥƥΤ롣ˡ֥ƥѥǽϤ롣",
		"ûΥλꤷ˥ƥݡȤ롣",
		"ƥ1Ĥ򤪶Ѥ롣",
		"ʬμϤˤ󥹥򸽺ߤγä롣񹳤̵",
		"˶ϤǵʽϤεġ",
		"֡ɤ̤ȴ뤳ȤǤ᡼ڸͩΤξ֤ѿȤ롣"
	},
	{
		"֡Ϥ롣",
		"3Фƹ⤹롣",
		"֡ҡʬˤʤ롣",
		"ꤷ˸äƹ®ưԤ",
		"֡ŷ⡢ꡢ䵤Ф롣ˤѤ롣",
		"֡볦ƤΥ󥹥ǧǤ롣",
		"⤷塢ȿ¦ȴ롣",
		"եΥܥȤġ",
		"֡AC徺롣",
		"ƥˤä夤롣",

		"βݤ롣",
		"ϤΥ󥸥ɤ餷ɤȾѤ롣",
		"֡ŷ⡢ꡢ䵤ǤФ롣ˤѤ롣",
		"//ܥȤ򶯤Ϥȯͤ롣",
		"ڤ̣Τ빶򷫤Ф",
		"֡®롣",
		"Фƹ⤹롣",
		"ƥŷ⡦бꡦ䵤ǽĤʤ褦ù롣",
		"ߤΥӡġ",
		"֡ˡɸϤAC夬ꡢܤȿǽϡΤ餺ͷ롣",

		"ɶˤ줿Ϥ˲롣",
		"᤯˶ꡢ򷫤Ф",
		"ٰʥ󥹥ιɤХꥢĥ롣",
		"βݤ®롣",
		"󥹥ۯ۰Ȥݤ빶򤹤롣",
		"̿Ψȥ᡼򶯲롣",
		"ɸ潤򶯲롣",
		"̵ŨΥХꥢڤԤ",
		"12ٹԤ",
		"ߤΥ֥쥹ƱιԤ",
		"Ǽ٤°Ĥ롣",
		"֡դACˡɸǽϤ徺롣"
	},
#else
	{
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
	},
	{
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
	},
	{
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",

		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
		"yet.",
	},
#endif
};

byte get_modified_smana(magic_type *s_ptr)
{
	int tmp_mana;

	if (!s_ptr) return 0;
	tmp_mana = s_ptr->smana;

	if (p_ptr->dec_mana) tmp_mana = tmp_mana * 3 / 4;
	if (tmp_mana < 1) tmp_mana = 1;

	return (byte)tmp_mana;
}

/*
 * Allow user to choose a spell/prayer from the given book.
 *
 * If a valid spell is chosen, saves it in '*sn' and returns TRUE
 * If the user hits escape, returns FALSE, and set '*sn' to -1
 * If there are no legal choices, returns FALSE, and sets '*sn' to -2
 *
 * The "prompt" should be "cast", "recite", or "study"
 * The "known" should be TRUE for cast/pray, FALSE for study
 */

bool select_spellbook = FALSE;

static int get_spell(int *sn, cptr prompt, int sval, bool known, int use_realm)
{
	int         i;
	int         spell = -1;
	int         num = 0;
	int         ask;
	byte        spells[32];
	bool        flag, redraw, okay;
	char        choice;
	magic_type  *s_ptr;
	char        out_val[160];
#ifdef JP
	char        jverb_buf[128];
	cptr        p = ((mp_ptr->spell_type == ST_PRAYER) ? "" : "ʸ");
#else
	cptr        p = ((mp_ptr->spell_type == ST_PRAYER) ? "prayer" : "spell");
#endif

#ifdef ALLOW_REPEAT /* TNB */

	/* Get the spell, if available */
	if (repeat_pull(sn))
	{
		/* Verify the spell */
		if (spell_okay(*sn, known, use_realm - 1))
		{
			/* Success */
			return (TRUE);
		}
	}

#endif /* ALLOW_REPEAT -- TNB */

	/* Extract spells */
	for (spell = 0; spell < 32; spell++)
	{
		/* Check for this spell */
		if ((fake_spell_flags[sval] & (1L << spell)))
		{
			/* Collect this spell */
			spells[num++] = spell;
		}
	}

	/* Assume no usable spells */
	okay = FALSE;

	/* Assume no spells available */
	(*sn) = -2;

	/* Check for "okay" spells */
	for (i = 0; i < num; i++)
	{
		/* Look for "okay" spells */
		if (spell_okay(spells[i], known, use_realm - 1)) okay = TRUE;
	}

	/* No "okay" spells */
	if (!okay) return (FALSE);

	/* Assume cancelled */
	*sn = (-1);

	/* Nothing chosen yet */
	flag = FALSE;

	/* No redraw yet */
	redraw = FALSE;

	/* Show choices */
	if (show_choices)
	{
		/* Update */
		p_ptr->window |= (PW_SPELL);

		/* Window stuff */
		window_stuff();
	}


	/* Build a prompt (accept all spells) */
#ifdef JP
	jverb( prompt, jverb_buf, JVERB_AND );
	(void) strnfmt(out_val, 78, "(%^s:%c-%c, '*'ǰ, ESC) ɤ%s%^sޤ? ",
		p, I2A(0), I2A(num - 1), p, jverb_buf );
#else
	(void)strnfmt(out_val, 78, "(%^ss %c-%c, *=List, ESC=exit) %^s which %s? ",
		p, I2A(0), I2A(num - 1), prompt, p);
#endif

	/* Get a spell from the user */
	choice = always_show_list ? ESCAPE : 1;
	while (!flag)
	{
		if( choice==ESCAPE ) choice = ' '; 
		else if( !get_com(out_val, &choice) )break; 

		/* Request redraw */
		if ((choice == ' ') || (choice == '*') || (choice == '?'))
		{
			/* Show the list */
			if (!redraw)
			{
				/* Show list */
				redraw = TRUE;

				/* Save the screen */
				screen_save();

				/* Display a list of spells */
				print_spells(spells, num, 1, 20, use_realm - 1);
			}

			/* Hide the list */
			else
			{
				/* Hide list */
				redraw = FALSE;

				/* Restore the screen */
				screen_load();
			}

			/* Redo asking */
			continue;
		}


		/* Note verify */
		ask = (isupper(choice));

		/* Lowercase */
		if (ask) choice = tolower(choice);

		/* Extract request */
		i = (islower(choice) ? A2I(choice) : -1);

		/* Totally Illegal */
		if ((i < 0) || (i >= num))
		{
			bell();
			continue;
		}

		/* Save the spell index */
		spell = spells[i];

		/* Require "okay" spells */
		if (!spell_okay(spell, known, use_realm - 1))
		{
			bell();
#ifdef JP
			msg_format("%s%sȤϤǤޤ", p, prompt);
#else
			msg_format("You may not %s that %s.", prompt, p);
#endif
			continue;
		}

		/* Verify it */
		if (ask)
		{
			char tmp_val[160];

			/* Access the spell */
			s_ptr = &mp_ptr->info[use_realm - 1][spell % 32];

			/* Prompt */
#ifdef JP
			jverb( prompt, jverb_buf, JVERB_AND);
			/* ڤؤǽб */
			(void)strnfmt(tmp_val, 78, "%s(MP%d, Ψ%d%%)%sޤ? ",
				spell_names[use_realm -1][spell % 32],
				get_modified_smana(s_ptr), spell_chance(spell, use_realm -1), jverb_buf);
#else
			(void)strnfmt(tmp_val, 78, "%^s %s (%d mana, %d%% fail)? ",
				prompt, spell_names[use_realm - 1][spell % 32],
				get_modified_smana(s_ptr), spell_chance(spell, use_realm - 1));
#endif

			/* Belay that order */
			if (!get_check(tmp_val)) continue;
		}

		/* Stop the loop */
		flag = TRUE;
	}


	/* Restore the screen */
	if (redraw) screen_load();


	/* Show choices */
	if (show_choices)
	{
		/* Update */
		p_ptr->window |= (PW_SPELL);

		/* Window stuff */
		window_stuff();
	}


	/* Abort if needed */
	if (!flag) return (FALSE);

	/* Save the choice */
	(*sn) = spell;

#ifdef ALLOW_REPEAT /* TNB */

	repeat_push(*sn);

#endif /* ALLOW_REPEAT -- TNB */

	/* Success */
	return (TRUE);
}


/*
 * Peruse the spells/prayers in a book
 *
 * Note that *all* spells in the book are listed
 *
 * Note that browsing is allowed while confused or blind,
 * and in the dark, primarily to allow browsing in stores.
 */
void do_cmd_browse(void)
{
	int		item, sval, use_realm = 0, j, line;
	int		spell = -1;
	int		num = 0;

	byte		spells[32];
	char		temp[62 * 4];

	object_type	*o_ptr;
	magic_type	*s_ptr;

	cptr q, s;

	/* Warriors are illiterate */
	if (!(p_ptr->realm1 || p_ptr->realm2))
	{
#ifdef JP
		msg_print("ܤɤळȤǤʤ");
#else
		msg_print("You cannot read books!");
#endif
		return;
	}

	/* Restrict choices to "useful" books */
	item_tester_tval = TV_LIFE_BOOK;

	/* Get an item */
#ifdef JP
	q = "ɤܤɤߤޤ? ";
	s = "ɤܤʤ";
#else
	q = "Browse which book? ";
	s = "You have no books that you can read.";
#endif

	select_spellbook = TRUE;
	if (!get_item(&item, q, s, (USE_INVEN | USE_FLOOR)))
	{
		select_spellbook = FALSE;
		return;
	}
	select_spellbook = FALSE;

	/* Get the item (in the pack) */
	if (item >= 0)
	{
		o_ptr = &inventory[item];
	}

	/* Get the item (on the floor) */
	else
	{
		o_ptr = &o_list[0 - item];
	}

	/* Access the item's sval */
	sval = o_ptr->sval;

	use_realm = tval2realm(o_ptr->tval);

	/* Track the object kind */
	object_kind_track(o_ptr->k_idx);

	/* Hack -- Handle stuff */
	handle_stuff();


	/* Extract spells */
	for (spell = 0; spell < 32; spell++)
	{
		/* Check for this spell */
		if ((fake_spell_flags[sval] & (1L << spell)))
		{
			/* Collect this spell */
			spells[num++] = spell;
		}
	}

	/* Save the screen */
	screen_save();

	/* Clear the top line */
	prt("", 0, 0);

	/* Keep browsing spells.  Exit browsing on cancel. */
	while(TRUE)
	{
		/* Ask for a spell, allow cancel */
#ifdef JP
		if (!get_spell(&spell, "ɤ", o_ptr->sval, TRUE, use_realm))
#else
		if (!get_spell(&spell, "browse", o_ptr->sval, TRUE, use_realm))
#endif
		{
			/* If cancelled, leave immediately. */
			if (spell == -1) break;

			/* Display a list of spells */
			print_spells(spells, num, 1, 15, use_realm - 1);

			/* Notify that there's nothing to see, and wait. */
#ifdef JP
			prt("ɤʸʤ", 0, 0);
#else
			prt("No spells to browse.", 0, 0);
#endif
			(void)inkey();

			/* Restore the screen */
			screen_load();

			return;
		}

		/* Clear lines, position cursor  (really should use strlen here) */
		Term_erase(14, 18, 255);
		Term_erase(14, 17, 255);
		Term_erase(14, 16, 255);
		Term_erase(14, 15, 255);

		s_ptr = &mp_ptr->info[use_realm - 1][spell];

		roff_to_buf(spell_tips[use_realm - 1][spell], 62, temp, sizeof(temp));
		for(j = 0, line = 15; temp[j]; j += (1 + strlen(&temp[j])))
		{
			prt(&temp[j], line, 15);
			line++;
		}
	}

	/* Restore the screen */
	screen_load();
}




/*
 * Study a book to gain a new spell/prayer
 */
void do_cmd_study(void)
{
	int	i, item, sval;
	int	increment = 0;

	/* Spells of realm2 will have an increment of +32 */
	int	spell = -1;

#ifdef JP
	cptr p = ((mp_ptr->spell_type == ST_PRAYER) ? "" : "ʸ");
#else
	cptr p = ((mp_ptr->spell_type == ST_PRAYER) ? "prayer" : "spell");
#endif

	object_type *o_ptr;

	cptr q, s;

	if (!p_ptr->realm1)
	{
#ifdef JP
		msg_print("ܤɤळȤǤʤ");
#else
		msg_print("You cannot read books!");
#endif
		return;
	}

	if (p_ptr->blind || no_lite())
	{
#ifdef JP
		msg_print("ܤʤ");
#else
		msg_print("You cannot see!");
#endif
		return;
	}

	if (p_ptr->confused)
	{
#ifdef JP
		msg_print("𤷤Ƥɤʤ");
#else
		msg_print("You are too confused!");
#endif
		return;
	}

	if (!(p_ptr->new_spells))
	{
#ifdef JP
		msg_format("%sФ뤳ȤϤǤʤ", p);
#else
		msg_format("You cannot learn any new %ss!", p);
#endif
		return;
	}

#ifdef JP
	if( p_ptr->new_spells < 10 )
		msg_format(" %d Ĥ%sؤ٤롣", p_ptr->new_spells, p);
	else
		msg_format(" %d Ĥ%sؤ٤롣", p_ptr->new_spells, p);
#else
	msg_format("You can learn %d new %s%s.", p_ptr->new_spells, p,
		(p_ptr->new_spells == 1?"":"s"));
#endif
	msg_print(NULL);


	/* Restrict choices to "useful" books */
	item_tester_tval = TV_LIFE_BOOK;

	/* Get an item */
#ifdef JP
	q = "ɤܤؤӤޤ? ";
	s = "ɤܤʤ";
#else
	q = "Study which book? ";
	s = "You have no books that you can read.";
#endif

	select_spellbook = TRUE;
	if (!get_item(&item, q, s, (USE_INVEN | USE_FLOOR))) return;
	select_spellbook = FALSE;

	/* Get the item (in the pack) */
	if (item >= 0)
	{
		o_ptr = &inventory[item];
	}

	/* Get the item (on the floor) */
	else
	{
		o_ptr = &o_list[0 - item];
	}

	/* Access the item's sval */
	sval = o_ptr->sval;

#if 0
	if (o_ptr->tval == REALM2_BOOK) increment = 32;
#endif

	/* Track the object kind */
	object_kind_track(o_ptr->k_idx);

	/* Hack -- Handle stuff */
	handle_stuff();

	/* Mage -- Learn a selected spell */
	if (mp_ptr->spell_type != ST_PRAYER)
	{
		/* Ask for a spell, allow cancel */
#ifdef JP
		if (!get_spell(&spell, "ؤ", sval, FALSE, tval2realm(o_ptr->tval))
		    && (spell == -1)) return;
#else
		if (!get_spell(&spell, "study", sval, FALSE, tval2realm(o_ptr->tval))
		    && (spell == -1)) return;
#endif
	}

	/* Priest -- Learn a random prayer */
	else
	{
		int k = 0;

		int gift = -1;

		/* Extract spells */
		for (spell = 0; spell < 32; spell++)
		{
			/* Check spells in the book */
			if ((fake_spell_flags[sval] & (1L << spell)))
			{
				/* Skip non "okay" prayers */
				if (!spell_okay(spell, FALSE,
					(increment ? p_ptr->realm2 - 1 : p_ptr->realm1 - 1))) continue;

				/* Hack -- Prepare the randomizer */
				k++;

				/* Hack -- Apply the randomizer */
				if (rand_int(k) == 0) gift = spell;
			}
		}

		/* Accept gift */
		spell = gift;
	}

	/* Nothing to study */
	if (spell < 0)
	{
		/* Message */
#ifdef JP
		msg_format("ܤˤϳؤ֤٤%sʤ", p);
#else
		msg_format("You cannot learn any %ss in that book.", p);
#endif
		/* Abort */
		return;
	}


	/* Take a turn */
	energy_use = 100;

	if (increment) spell += increment;

	/* Learn the spell */
	if (spell < 32)
	{
		spell_learned1 |= (1L << spell);
	}
	else
	{
		spell_learned2 |= (1L << (spell - 32));
	}

	/* Find the next open entry in "spell_order[]" */
	for (i = 0; i < 64; i++)
	{
		/* Stop at the first empty space */
		if (spell_order[i] == 99) break;
	}

	/* Add the spell to the known list */
	spell_order[i++] = spell;

	/* Mention the result */
#ifdef JP
	/* ڤؤǽб */
		msg_format("%s%sؤ",
			    spell_names
		[(increment ? p_ptr->realm2 - 1 : p_ptr->realm1 - 1)][spell % 32] ,p);
#else
	msg_format("You have learned the %s of %s.",
		p, spell_names
		[(increment ? p_ptr->realm2 - 1 : p_ptr->realm1 - 1)][spell % 32]);
#endif

	/* Sound */
	sound(SOUND_STUDY);

	/* One less spell available */
	p_ptr->new_spells--;

	/* Message if needed */
	if (p_ptr->new_spells)
	{
		/* Message */
#ifdef JP
			if( p_ptr->new_spells < 10 )
				msg_format(" %d Ĥ%sؤ٤롣", p_ptr->new_spells, p);
			else
				msg_format(" %d Ĥ%sؤ٤롣", p_ptr->new_spells, p);
#else
		msg_format("You can learn %d more %s%s.",
			p_ptr->new_spells, p,
			(p_ptr->new_spells != 1) ? "s" : "");
#endif
	}

	/* Save the new_spells value */
	p_ptr->old_spells = p_ptr->new_spells;

	/* Redraw Study Status */
	p_ptr->redraw |= (PR_STUDY);
}


static bool cast_life_spell(int spell)
{
	int	dir;
	int	plev = p_ptr->lev;

	switch (spell)
	{
	case 0: /* Detect Evil */
		(void)detect_monsters_evil(DETECT_RAD_DEFAULT);
		break;
	case 1: /* Cure Light Wounds */
		(void)hp_player(damroll(2, 10));
		(void)set_cut(p_ptr->cut - 10);
		break;
	case 2: /* Bless */
		(void)set_blessed(p_ptr->blessed + randint(12) + 12);
		break;
	case 3: /* Remove Fear */
		(void)set_afraid(0);
		break;
	case 4: /* Call Light */
		(void)lite_area(damroll(2, (plev / 2)), (plev / 10) + 1);
		break;
	case 5: /* Detect Traps + Secret Doors */
		(void)detect_traps(DETECT_RAD_DEFAULT);
		(void)detect_doors(DETECT_RAD_DEFAULT);
		(void)detect_stairs(DETECT_RAD_DEFAULT);
		break;
	case 6: /* Cure Medium Wounds */
		(void)hp_player(damroll(4, 10));
		(void)set_cut((p_ptr->cut / 2) - 20);
		break;
	case 7: /* Satisfy Hunger */
		(void)set_food(PY_FOOD_MAX - 1);
		break;
	case 8: /* Holy Lite */
		if (!get_aim_dir(&dir)) return FALSE;
#ifdef JP
		msg_print("ʤ줿");
#else
		msg_print("A line of sunlight appears.");
#endif
		(void)lite_line(dir);
		break;
	case 9: /* Remove Curse */
		if (remove_curse())
		{
#ifdef JP
			msg_print("ï˸Ƥ褦ʵ롣");
#else
			msg_print("You feel as if someone is watching over you.");
#endif
		}
		break;
	case 10: /* Cure Critical Wounds */
		(void)hp_player(damroll(8, 10));
		(void)set_stun(0);
		(void)set_cut(0);
		(void)set_poisoned(0);
		break;
	case 11: /* Orb or Draining */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_ball(GF_HOLY_FIRE, dir,
			  (damroll(3, 6) + plev +
			  (plev / ((p_ptr->pclass == CLASS_PRIEST) ? 2 : 4))),
			  ((plev < 30) ? 2 : 3));
		break;
	case 12: /* Nature Awareness -- downgraded */
		map_area(DETECT_RAD_MAP);
		(void)detect_traps(DETECT_RAD_DEFAULT);
		(void)detect_doors(DETECT_RAD_DEFAULT);
		(void)detect_stairs(DETECT_RAD_DEFAULT);
		(void)detect_monsters_normal(DETECT_RAD_DEFAULT);
		break;
	case 13: /* Resistance Heat & Cold */
		(void)set_oppose_fire(p_ptr->oppose_fire + randint(20) + 20);
		(void)set_oppose_cold(p_ptr->oppose_cold + randint(20) + 20);
		break;
	case 14: /* Charm Monster */
		if (!get_aim_dir(&dir)) return FALSE;

		(void)charm_monster(dir, plev);
		break;
	case 15: /* Protection from Evil */
		(void)set_protevil(p_ptr->protevil + randint(25) + 3 * p_ptr->lev);
		break;
	case 16: /* Healing */
		(void)hp_player(300);
		(void)set_stun(0);
		(void)set_cut(0);
		break;
	case 17: /* Banishment */
		if (banish_evil(100))
		{
#ifdef JP
			msg_print("θϤٰǤʧä");
#else
			msg_print("The power of your god banishes evil!");
#endif
		}
		break;
	case 18: /* Star Burst */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_ball(GF_LITE, dir, (100 + plev * 2), 3);
		break;
	case 19: /* Glyph of Warding */
		warding_glyph();
		break;
	case 20: /* Heroism */
		(void)set_hero(p_ptr->hero + randint(25) + 25);
		(void)hp_player(10);
		(void)set_afraid(0);
		break;
	case 21: /* Dispel Curse */
		if (remove_all_curse())
		{
#ifdef JP
			msg_print("ï˸Ƥ褦ʵ롣");
#else
			msg_print("You feel as if someone is watching over you.");
#endif
		}
		break;
	case 22: /* Resistance True */
		(void)set_oppose_acid(p_ptr->oppose_acid + randint(20) + 20);
		(void)set_oppose_elec(p_ptr->oppose_elec + randint(20) + 20);
		(void)set_oppose_fire(p_ptr->oppose_fire + randint(20) + 20);
		(void)set_oppose_cold(p_ptr->oppose_cold + randint(20) + 20);
		(void)set_oppose_pois(p_ptr->oppose_pois + randint(20) + 20);
		break;
	case 23: /* Dispel Evil */
		(void)dispel_evil(plev * 4);
		break;
	case 24: /* 'Day of the Dove' */
		charm_monsters(plev * 2);
		break;
	case 25: /* Holy Word */
		(void)dispel_evil(plev * 4);
		(void)hp_player(500);
		(void)set_afraid(0);
		(void)set_poisoned(0);
		(void)set_stun(0);
		(void)set_cut(0);
		break;
	case 26: /* Bless Weapon */
		return bless_weapon();
	case 27: /* Restoration */
		(void)do_res_stat(A_STR);
		(void)do_res_stat(A_INT);
		(void)do_res_stat(A_WIS);
		(void)do_res_stat(A_DEX);
		(void)do_res_stat(A_CON);
		(void)do_res_stat(A_CHR);
		(void)restore_level();
		break;
	case 28: /* Nanka(ʤ) */
		project(0, 8, py, px, 150, GF_LITE, PROJECT_KILL | PROJECT_ITEM);
		(void)lite_area(damroll(2, (plev / 2)), (plev / 10) + 1);
		wiz_lite();
		break;
	case 29: /* Holy Vision */
		return identify_fully(FALSE);
	case 30: /* Healing True */
		(void)hp_player(1000);
		(void)set_stun(0);
		(void)set_cut(0);
		break;
	case 31: /* Divine Intervention */
		project(0, 1, py, px, 600, GF_HOLY_FIRE, PROJECT_KILL);
		dispel_monsters(plev * 4);
		slow_monsters();
		stun_monsters(plev * 4);
		confuse_monsters(plev * 4);
		turn_monsters(plev * 4);
		stasis_monsters(plev * 4);
		(void)set_shero(p_ptr->shero + randint(25) + 25);
		(void)hp_player(300);
		if (!p_ptr->fast)
		{   /* Haste */
			(void)set_fast(randint(20 + plev) + plev);
		}
		else
		{
			(void)set_fast(p_ptr->fast + randint(5));
		}
		(void)set_afraid(0);
		break;
	default:
#ifdef JP
		msg_format("ʤʥ饤դμʸ %d 򾧤", spell);
#else
		msg_format("You cast an unknown Life spell: %d.", spell);
#endif
		msg_print(NULL);
	}

	return TRUE;
}


static bool cast_sorcery_spell(int spell)
{
	int	dir;
	int	beam;
	int	plev = p_ptr->lev;

	if (p_ptr->pclass == CLASS_MAGE) beam = plev;
	else beam = plev / 2;

	switch (spell)
	{
	case 0: /* Magic Missile */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_bolt_or_beam(beam - 10, GF_MISSILE, dir,
			damroll(3 + ((plev - 1) / 5), 4));
		break;
	case 1: /* Detect Monsters */
		(void)detect_monsters_normal(DETECT_RAD_DEFAULT);
		break;
	case 2: /* Phase Door */
		teleport_player(10);
		break;
	case 3: /* Detect Doors and Traps */
		(void)detect_traps(DETECT_RAD_DEFAULT);
		(void)detect_doors(DETECT_RAD_DEFAULT);
		(void)detect_stairs(DETECT_RAD_DEFAULT);
		break;
	case 4: /* Light Area */
		(void)lite_area(damroll(2, (plev / 2)), (plev / 10) + 1);
		break;
	case 5: /* Trap & Door Destruction */
		if (!get_aim_dir(&dir)) return FALSE;

		(void)destroy_door(dir);
		break;
	case 6: /* Teleport Self */
		teleport_player(plev * 5);
		break;
	case 7: /* Satisfy Hunger */
		(void)set_food(PY_FOOD_MAX - 1);
		break;
	case 8: /* Manaburst */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_ball(GF_MISSILE, dir,
			  (damroll(3, 5) + plev +
			   (plev / ((p_ptr->pclass == CLASS_MAGE) ? 2 : 4))),
			   (plev < 30) ? 2 : 3);
			/* Shouldn't actually use GF_MANA, as it will destroy all
			 * items on the floor */
		break;
	case 9: /* Word of Recall */
		word_of_recall();
		break;
	case 10: /* Stone to Mud */
		if (!get_aim_dir(&dir)) return FALSE;

		(void)wall_to_mud(dir);
		break;
	case 11: /* Recharging */
		return recharge(plev * 4);
	case 12: /* Magic Mapping */
		map_area(DETECT_RAD_MAP);
		break;
	case 13: /* Identify */
		if(plev < 35)
		{
			return ident_spell(FALSE);
		}
		else
		{
			return identify_fully(FALSE);
		}
	case 14: /* Sense Minds */
		(void)set_tim_esp(p_ptr->tim_esp + randint(30) + 25);
		break;
	case 15: /* Doom Bolt -- always beam in 2.0.7 or later */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_beam(GF_MANA, dir, damroll(11 + ((plev - 5) / 4), 8));
		break;
	case 16: /* Haste Self */
		if (!p_ptr->fast)
		{
			(void)set_fast(randint(20 + plev) + plev);
		}
		else
		{
			(void)set_fast(p_ptr->fast + randint(5));
		}
		break;
	case 17: /* Teleport Other */
		if (!get_aim_dir(&dir)) return FALSE;

		(void)fire_beam(GF_AWAY_ALL, dir, plev);
		break;
	case 18: /* Magic Rocket */
		if (!get_aim_dir(&dir)) return FALSE;

#ifdef JP
		msg_print("åȯ͡");
#else
		msg_print("You launch a rocket!");
#endif
		fire_ball(GF_ROCKET, dir, (120 + plev * 2), 2);
		break;
	case 19: /* Genocide */
		(void)genocide(plev + 50);
		break;
	case 20: /* Probing */
		probing();
		break;
	case 21: /* Teleport monster to player */
	{
		monster_type *m_ptr;
		monster_race *r_ptr;
		char m_name[80];

		if (!target_set(TARGET_KILL)) return FALSE;
		if (!cave[target_row][target_col].m_idx) break;
		if (!player_has_los_bold(target_row, target_col)) break;
		m_ptr = &m_list[cave[target_row][target_col].m_idx];
		r_ptr = &r_info[m_ptr->r_idx];
		monster_desc(m_name, m_ptr, 0);
		if (r_ptr->flags3 & (RF3_RES_TELE))
		{
			if (r_ptr->flags1 & (RF1_UNIQUE))
			{
				r_ptr->r_flags3 |= RF3_RES_TELE;
#ifdef JP
				msg_format("%sˤϸ̤ʤä", m_name);
#else
				msg_format("%s is unaffected!", m_name);
#endif
				break;
			}
			else if (r_ptr->level > randint1(100))
			{
				r_ptr->r_flags3 |= RF3_RES_TELE;
#ifdef JP
				msg_format("%sˤ롪", m_name);
#else
				msg_format("%s resists!", m_name);
#endif
				break;
			}
		}
#ifdef JP
		msg_format("%sᤷ", m_name);
#else
		msg_format("You command %s to return.", m_name);
#endif
		teleport_to_player(cave[target_row][target_col].m_idx, 100);
		break;
	}
	case 22: /* Teleport Level */
		(void)teleport_player_level();
		break;
	case 23: /* Death Ray */
		if (!get_aim_dir(&dir)) return FALSE;

		(void)death_ray(dir, plev);
		break;
	case 24: /* Detection */
		(void)detect_all(DETECT_RAD_DEFAULT);
		break;
	case 25: /* Word of Destruction */
		destroy_area(py, px, 15, TRUE);
		break;
	case 26: /* Clairvoyance */
		wiz_lite();
		if (!p_ptr->telepathy)
		{
			(void)set_tim_esp(p_ptr->tim_esp + randint(30) + 25);
		}
		break;
	case 27: /* Dimension Door */
#ifdef JP
		msg_print("⤬ŪϤǲ");
#else
		msg_print("You open a dimensional gate. Choose a destination.");
#endif
		return dimension_door();
	case 28: /* Alchemy */
		return alchemy();
	case 29: /* Mass Genocide */
		(void)mass_genocide(plev + 50);
		break;
	case 30: /* Mana Storm */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_ball(GF_MANA, dir, 300 + (plev * 5), 4);
		break;
	case 31: /* Wraithform */
		set_wraith_form(p_ptr->tim_wraith + randint(plev / 2) + (plev / 2));
		break;
	default:
#ifdef JP
		msg_format("ʤʥ꡼μʸ %d 򾧤", spell);
#else
		msg_format("You cast an unknown Sorcery spell: %d.", spell);
#endif
		msg_print(NULL);
	}

	return TRUE;
}


static bool cast_musou_spell(int spell)
{
	int	    dir;
	int	    plev = p_ptr->lev;
	int	    x, y;

	switch (spell)
	{
	case 0: /* Regenerate */
		(void)set_tim_regen(p_ptr->tim_regen + 80 + randint(80));
		break;
	case 1: /* 3way attack */
	{
		int cdir;
		if (!get_rep_dir(&dir)) return FALSE;

		if (dir == 5) return FALSE;
		for (cdir = 0;cdir < 8; cdir++)
		{
			if (cdd[cdir] == dir) break;
		}
		if (cdir == 8) return FALSE;
		y = py + ddy_cdd[cdir];
		x = px + ddx_cdd[cdir];
		if (cave[y][x].m_idx)
			py_attack(y, x);
		else
			msg_print("϶ڤä");
		y = py + ddy_cdd[(cdir + 7) % 8];
		x = px + ddx_cdd[(cdir + 7) % 8];
		if (cave[y][x].m_idx)
			py_attack(y, x);
		else
			msg_print("϶ڤä");
		y = py + ddy_cdd[(cdir + 1) % 8];
		x = px + ddx_cdd[(cdir + 1) % 8];
		if (cave[y][x].m_idx)
			py_attack(y, x);
		else
			msg_print("϶ڤä");
		break;
	}
	case 2: /* Heroism */
		(void)set_hero(p_ptr->hero + randint(25) + 25);
		(void)hp_player(10);
		(void)set_afraid(0);
		break;
	case 3: /* Jumping */
		if (!get_aim_dir(&dir)) return FALSE;

		project_hook(GF_JUMP, dir, 0, PROJECT_STOP);
		break;
	case 4: /* Resist Environment */
		(void)set_oppose_cold(p_ptr->oppose_cold + randint(20) + 20);
		(void)set_oppose_fire(p_ptr->oppose_fire + randint(20) + 20);
		(void)set_oppose_elec(p_ptr->oppose_elec + randint(20) + 20);
		break;
	case 5: /* Radar Eye */
		(void)set_tim_radar(p_ptr->tim_radar + randint(25) + 30);
		break;
	case 6: /* Charge */
		return charge_monster();
	case 7: /* Slash Wind */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_bolt(GF_FORCE, dir, damroll(3 + ((plev - 5) / 4), 8));
		break;
	case 8: /* Stone Skin */
		(void)set_shield(p_ptr->shield + randint(25) + 30);
		break;
	case 9: /* Remove Curse */
		if (remove_curse())
		{
#ifdef JP
			msg_print("ï˸Ƥ褦ʵ롣");
#else
			msg_print("You feel as if someone is watching over you.");
#endif
		}
		break;
	case 10: /* Berserk */
		(void)set_shero(p_ptr->shero + randint(25) + 25);
		(void)hp_player(30);
		(void)set_afraid(0);
		break;
	case 11: /* Earthquake */
		earthquake(py, px, 10);
		break;
	case 12: /* Resistance True */
		(void)set_oppose_acid(p_ptr->oppose_acid + randint(20) + 20);
		(void)set_oppose_elec(p_ptr->oppose_elec + randint(20) + 20);
		(void)set_oppose_fire(p_ptr->oppose_fire + randint(20) + 20);
		(void)set_oppose_cold(p_ptr->oppose_cold + randint(20) + 20);
		(void)set_oppose_pois(p_ptr->oppose_pois + randint(20) + 20);
		break;
	case 13: /* Aura Arrow */
		superb_shot = TRUE;
		do_cmd_fire();
		superb_shot = FALSE;
		break;
	case 14: /* Fast Slicing */
		if (!get_rep_dir(&dir)) return FALSE;

		y = py + ddy[dir];
		x = px + ddx[dir];
		if(cave[y][x].m_idx)
		{
			py_attack_special(y, x, ATTACK_VORPAL);
		}
		else
		{
#ifdef JP
			msg_print("ˤϥ󥹥Ϥޤ");
#else
			msg_print("You don't see any monster in this direction");
#endif
			return FALSE;
		}
		break;
	case 15: /* Haste Self */
		if (!p_ptr->fast)
		{
			(void)set_fast(randint(20 + plev) + plev);
		}
		else
		{
			(void)set_fast(p_ptr->fast + randint(5));
		}
		break;
	case 16: /* Whirlwind Attack */
		{
			int y = 0, x = 0;
			cave_type       *c_ptr;
			monster_type    *m_ptr;

			for (dir = 0; dir <= 9; dir++)
			{
				y = py + ddy[dir];
				x = px + ddx[dir];
				c_ptr = &cave[y][x];

				/* Get the monster */
				m_ptr = &m_list[c_ptr->m_idx];

				/* Hack -- attack monsters */
				if (c_ptr->m_idx && (m_ptr->ml || cave_floor_bold(y, x)))
					py_attack(y, x);
			}
		}
		break;
	case 17: /* Coating */
		(void)coat_equip();
		break;
	case 18: /* Nanka() */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_beam(GF_INERTIA, dir, (plev * 3 / 2) + randint(plev * 2));
		break;
	case 19: /* Magical Armour */
		(void)set_magicdef(p_ptr->magicdef + randint(20) + 20);
		break;
	case 20: /* Remove Enchantment */
		if(!mundane_spell()) return FALSE;
		break;
	case 21: /* Jump and Attack */
		if (!get_aim_dir(&dir)) return FALSE;

		project_hook(GF_JUMP_ATTACK, dir, 0, PROJECT_STOP | PROJECT_KILL);
		break;
	case 22: /* Protection from Evil */
		(void)set_protevil(p_ptr->protevil + randint(25) + 3 * p_ptr->lev);
		break;
	case 23: /* Battle Frenzy */
		(void)set_shero(p_ptr->shero + randint(25) + 25);
		(void)hp_player(30);
		(void)set_afraid(0);
		if (!p_ptr->fast)
		{
			(void)set_fast(randint(20 + (plev / 2)) + (plev / 2));
		}
		else
		{
			(void)set_fast(p_ptr->fast + randint(5));
		}
		break;
	case 24: /* Bombing Attack */
		if (!get_rep_dir(&dir)) return FALSE;

		y = py + ddy[dir];
		x = px + ddx[dir];
		if(cave[y][x].m_idx)
		{
			py_attack_special(y, x, ATTACK_BAKUSIN);
		}
		else
		{
#ifdef JP
			msg_print("ˤϥ󥹥Ϥޤ");
#else
			msg_print("You don't see any monster in this direction");
#endif
			return FALSE;
		}
		break;
	case 25: /* Enchant Weapon */
		return enchant_spell(rand_int(4) + 1, rand_int(4) + 1, 0);
	case 26: /* Enchant Armour */
		return enchant_spell(0, 0, rand_int(3) + 2);
	case 27: /* Nanka(Ʈη) */
		if (!get_rep_dir(&dir)) return FALSE;

		y = py + ddy[dir];
		x = px + ddx[dir];
		if(cave[y][x].m_idx)
		{
			py_attack_special(y, x, ATTACK_TOUKI);
		}
		else
		{
#ifdef JP
			msg_print("ˤϥ󥹥Ϥޤ");
#else
			msg_print("You don't see any monster in this direction");
#endif
			return FALSE;
		}
		break;
	case 28: /* Nanka(Ȼ¤) */
		if (!get_rep_dir(&dir)) return FALSE;

		y = py + ddy[dir];
		x = px + ddx[dir];
		if(cave[y][x].m_idx)
		{
			py_attack(y, x);
		}
		else
		{
#ifdef JP
			msg_print("ˤϥ󥹥Ϥޤ");
#else
			msg_print("You don't see any monster in this direction");
#endif
			return FALSE;
		}
		if (cave[y][x].m_idx)
		{
			handle_stuff();
			py_attack(y, x);
		}
		break;
	case 29: /* Nanka(ε) */
		if (!get_aim_dir(&dir)) return FALSE;

		fire_ball(GF_INERTIA, dir, 175+randint(plev*2), -3);
		break;
	case 30: /* Nanka(Ǽ٤ο) */
		brand_weapon(5);
		break;
	case 31: /* Nanka(̵) */
	{
		int v = randint(25) + 25;
		(void)set_hero(p_ptr->hero + v);
		(void)set_shero(p_ptr->shero + v);
		(void)hp_player(30);
		(void)set_afraid(0);
		if (p_ptr->fast < v)
		{
			(void)set_fast(v);
		}
		else
		{
			(void)set_fast(p_ptr->fast + randint(5));
		}
		(void)set_shield(p_ptr->shield + v);
		(void)set_oppose_acid(p_ptr->oppose_acid + v);
		(void)set_oppose_elec(p_ptr->oppose_elec + v);
		(void)set_oppose_fire(p_ptr->oppose_fire + v);
		(void)set_oppose_cold(p_ptr->oppose_cold + v);
		(void)set_oppose_pois(p_ptr->oppose_pois + v);
		(void)set_musou(p_ptr->musou + v);
		break;
	}
	default:
#ifdef JP
		msg_format("ʤ̵Фμʸ %d 򾧤", spell);
#else
		msg_format("You cast an unknown Musou spell: %d.", spell);
#endif
		msg_print(NULL);
	}

	return TRUE;
}


/*
 * Cast a spell
 */
void do_cmd_cast(void)
{
	int	item, sval, spell, realm;
	int	chance;
	int	increment = 0;
	int	use_realm;
	bool cast;
	byte use_mana;

#ifdef JP
	const cptr prayer = ((mp_ptr->spell_type == ST_PRAYER) ? "" : "ʸ");
#else
	const cptr prayer = ((mp_ptr->spell_type == ST_PRAYER) ? "prayer" : "spell");
#endif

	object_type	*o_ptr;

	magic_type	*s_ptr;

	cptr q, s;

	/* Require spell ability */
	if (!p_ptr->realm1)
	{
#ifdef JP
		msg_print("ʸ򾧤ʤ");
#else
		msg_print("You cannot cast spells!");
#endif
		return;
	}

	/* Require lite */
	if (p_ptr->blind || no_lite())
	{
#ifdef JP
		msg_print("ܤʤ");
#else
		msg_print("You cannot see!");
#endif
		return;
	}

	/* Not when confused */
	if (p_ptr->confused)
	{
#ifdef JP
		msg_print("𤷤ƤƳؤ٤ʤ");
#else
		msg_print("You are too confused!");
#endif
		return;
	}

	/* Restrict choices to spell books */
	item_tester_tval = TV_LIFE_BOOK;

	/* Get an item */
#ifdef JP
	q = "ɤμʸȤޤ? ";
	s = "ʸ񤬤ʤ";
#else
	q = "Use which book? ";
	s = "You have no spell books!";
#endif

	select_spellbook = TRUE;
	if (!get_item(&item, q, s, (USE_INVEN | USE_FLOOR)))
	{
		select_spellbook = FALSE;
		return;
	}
	select_spellbook = FALSE;

	/* Get the item (in the pack) */
	if (item >= 0)
	{
		o_ptr = &inventory[item];
	}

	/* Get the item (on the floor) */
	else
	{
		o_ptr = &o_list[0 - item];
	}

	/* Access the item's sval */
	sval = o_ptr->sval;

#if 0
	if (o_ptr->tval == REALM2_BOOK) increment = 32;
#endif

	/* Track the object kind */
	object_kind_track(o_ptr->k_idx);

	/* Hack -- Handle stuff */
	handle_stuff();

	if (increment) realm = p_ptr->realm2;
	else realm = p_ptr->realm1;

	/* Ask for a spell */
#ifdef JP
	if (!get_spell(&spell,  
		((mp_ptr->spell_type == ST_PRAYER) ? "Ӿ" : ""),
		sval, TRUE, realm))
	{
		if (spell == -2) msg_format("ܤˤΤäƤ%sʤ", prayer);
		return;
	}
#else
	if (!get_spell(&spell, ((mp_ptr->spell_type == ST_PRAYER) ? "recite" : "cast"),
		sval, TRUE, realm))
	{
		if (spell == -2)
			msg_format("You don't know any %ss in that book.", prayer);
		return;
	}
#endif



	/* Access the spell */
	use_realm = (increment?p_ptr->realm2:p_ptr->realm1);

	s_ptr = &mp_ptr->info[use_realm - 1][spell];
	use_mana = get_modified_smana(s_ptr);

	/* Verify "dangerous" spells */
	if (use_mana > p_ptr->csp)
	{
		/* Warning */
#ifdef JP
		msg_format("%s%sΤ˽ʬʥޥåݥȤʤ",prayer,
			((mp_ptr->spell_type == ST_PRAYER) ? "Ӿ" : ""));
#else
		msg_format("You do not have enough mana to %s this %s.",
			((mp_ptr->spell_type == ST_PRAYER) ? "recite" : "cast"),
			prayer);
#endif
		if (!over_exert) return;

		/* Verify */
#ifdef JP
		if (!get_check("Ǥĩ路ޤ? ")) return;
#else
		if (!get_check("Attempt it anyway? ")) return;
#endif
	}


	/* Spell failure chance */
	chance = spell_chance(spell, use_realm - 1);

	/* Failed spell */
	if (rand_int(100) < chance)
	{
		if (flush_failure) flush();

#ifdef JP
		msg_format("%s򤦤ޤʤä", prayer);
#else
		msg_format("You failed to get the %s off!", prayer);
#endif
		sound(SOUND_FAIL);
	}

	/* Process spell */
	else
	{
		/* Spells.  */
		switch (realm)
		{
		case REALM_LIFE: /* * LIFE * */
			cast = cast_life_spell(spell);
			break;
		case REALM_SORCERY: /* * SORCERY * */
			cast = cast_sorcery_spell(spell);
			break;
		case REALM_MUSOU: /* * MUSOU * */
			cast = cast_musou_spell(spell);
			break;
		default:
			cast = FALSE;
			msg_format("You cast a spell from an unknown realm: realm %d, spell %d.", realm, spell);
			msg_print(NULL);
		}

		/* Canceled spells cost neither a turn nor mana */
		if (!cast) return;

		/* A spell was cast */
		if (!(increment ?
		    (spell_worked2 & (1L << spell)) :
		    (spell_worked1 & (1L << spell))))
		{
			int e = s_ptr->sexp;

			/* The spell worked */
			if (realm == p_ptr->realm1)
			{
				spell_worked1 |= (1L << spell);
			}
			else
			{
				spell_worked2 |= (1L << spell);
			}

			/* Gain experience */
			gain_exp(e * s_ptr->slevel);
		}
	}

	/* Take a turn */
	if (!((realm == REALM_MUSOU) && (spell == 13)))
		energy_use = 100;
	else /* Aura Arrow */
	{
		object_type *j_ptr = &inventory[INVEN_BOW];

		if (j_ptr->tval)
		{
			int thits = p_ptr->num_fire;
			int tmul = bow_tmul(j_ptr->sval);

			energy_use = bow_energy(j_ptr->sval);
			energy_use = (energy_use * 100 / thits);
		}
		else
		{
			energy_use = 100;
		}
	}

	/* Sufficient mana */
	if (use_mana <= p_ptr->csp)
	{
		/* Use some mana */
		p_ptr->csp -= use_mana;
	}

	/* Over-exert the player */
	else
	{
		int oops = use_mana - p_ptr->csp;

		/* No mana left */
		p_ptr->csp = 0;
		p_ptr->csp_frac = 0;

		/* Message */
#ifdef JP
		msg_print("椷Ƶ򼺤äƤޤä");
#else
		msg_print("You faint from the effort!");
#endif

		/* Hack -- Bypass free action */
		(void)set_paralyzed(p_ptr->paralyzed + randint(5 * oops + 1));

		/* Damage CON (possibly permanently) */
		if (rand_int(100) < 50)
		{
			bool perm = (rand_int(100) < 25);

			/* Message */
#ifdef JP
			msg_print("Τ򰭤Ƥޤä");
#else
			msg_print("You have damaged your health!");
#endif

			/* Reduce constitution */
			(void)dec_stat(A_CON, 15 + randint(10), perm);
		}
	}

	/* Redraw mana */
	p_ptr->redraw |= (PR_MANA);

	/* Window stuff */
	p_ptr->window |= (PW_PLAYER);
	p_ptr->window |= (PW_SPELL);
}


/*
 * Pray a prayer -- Unused in Zangband
 */
void do_cmd_pray(void)
{
	msg_print("Praying is not used in Zangband. Use magic spell casting instead.");
}


/*
 * Issue a pet command
 */
void do_cmd_pet(void)
{
	int			i = 0;
	int			num;
	int			powers[36];
	cptr			power_desc[36];
	bool			flag, redraw;
	int			ask;
	char			choice;
	char			out_val[160];
	int			pets = 0, pet_ctr;
	bool			all_pets = FALSE;
	monster_type	*m_ptr;

	int mode = 0;

	byte y = 1, x = 0;
	int ctr = 0;
	char buf[160];

	num = 0;

	/* Calculate pets */
	/* Process the monsters (backwards) */
	for (pet_ctr = m_max - 1; pet_ctr >= 1; pet_ctr--)
	{
		/* Access the monster */
		m_ptr = &m_list[pet_ctr];

		if (is_pet(m_ptr)) pets++;
	}

	if (pets)
	{
#ifdef JP
		power_desc[num] = "ڥåȤ";
#else
		power_desc[num] = "dismiss pets";
#endif
		powers[num++] = PET_DISMISS;
	}

#ifdef JP
	power_desc[num] = "᤯ˤ";
#else
	power_desc[num] = "stay close";
#endif

	if (p_ptr->pet_follow_distance == PET_CLOSE_DIST) mode = num;
	powers[num++] = PET_STAY_CLOSE;

#ifdef JP
	power_desc[num] = "Ĥ褤";
#else
	power_desc[num] = "follow me";
#endif

	if (p_ptr->pet_follow_distance == PET_FOLLOW_DIST) mode = num;
	powers[num++] = PET_FOLLOW_ME;

#ifdef JP
	power_desc[num] = "Ũ򸫤Ĥݤ";
#else
	power_desc[num] = "seek and destroy";
#endif

	if (p_ptr->pet_follow_distance == PET_DESTROY_DIST) mode = num;
	powers[num++] = PET_SEEK_AND_DESTROY;

#ifdef JP
	power_desc[num] = "ΥƤ";
#else
	power_desc[num] = "give me space";
#endif

	if (p_ptr->pet_follow_distance == PET_SPACE_DIST) mode = num;
	powers[num++] = PET_ALLOW_SPACE;

#ifdef JP
	power_desc[num] = "ΥƤ";
#else
	power_desc[num] = "stay away";
#endif

	if (p_ptr->pet_follow_distance == PET_AWAY_DIST) mode = num;
	powers[num++] = PET_STAY_AWAY;

	if (p_ptr->pet_open_doors)
	{
#ifdef JP
		power_desc[num] = "ɥ򳫤";
#else
		power_desc[num] = "pets may open doors";
#endif
	}
	else
	{
#ifdef JP
		power_desc[num] = "ɥ򳫤ʤ";
#else
		power_desc[num] = "pets may not open doors";
#endif
	}
	powers[num++] = PET_OPEN_DOORS;

	if (p_ptr->pet_pickup_items)
	{
#ifdef JP
		power_desc[num] = "ƥ򽦤碌";
#else
		power_desc[num] = "pets may pick up items";
#endif
	}
	else
	{
#ifdef JP
		power_desc[num] = "ƥ򽦤碌ʤ";
#else
		power_desc[num] = "pets may not pick up items";
#endif

	}
	powers[num++] = PET_TAKE_ITEMS;

	/* Nothing chosen yet */
	flag = FALSE;

	/* Build a prompt (accept all spells) */
	if (num <= 26)
	{
		/* Build a prompt (accept all spells) */
#ifdef JP
		strnfmt(out_val, 78, "(ޥ %c-%c'*'=ESC=λ) ޥɤǤ:",
#else
		strnfmt(out_val, 78, "(Command %c-%c, *=List, ESC=exit) Select a command: ",
#endif
			I2A(0), I2A(num - 1));
	}
	else
	{
#ifdef JP
		strnfmt(out_val, 78, "(ޥ %c-%c'*'=ESC=λ) ޥɤǤ:",
#else
		strnfmt(out_val, 78, "(Command %c-%c, *=List, ESC=exit) Select a command: ",
#endif
			I2A(0), '0' + num - 27);
	}

	/* Show list */
	redraw = TRUE;

	/* Save the screen */
	Term_save();

	prt("", y++, x);

	while (ctr < num)
	{
		sprintf(buf, "%s%c) %s", (ctr == mode) ? "*" : " ", I2A(ctr), power_desc[ctr]);
		prt(buf, y + ctr, x);
		ctr++;
	}

	if (ctr < 17)
	{
		prt("", y + ctr, x);
	}
	else
	{
		prt("", y + 17, x);
	}

	/* Get a command from the user */
	while (!flag && get_com(out_val, &choice))
	{
		/* Request redraw */
		if ((choice == ' ') || (choice == '*') || (choice == '?'))
		{
			/* Show the list */
			if (!redraw)
			{
				y = 1;
				x = 0;
				ctr = 0;

				/* Show list */
				redraw = TRUE;

				/* Save the screen */
				Term_save();

				prt("", y++, x);

				while (ctr < num)
				{
					sprintf(buf, "%s%c) %s", (ctr == mode) ? "*" : " ", I2A(ctr), power_desc[ctr]);
					prt(buf, y + ctr, x);
					ctr++;
				}

				if (ctr < 17)
				{
					prt("", y + ctr, x);
				}
				else
				{
					prt("", y + 17, x);
				}
			}

			/* Hide the list */
			else
			{
				/* Hide list */
				redraw = FALSE;

				/* Restore the screen */
				Term_load();
			}

			/* Redo asking */
			continue;
		}

		if (choice == '\r' && num == 1)
		{
			choice = 'a';
		}

		if (isalpha(choice))
		{
			/* Note verify */
			ask = (isupper(choice));

			/* Lowercase */
			if (ask) choice = tolower(choice);

			/* Extract request */
			i = (islower(choice) ? A2I(choice) : -1);
		}
		else
		{
			ask = FALSE; /* Can't uppercase digits */

			i = choice - '0' + 26;
		}

		/* Totally Illegal */
		if ((i < 0) || (i >= num))
		{
			bell();
			continue;
		}

		/* Verify it */
		if (ask)
		{
			/* Prompt */
#ifdef JP
			strnfmt(buf, 78, "%sȤޤ ", power_desc[i]);
#else
			strnfmt(buf, 78, "Use %s? ", power_desc[i]);
#endif

			/* Belay that order */
			if (!get_check(buf)) continue;
		}

		/* Stop the loop */
		flag = TRUE;
	}

	/* Restore the screen */
	if (redraw) Term_load();

	/* Abort if needed */
	if (!flag)
	{
		energy_use = 0;
		return;
	}

	switch (powers[i])
	{
		case PET_DISMISS: /* Dismiss pets */
		{
			int Dismissed = 0;

#ifdef JP
			if (get_check("٤ƤΥڥåȤޤ")) all_pets = TRUE;
#else
			if (get_check("Dismiss all pets? ")) all_pets = TRUE;
#endif

			/* Process the monsters (backwards) */
			for (pet_ctr = m_max - 1; pet_ctr >= 1; pet_ctr--)
			{
				/* Access the monster */
				m_ptr = &m_list[pet_ctr];

				if (is_pet(m_ptr))
				{
					bool delete_this = FALSE;

					if (all_pets)
						delete_this = TRUE;
					else
					{
						char friend_name[80], check_friend[80];
						monster_desc(friend_name, m_ptr, 0x80);
#ifdef JP
						sprintf(check_friend, "%sޤ ", friend_name);
#else
						sprintf(check_friend, "Dismiss %s? ", friend_name);
#endif

						if (get_check(check_friend))
							delete_this = TRUE;
					}

					if (delete_this)
					{
						delete_monster_idx(pet_ctr);
						Dismissed++;
					}
				}
			}

#ifdef JP
			msg_format("%d ɤΥڥåȤޤ", Dismissed);
#else
			msg_format("You have dismissed %d pet%s.", Dismissed,
				(Dismissed == 1 ? "" : "s"));
#endif
			break;
		}
		/* Call pets */
		case PET_STAY_CLOSE:
		{
			p_ptr->pet_follow_distance = PET_CLOSE_DIST;
			break;
		}
		/* "Follow Me" */
		case PET_FOLLOW_ME:
		{
			p_ptr->pet_follow_distance = PET_FOLLOW_DIST;
			break;
		}
		/* "Seek and destoy" */
		case PET_SEEK_AND_DESTROY:
		{
			p_ptr->pet_follow_distance = PET_DESTROY_DIST;
			break;
		}
		/* "Give me space" */
		case PET_ALLOW_SPACE:
		{
			p_ptr->pet_follow_distance = PET_SPACE_DIST;
			break;
		}
		/* "Stay away" */
		case PET_STAY_AWAY:
		{
			p_ptr->pet_follow_distance = PET_AWAY_DIST;
			break;
		}
		/* flag - allow pets to open doors */
		case PET_OPEN_DOORS:
		{
			p_ptr->pet_open_doors = !p_ptr->pet_open_doors;
			break;
		}
		/* flag - allow pets to pickup items */
		case PET_TAKE_ITEMS:
		{
			p_ptr->pet_pickup_items = !p_ptr->pet_pickup_items;

			/* Drop objects being carried by pets */
			if (!p_ptr->pet_pickup_items)
			{
				for (pet_ctr = m_max - 1; pet_ctr >= 1; pet_ctr--)
				{
					/* Access the monster */
					m_ptr = &m_list[pet_ctr];

					if (is_pet(m_ptr))
					{
						monster_drop_carried_objects(m_ptr);
					}
				}
			}

			break;
		}
	}
}
