/* File: racial.c */

/* Purpose: Racial powers (and mutations) */

/*
 * Copyright (c) 1989 James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research, and
 * not for profit purposes provided that this copyright and statement are
 * included in all such copies.
 */

#include "angband.h"


/*
 * Returns the chance to activate a racial power/mutation
 */
static int racial_chance(s16b min_level, int use_stat, int difficulty)
{
	int i;
	int val;
	int sum = 0;
	int stat = p_ptr->stat_cur[use_stat];

	/* No chance for success */
	if ((p_ptr->lev < min_level) || p_ptr->confused)
	{
		return (0);
	}

	if (difficulty == 0) return (100);

	/* Calculate difficulty */
	if (p_ptr->stun)
	{
		difficulty += p_ptr->stun;
	}
	else if (p_ptr->lev > min_level)
	{
		int lev_adj = ((p_ptr->lev - min_level) / 3);
		if (lev_adj > 10) lev_adj = 10;
		difficulty -= lev_adj;
	}

	if (difficulty < 5) difficulty = 5;

	/* We only need halfs of the difficulty */
	difficulty = difficulty / 2;

	for (i = 1; i <= stat; i++)
	{
		val = i - difficulty;
		if (val > 0)
			sum += (val <= difficulty) ? val : difficulty;
	}

	if (difficulty == 0)
		return (100);
	else
		return (((sum * 100) / difficulty) / stat);
}


/* Note: return value indicates that we have succesfully used the power */

bool racial_aux(s16b min_level, int cost, int use_stat, int difficulty)
{
	bool use_hp = FALSE;

	/* Not enough mana - use hp */
	if (p_ptr->csp < cost) use_hp = TRUE;

	/* Power is not available yet */
	if (p_ptr->lev < min_level)
	{
#ifdef JP
		msg_format("ǽϤѤˤϥ٥ %d ãƤʤФʤޤ", min_level);
#else
		msg_format("You need to attain level %d to use this power.", min_level);
#endif
		energy_use = 0;
		return FALSE;
	}

	/* Too confused */
	else if (p_ptr->confused)
	{
#ifdef JP
		msg_print("𤷤ƤƤǽϤϻȤʤ");
#else
		msg_print("You are too confused to use this power.");
#endif
		energy_use = 0;
		return FALSE;
	}

	/* Risk death? */
	else if (use_hp && (p_ptr->chp < cost))
	{
#ifdef JP
		if (!get_check("˺ο夷֤ǤǽϤȤޤ"))
#else
		if (!get_check("Really use the power in your weakened state? "))
#endif
		{
			energy_use = 0;
			return FALSE;
		}
	}

	/* Else attempt to do it! */
	if (difficulty)
	{
		if (p_ptr->stun)
		{
			difficulty += p_ptr->stun;
		}
		else if (p_ptr->lev > min_level)
		{
			int lev_adj = ((p_ptr->lev - min_level) / 3);
			if (lev_adj > 10) lev_adj = 10;
			difficulty -= lev_adj;
		}

		if (difficulty < 5) difficulty = 5;
	}

	/* take time and pay the price */
	energy_use = 100;

	if (use_hp)
	{
		take_hit((cost / 2) + randint(cost / 2),
#ifdef JP
			"٤ν");
#else
			"concentrating too hard");
#endif
	}
	else
	{
		p_ptr->csp -= (cost / 2) + randint(cost / 2);
	}


	/* Redraw mana and hp */
	p_ptr->redraw |= (PR_HP | PR_MANA);

	/* Window stuff */
	p_ptr->window |= (PW_PLAYER | PW_SPELL);

	/* Success? */
	if (randint(p_ptr->stat_cur[use_stat]) >=
	    ((difficulty / 2) + randint(difficulty / 2)))
	{
		return TRUE;
	}

#ifdef JP
	msg_print("ʬ˽Ǥʤä");
#else
	msg_print("You've failed to concentrate hard enough.");
#endif
	return FALSE;
}

/**** The monster bashing code. -LM- ****/
static bool do_cmd_shield_bash()
{
	int bash_chance, bash_quality, bash_dam;
	int y, x, dir = 0;
	bool fear;
	char m_name[80];
	cave_type *c_ptr;
	object_type *o_ptr;
	monster_type *m_ptr;
	monster_race *r_ptr;

	/* No shield on arm, no bash.  */
	if (!inventory[INVEN_ARM].k_idx)
	{
#ifdef JP
		msg_print("Ƥʤ");
#else
		msg_print("You weild no shields !");
#endif
		return (FALSE);
	}

	o_ptr = &inventory[INVEN_ARM];

	/* Only works on adjacent monsters */
	if (!get_rep_dir(&dir)) return (FALSE);   /* was get_aim_dir */
	y = py + ddy[dir];
	x = px + ddx[dir];
	c_ptr = &cave[y][x];

	if (!c_ptr->m_idx)
	{
#ifdef JP
		msg_print("ˤϤʤˤ⤤ޤ");
#else
		msg_print("Nobody is there.");
#endif
		return (FALSE);
	}

	m_ptr = &m_list[c_ptr->m_idx];
	r_ptr = &r_info[m_ptr->r_idx];

	/* Extract monster name (or "it") */
	monster_desc(m_name, m_ptr, 0);

	/* Auto-Recall if possible and visible */
	if (m_ptr->ml) monster_race_track(m_ptr->r_idx);

	/* Track a new monster */
	if (m_ptr->ml) health_track(c_ptr->m_idx);

	/* Bashing chance depends on melee Skill, Dex, and a class level bonus. */
	bash_chance = p_ptr->skill_thn + (p_ptr->to_h * BTH_PLUS_ADJ);

	/* Try to get in a shield bash. */
	if (rand_int(bash_chance) >= (r_ptr->ac * 3 / 4))
	{
#ifdef JP
		msg_print("ɤǥ󥹥ꤷ");
#else
		msg_print("You get in a shield bash!");
#endif

		/* Calculate attack quality, a mix of momentum and accuracy. */
		bash_quality = p_ptr->skill_thn + (p_ptr->wt / 8) +
			(p_ptr->total_weight / 80) + (inventory[INVEN_ARM].weight / 3);

		/* Calculate damage.  Big shields are deadly. */
		bash_dam = damroll(inventory[INVEN_ARM].dd, inventory[INVEN_ARM].ds);

		/* Multiply by quality and experience factors */
		bash_dam *= bash_quality / 20 + p_ptr->lev / 7;

		/* Strength bonus. */
		bash_dam += (adj_str_td[p_ptr->stat_ind[A_STR]] - 128);

		/* Paranoia. */
		if (bash_dam > 125) bash_dam = 125;

		/* Encourage the player to keep wearing that heavy shield. */
		if (randint(bash_dam) > 30 + randint(bash_dam / 2))
#ifdef JP
			msg_print("С");
#else
			msg_print("WHAMM!");
#endif

		/* Complex message */
		if (wizard)
		{
#ifdef JP
			msg_format("%d/%d Υ᡼Ϳ", bash_dam, m_ptr->hp);
#else
			msg_format("You do %d (out of %d) damage.", bash_dam, m_ptr->hp);
#endif
		}

		/* Damage, check for fear and death. */
		if (mon_take_hit(c_ptr->m_idx, bash_dam, &fear, NULL))
		{
			/* Fight's over. */
			return (TRUE);
		}

		/* Stunning. Must be stunned. */
#ifdef JP
		msg_format("%^sϥեեˤʤä", m_name);
#else
		msg_format("%^s is stunned.", m_name);
#endif
		m_ptr->stunned += rand_int(p_ptr->lev / 5) + 4;
		if (m_ptr->stunned > 24) m_ptr->stunned = 24;

		/* Confusion. */
		if (bash_quality + p_ptr->lev > randint(200 + r_ptr->level * 4) &&
			(!r_ptr->flags3 & (RF3_NO_CONF)))
		{
#ifdef JP
			msg_format("%^sϺ𤷤褦", m_name);
#else
			msg_format("%^s appears confused.", m_name);
#endif
			m_ptr->confused += rand_int(p_ptr->lev / 5) + 4;
		}

		/* Hack -- delay fear messages */
		if (fear && m_ptr->ml)
		{
			/* Sound */
			sound(SOUND_FLEE);

			/* Message */
#ifdef JP
			msg_format("%^s϶ݤƨФ", m_name);
#else
			msg_format("%^s flees in terror!", m_name);
#endif
		}
	}
	/* Player misses */
	else
	{
		/* Sound */
		sound(SOUND_MISS);

		/* Message */
#ifdef JP
		msg_format("ߥ %sˤ蘆줿", m_name);
#else
		msg_format("You miss %s.", m_name);
#endif
	}

	/* Monster is not dead */
	return (TRUE);
}

/* Mei-kyo-Shi-Sui */
static bool do_cmd_restore_mana()
{
	int use_stat = m_info[p_ptr->pclass].spell_stat;

	if (total_friends)
	{
#ifdef JP
		msg_print("ϥڥåȤ뤳Ȥ˽椷Ƥʤȡ");
#else
		msg_print("You need concentration on the pets now.");
#endif
		return FALSE;
	}

	if (racial_aux(15, 0, use_stat, 10))
	{
#ifdef JP
		msg_print("Ƭϥåꤷ");
#else
		msg_print("You feel your head clear a little.");
#endif
		p_ptr->csp += (3 + p_ptr->lev/15);
		if (p_ptr->csp >= p_ptr->msp)
		{
			p_ptr->csp = p_ptr->msp;
			p_ptr->csp_frac = 0;
		}
	}
	return TRUE;
}

static bool item_tester_hook_bows_and_ammos(object_type *o_ptr)
{
	switch (o_ptr->tval)
	{
		case TV_SHOT:
		case TV_ARROW:
		case TV_BOLT:
		case TV_BOW:
		{
			return (bool) (!object_known_p(o_ptr) ||
				(!(o_ptr->ident & IDENT_MENTAL) && (p_ptr->lev >= 30)));
		}
	}
	return (FALSE);
}

static bool do_cmd_identify_bows_and_ammos()
{
	int item;
	cptr q, s;
	object_type *o_ptr;
	char o_name[MAX_NLEN];
	bool old_known;
	int idx;

	item_tester_hook = item_tester_hook_bows_and_ammos;

	/* Get an item */
#ifdef JP
	q = "ɤΥƥꤷޤ? ";
	s = "Ǥ륢ƥबʤ";
#else
	q = "Identify which item? ";
	s = "You have nothing to identify.";
#endif

	if (!get_item(&item, q, s, (USE_EQUIP | USE_INVEN | USE_FLOOR))) return (FALSE);

	/* Access the item (if in the pack) */
	if (item >= 0)
	{
		o_ptr = &inventory[item];
	}
	else
	{
		o_ptr = &o_list[0 - item];
	}

	/* Identify it */
	old_known = identify_item(o_ptr);

	/* Mark the item as fully known */
	if (p_ptr->lev >= 30) o_ptr->ident |= (IDENT_MENTAL);

	/* Handle stuff */
	handle_stuff();

	/* Description */
	object_desc(o_name, o_ptr, TRUE, 3);

	/* Describe */
	if (item >= INVEN_WIELD)
	{
#ifdef JP
		msg_format("%^s: %s(%c)",
#else
		msg_format("%^s: %s (%c).",
#endif
			   describe_use(item), o_name, index_to_label(item));
	}
	else if (item >= 0)
	{
#ifdef JP
		msg_format("å: %s(%c)",
#else
		msg_format("In your pack: %s (%c).",
#endif
			   o_name, index_to_label(item));
	}
	else
	{
#ifdef JP
		msg_format(": %s",
#else
		msg_format("On the ground: %s.",
#endif
			   o_name);
	}

	if (p_ptr->lev >= 30) identify_fully_aux(o_ptr, TRUE);
	else identify_item(o_ptr);

	/* Auto-inscription/destroy */
	idx = is_autopick(o_ptr);
	auto_inscribe_item(item, idx);
	if (!old_known) auto_destroy_item(item, idx);

	return (TRUE);
}

static void cmd_racial_power_aux(s32b command)
{
	s16b        plev = p_ptr->lev;
	int         dir = 0;

	if (command <= -3)
	{
		switch (p_ptr->pclass)
		{
			case CLASS_WARRIOR:
			case CLASS_PALADIN:
			case CLASS_WARRIOR_MAGE:
				if (racial_aux(1, 15, A_DEX, 0))
				{
					(void)do_cmd_shield_bash();
					break;
				}
			case CLASS_MAGE:
			case CLASS_PRIEST:
			case CLASS_MINDCRAFTER:
				(void)do_cmd_restore_mana();
				break;
			case CLASS_RANGER:
			case CLASS_ARCHER:
				if (racial_aux(10, 15, A_INT, 15))
				{
					(void)do_cmd_identify_bows_and_ammos();
					break;
				}
			case CLASS_ARCHAEOLOGIST:
				if ((command == -3) && racial_aux(5, 5, A_INT, 10))
				{
					(void)detect_treasure(DETECT_RAD_DEFAULT);
				}
				else if ((command == -4) && racial_aux(15, 15, A_INT, 15))
				{
					(void)detect_objects_magic(DETECT_RAD_DEFAULT);
				}
				break;
		}
	}
	else switch (p_ptr->prace)
	{
		case RACE_DWARF:
			if (racial_aux(5, 5, A_WIS, 12))
			{
#ifdef JP
				msg_print("ϤĴ٤");
#else
				msg_print("You examine your surroundings.");
#endif
				(void)detect_traps(DETECT_RAD_DEFAULT);
				(void)detect_doors(DETECT_RAD_DEFAULT);
				(void)detect_stairs(DETECT_RAD_DEFAULT);
			}
			break;

		case RACE_HOBBIT:
			if (racial_aux(15, 10, A_INT, 10))
			{
				object_type *q_ptr;
				object_type forge;

				/* Get local object */
				q_ptr = &forge;

				/* Create the food ration */
				object_prep(q_ptr, 21);

				/* Drop the object from heaven */
				(void)drop_near(q_ptr, -1, py, px);
#ifdef JP
				msg_print("ƺä");
#else
				msg_print("You cook some food.");
#endif
			}
			break;

		case RACE_GNOME:
			if (racial_aux(5, (5 + (plev / 5)), A_INT, 12))
			{
#ifdef JP
				msg_print("ѥá");
#else
				msg_print("Blink!");
#endif
				teleport_player(10);
			}
			break;

		case RACE_HALF_ORC:
			if (racial_aux(3, 5, A_WIS,
			    (p_ptr->pclass == CLASS_WARRIOR ? 5 : 10)))
			{
#ifdef JP
				msg_print("ͦФ");
#else
				msg_print("You play tough.");
#endif
				(void)set_afraid(0);
			}
			break;

		case RACE_HALF_TROLL:
			if (racial_aux(10, 12, A_WIS,
			    (p_ptr->pclass == CLASS_WARRIOR ? 6 : 12)))
			{
#ifdef JP
				msg_print("");
#else
				msg_print("RAAAGH!");
#endif
				(void)set_afraid(0);

				(void)set_shero(p_ptr->shero + 10 + randint(plev));
				(void)hp_player(30);
			}
			break;

		case RACE_AMBERITE:
			if (command == -2)
			{
				if (racial_aux(40, 75, A_WIS, 50))
				{
#ifdef JP
					msg_print("ʤϡ֥ѥפ򿴤Ƥξ⤤...");
#else
					msg_print("You picture the Pattern in your mind and walk it...");
#endif
					(void)set_poisoned(0);
					(void)set_image(0);
					(void)set_stun(0);
					(void)set_cut(0);
					(void)set_blind(0);
					(void)set_afraid(0);
					(void)do_res_stat(A_STR);
					(void)do_res_stat(A_INT);
					(void)do_res_stat(A_WIS);
					(void)do_res_stat(A_DEX);
					(void)do_res_stat(A_CON);
					(void)do_res_stat(A_CHR);
					(void)restore_level();
				}
			}

			else if (command == -1)
			{
				if (racial_aux(30, 50, A_INT, 50))
				{
					/* No effect in arena or quest */
					if (p_ptr->inside_arena || p_ptr->inside_quest)
					{
#ifdef JP
						msg_print("̤ʤä");
#else
						msg_print("There is no effect.");
#endif
					}
					else
					{
#ifdef JP
						msg_print("ʤ⤭Ϥ᤿ϤѲƤ롣");
#else
						msg_print("You start walking around. Your surroundings change.");
#endif

						if (autosave_l) do_cmd_save_game(TRUE);

						/* Leaving */
						p_ptr->leaving = TRUE;
					}
				}
			}
			break;

		case RACE_BARBARIAN:
			if (racial_aux(8, 10, A_WIS, (p_ptr->pclass == CLASS_WARRIOR ? 6 : 12)))
			{
#ifdef JP
				msg_print("");
#else
				msg_print("Raaagh!");
#endif
				(void)set_afraid(0);

				(void)set_shero(p_ptr->shero + 10 + randint(plev));
				(void)hp_player(30);
			}
			break;

		case RACE_HALF_OGRE:
			if (racial_aux(25, 35, A_INT, 15))
			{
#ifdef JP
				msg_print("ȯΥ롼򿵽Ť˻ųݤ...");
#else
				msg_print("You carefully set an explosive rune...");
#endif
				explosive_rune();
			}
			break;

		case RACE_HALF_GIANT:
			if (racial_aux(20, 10, A_STR, 12))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("ФɤáĤ");
#else
				msg_print("You bash at a stone wall.");
#endif
				(void)wall_to_mud(dir);
			}
			break;

		case RACE_HALF_TITAN:
			if (racial_aux(35, 20, A_INT, 12))
			{
#ifdef JP
				msg_print("ŨĴ...");
#else
				msg_print("You examine your foes...");
#endif
				probing();
			}
			break;

		case RACE_CYCLOPS:
			if (racial_aux(20, 15, A_STR, 12))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("ʴꤲ");
#else
				msg_print("You throw a huge boulder.");
#endif
				fire_bolt(GF_MISSILE, dir, (3 * plev) / 2);
			}
			break;

		case RACE_YEEK:
			if (racial_aux(15, 15, A_WIS, 10))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("ȤӤĶ夲");
#else
				msg_print("You make a horrible scream!");
#endif
				(void)fear_monster(dir, plev);
			}
			break;

		case RACE_KLACKON:
			if (racial_aux(9, 9, A_DEX, 14))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("Ǥ");
#else
				msg_print("You spit acid.");
#endif
				if (plev < 25)
					fire_bolt(GF_ACID, dir, plev);
				else
					fire_ball(GF_ACID, dir, plev, 2);
			}
			break;

		case RACE_KOBOLD:
			if (racial_aux(12, 8, A_DEX, 14))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("ǤΥĤꤲ");
#else
				msg_print("You throw a dart of poison.");
#endif
				fire_bolt(GF_POIS, dir, plev);
			}
			break;

		case RACE_NIBELUNG:
			if (racial_aux(10, 5, A_WIS, 10))
			{
#ifdef JP
				msg_print("ϤĴ");
#else
				msg_print("You examine your surroundings.");
#endif
				(void)detect_traps(DETECT_RAD_DEFAULT);
				(void)detect_doors(DETECT_RAD_DEFAULT);
				(void)detect_stairs(DETECT_RAD_DEFAULT);
			}
			break;

		case RACE_DARK_ELF:
			if (racial_aux(2, 2, A_INT, 9))
			{
				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_print("ޥåߥä");
#else
				msg_print("You cast a magic missile.");
#endif
				fire_bolt_or_beam(10, GF_MISSILE, dir,
				    damroll(3 + ((plev - 1) / 5), 4));
			}
			break;

		case RACE_DRACONIAN:
			if (racial_aux(1, plev, A_CON, 12))
			{
				int  Type = ((randint(3) == 1) ? GF_COLD : GF_FIRE);
#ifdef JP
				cptr Type_desc = ((Type == GF_COLD) ? "䵤" : "");
#else
				cptr Type_desc = ((Type == GF_COLD) ? "cold" : "fire");
#endif

				if (randint(100) < plev)
				{
					switch (p_ptr->pclass)
					{
						case CLASS_WARRIOR:
						case CLASS_RANGER:
						case CLASS_ARCHER:
							if (randint(3) == 1)
							{
								Type = GF_MISSILE;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "the elements";
#endif
							}
							else
							{
								Type = GF_SHARDS;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "shards";
#endif
							}
							break;
						case CLASS_MAGE:
						case CLASS_WARRIOR_MAGE:
						case CLASS_HIGH_MAGE:
						case CLASS_ARCHAEOLOGIST:
							if (randint(3) == 1)
							{
								Type = GF_MANA;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "mana";
#endif
							}
							else
							{
								Type = GF_DISENCHANT;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "disenchantment";
#endif
							}
							break;
						case CLASS_MONK:
							if (randint(3) != 1)
							{
								Type = GF_CONFUSION;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "confusion";
#endif
							}
							else
							{
								Type = GF_SOUND;
#ifdef JP
								Type_desc = "첻";
#else
								Type_desc = "sound";
#endif
							}
							break;
						case CLASS_MINDCRAFTER:
							if (randint(3) != 1)
							{
								Type = GF_CONFUSION;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "confusion";
#endif
							}
							else
							{
								Type = GF_PSI;
#ifdef JP
								Type_desc = "ͥ륮";
#else
								Type_desc = "mental energy";
#endif
							}
							break;
						case CLASS_PRIEST:
						case CLASS_PALADIN:
							if (randint(3) == 1)
							{
								Type = GF_HELL_FIRE;
#ifdef JP
								Type_desc = "Ϲι";
#else
								Type_desc = "hellfire";
#endif
							}
							else
							{
								Type = GF_HOLY_FIRE;
#ifdef JP
								Type_desc = "ʤ";
#else
								Type_desc = "holy fire";
#endif
							}
							break;
						case CLASS_ROGUE:
							if (randint(3) == 1)
							{
								Type = GF_DARK;
#ifdef JP
								Type_desc = "Ź";
#else
								Type_desc = "darkness";
#endif
							}
							else
							{
								Type = GF_POIS;
#ifdef JP
								Type_desc = "";
#else
								Type_desc = "poison";
#endif
							}
							break;
					}
				}

				if (!get_aim_dir(&dir)) break;
#ifdef JP
				msg_format("ʤ%sΥ֥쥹Ǥ", Type_desc);
#else
				msg_format("You breathe %s.", Type_desc);
#endif
				fire_ball(Type, dir, plev * 2,
				    (plev / 15) + 1);
			}
			break;

		case RACE_MIND_FLAYER:
			if (racial_aux(15, 12, A_INT, 14))
			{
				if (!get_aim_dir(&dir)) break;
				else
				{
#ifdef JP
					msg_print("ʤϽ椷ܤ֤...");
#else
					msg_print("You concentrate and your eyes glow red...");
#endif
					fire_bolt(GF_PSI, dir, plev);
				}
			}
			break;

		case RACE_IMP:
			if (racial_aux(9, 15, A_WIS, 15))
			{
				if (!get_aim_dir(&dir)) break;
				if (plev >= 30)
				{
#ifdef JP
					msg_print("եܡä");
#else
					msg_print("You cast a ball of fire.");
#endif
					fire_ball(GF_FIRE, dir, plev, 2);
				}
				else
				{
#ifdef JP
					msg_print("եܥȤä");
#else
					msg_print("You cast a bolt of fire.");
#endif
					fire_bolt(GF_FIRE, dir, plev);
				}
			}
			break;

		case RACE_GOLEM:
			if (racial_aux(20, 15, A_CON, 8))
			{
				(void)set_shield(p_ptr->shield + randint(20) + 30);
			}
			break;

		case RACE_SKELETON:
		case RACE_ZOMBIE:
			if (racial_aux(30, 30, A_WIS, 18))
			{
#ifdef JP
				msg_print("ʤϼäͥ륮᤽Ȼߤ");
#else
				msg_print("You attempt to restore your lost energies.");
#endif
				(void)restore_level();
			}
			break;

		case RACE_VAMPIRE:
			if (racial_aux(2, (1 + (plev / 3)), A_CON, 9))
			{
				int y, x, dummy = 0;
				cave_type *c_ptr;

				/* Only works on adjacent monsters */
				if (!get_rep_dir(&dir)) break;   /* was get_aim_dir */
				y = py + ddy[dir];
				x = px + ddx[dir];
				c_ptr = &cave[y][x];

				if (!c_ptr->m_idx)
				{
#ifdef JP
					msg_print("ʤ˳ߤĤ");
#else
					msg_print("You bite into thin air!");
#endif
					break;
				}

#ifdef JP
				msg_print("ʤϥ˥ȤƲत...");
#else
				msg_print("You grin and bare your fangs...");
#endif
				dummy = plev + randint(plev) * MAX(1, plev / 10);   /* Dmg */
				if (drain_life(dir, dummy))
				{
					if (p_ptr->food < PY_FOOD_FULL)
						/* No heal if we are "full" */
						(void)hp_player(dummy);
					else
#ifdef JP
						msg_print("ʤ϶ʢǤϤޤ");
#else
						msg_print("You were not hungry.");
#endif

						/* Gain nutritional sustenance: 150/hp drained */
						/* A Food ration gives 5000 food points (by contrast) */
						/* Don't ever get more than "Full" this way */
						/* But if we ARE Gorged,  it won't cure us */
						dummy = p_ptr->food + MIN(5000, 100 * dummy);
					if (p_ptr->food < PY_FOOD_MAX)   /* Not gorged already */
						(void)set_food(dummy >= PY_FOOD_MAX ? PY_FOOD_MAX - 1 : dummy);
				}
				else
#ifdef JP
					msg_print("Ҥɤ̣");
#else
					msg_print("Yechh. That tastes foul.");
#endif
			}
			break;

		case RACE_SPECTRE:
			if (racial_aux(4, 6, A_INT, 3))
			{
#ifdef JP
				msg_print("ʤϤɤɤ򤢤");
#else
				msg_print("You emit an eldritch howl!");
#endif
				if (!get_aim_dir(&dir)) break;
				(void)fear_monster(dir, plev);
			}
			break;

		case RACE_SPRITE:
			if (racial_aux(12, 12, A_INT, 15))
			{
#ifdef JP
				msg_print("ʤˡʴꤲĤ...");
#else
				msg_print("You throw some magic dust...");
#endif
				if (plev < 25)
					sleep_monsters_touch();
				else
					(void)sleep_monsters();
			}
			break;

		default:
#ifdef JP
			msg_print("μ²üǽϤäƤޤ");
#else
			msg_print("This race has no bonus power.");
#endif
			energy_use = 0;
	}

	/* Redraw mana and hp */
	p_ptr->redraw |= (PR_HP | PR_MANA);

	/* Window stuff */
	p_ptr->window |= (PW_PLAYER | PW_SPELL);
}


typedef struct power_desc_type power_desc_type;

struct power_desc_type
{
	char name[40];
	int  level;
	int  cost;
	int  fail;
	int  number;
};


/*
 * Allow user to choose a power (racial / mutation) to activate
 */
void do_cmd_racial_power(void)
{
	power_desc_type power_desc[36];
	int  num, ask, i = 0;
	int  lvl = p_ptr->lev;
	int use_stat;
	bool flag, redraw;
	bool warrior = ((p_ptr->pclass == CLASS_WARRIOR) ? TRUE : FALSE);
	char choice;
	char out_val[160];


	for (num = 0; num < 36; num++)
	{
		strcpy(power_desc[num].name, "");
		power_desc[num].number = 0;
	}

	num = 0;

	if (p_ptr->confused)
	{
#ifdef JP
		msg_print("𤷤ƤüǽϤȤޤ");
#else
		msg_print("You are too confused to use any powers!");
#endif
		energy_use = 0;
		return;
	}

	switch (p_ptr->pclass)
	{
		case CLASS_WARRIOR:
		case CLASS_PALADIN:
		case CLASS_WARRIOR_MAGE:
#ifdef JP
			strcpy(power_desc[num].name, "ɡХå");
#else
			strcpy(power_desc[num].name, "Shield bashing");
#endif
			power_desc[num].level = 1;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(1, A_DEX, 0);
			power_desc[num++].number = -3;
			break;
		case CLASS_MAGE:
		case CLASS_PRIEST:
		case CLASS_MINDCRAFTER:
#ifdef JP
			strcpy(power_desc[num].name, "߿");
#else
			strcpy(power_desc[num].name, "Clear mind");
#endif
			use_stat = m_info[p_ptr->pclass].spell_stat;
			power_desc[num].level = 15;
			power_desc[num].cost = 0;
			power_desc[num].fail = 100 - racial_chance(15, use_stat, 10);
			power_desc[num++].number = -3;
			break;
		case CLASS_RANGER:
		case CLASS_ARCHER:
#ifdef JP
			strcpy(power_desc[num].name, "/δ");
#else
			strcpy(power_desc[num].name, "Idenfify bows/ammos");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(15, A_INT, 15);
			power_desc[num++].number = -3;
			break;
		case CLASS_ARCHAEOLOGIST:
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "Detect treasure");
#endif
			power_desc[num].level = 5;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(5, A_INT, 10);
			power_desc[num++].number = -3;
#ifdef JP
			strcpy(power_desc[num].name, "ˡ");
#else
			strcpy(power_desc[num].name, "Detect magical objs");
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(15, A_INT, 15);
			power_desc[num++].number = -4;
			break;
		default:
#ifdef JP
			strcpy(power_desc[num].name, "(ʤ)");
#else
			strcpy(power_desc[num].name, "(none)");
#endif
			break;
	}

	switch (p_ptr->prace)
	{
		case RACE_DWARF:
#ifdef JP
			strcpy(power_desc[num].name, "ɥ ");
#else
			strcpy(power_desc[num].name, "detect doors+traps");
#endif
			power_desc[num].level = 5;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(5, A_WIS, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_NIBELUNG:
#ifdef JP
			strcpy(power_desc[num].name, "ɥ ");
#else
			strcpy(power_desc[num].name, "detect doors+traps");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(10, A_WIS, 10);
			power_desc[num++].number = -1;
			break;
		case RACE_HOBBIT:
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "create food");
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 10;
			power_desc[num].fail = 100 - racial_chance(15, A_INT, 10);
			power_desc[num++].number = -1;
			break;
		case RACE_GNOME:
#ifdef JP
			sprintf(power_desc[num].name, "硼ȡƥݡ");
#else
			sprintf(power_desc[num].name, "Phase door");
#endif
			power_desc[num].level = 5;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(5, A_INT, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_HALF_ORC:
#ifdef JP
			strcpy(power_desc[num].name, "ݽ");
#else
			strcpy(power_desc[num].name, "remove fear");
#endif
			power_desc[num].level = 3;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(3, A_WIS, (warrior ? 5 : 10));
			power_desc[num++].number = -1;
			break;
		case RACE_HALF_TROLL:
#ifdef JP
			strcpy(power_desc[num].name, "ò");
#else
			strcpy(power_desc[num].name, "berserk");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(10, A_WIS, (warrior ? 6 : 12));
			power_desc[num++].number = -1;
			break;
		case RACE_BARBARIAN:
#ifdef JP
			strcpy(power_desc[num].name, "ò");
#else
			strcpy(power_desc[num].name, "berserk");
#endif
			power_desc[num].level = 8;
			power_desc[num].cost = 10;
			power_desc[num].fail = 100 - racial_chance(8, A_WIS, (warrior ? 6 : 12));
			power_desc[num++].number = -1;
			break;
		case RACE_AMBERITE:
#ifdef JP
			strcpy(power_desc[num].name, "ɥե");
#else
			strcpy(power_desc[num].name, "Shadow Shifting");
#endif
			power_desc[num].level = 30;
			power_desc[num].cost = 50;
			power_desc[num].fail = 100 - racial_chance(30, A_INT, 50);
			power_desc[num++].number = -1;
#ifdef JP
			strcpy(power_desc[num].name, "ѥ󡦥");
#else
			strcpy(power_desc[num].name, "Pattern Mindwalking");
#endif
			power_desc[num].level = 40;
			power_desc[num].cost = 75;
			power_desc[num].fail = 100 - racial_chance(40, A_WIS, 50);
			power_desc[num++].number = -2;
			break;
		case RACE_HALF_OGRE:
#ifdef JP
			strcpy(power_desc[num].name, "ȯΥ롼");
#else
			strcpy(power_desc[num].name, "explosive rune");
#endif
			power_desc[num].level = 25;
			power_desc[num].cost = 35;
			power_desc[num].fail = 100 - racial_chance(25, A_INT, 15);
			power_desc[num++].number = -1;
			break;
		case RACE_HALF_GIANT:
#ifdef JP
			strcpy(power_desc[num].name, "ϲ");
#else
			strcpy(power_desc[num].name, "stone to mud");
#endif
			power_desc[num].level = 20;
			power_desc[num].cost = 10;
			power_desc[num].fail = 100 - racial_chance(20, A_STR, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_HALF_TITAN:
#ifdef JP
			strcpy(power_desc[num].name, "󡦥󥹥");
#else
			strcpy(power_desc[num].name, "probing");
#endif
			power_desc[num].level = 35;
			power_desc[num].cost = 20;
			power_desc[num].fail = 100 - racial_chance(35, A_INT, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_CYCLOPS:
#ifdef JP
			sprintf(power_desc[num].name, "ꤲʥ᡼ %d", (3 * lvl) / 2);
#else
			sprintf(power_desc[num].name, "throw boulder (dam %d)", (3 * lvl) / 2);
#endif
			power_desc[num].level = 20;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(20, A_STR, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_YEEK:
#ifdef JP
			strcpy(power_desc[num].name, "󥹥");
#else
			strcpy(power_desc[num].name, "scare monster");
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(15, A_WIS, 10);
			power_desc[num++].number = -1;
			break;
		case RACE_SPECTRE:
#ifdef JP
			strcpy(power_desc[num].name, "󥹥");
#else
			strcpy(power_desc[num].name, "scare monster");
#endif
			power_desc[num].level = 4;
			power_desc[num].cost = 6;
			power_desc[num].fail = 100 - racial_chance(4, A_INT, 3);
			power_desc[num++].number = -1;
			break;
		case RACE_KLACKON:
#ifdef JP
			sprintf(power_desc[num].name, " (᡼ %d)", lvl);
#else
			sprintf(power_desc[num].name, "spit acid (dam %d)", lvl);
#endif
			power_desc[num].level = 9;
			power_desc[num].cost = 9;
			power_desc[num].fail = 100 - racial_chance(9, A_DEX, 14);
			power_desc[num++].number = -1;
			break;
		case RACE_KOBOLD:
#ifdef JP
			sprintf(power_desc[num].name, "ǤΥ (᡼ %d)", lvl);
#else
			sprintf(power_desc[num].name, "poison dart (dam %d)", lvl);
#endif
			power_desc[num].level = 12;
			power_desc[num].cost = 8;
			power_desc[num].fail = 100 - racial_chance(12, A_DEX, 14);
			power_desc[num++].number = -1;
			break;
		case RACE_DARK_ELF:
#ifdef JP
			sprintf(power_desc[num].name, "ޥåߥ (᡼ %dd%d)", 3 + ((lvl - 1) / 5), 4);
#else
			sprintf(power_desc[num].name, "magic missile (dm %dd%d)", 3 + ((lvl - 1) / 5), 4);
#endif
			power_desc[num].level = 2;
			power_desc[num].cost = 2;
			power_desc[num].fail = 100 - racial_chance(2, A_INT, 9);
			power_desc[num++].number = -1;
			break;
		case RACE_DRACONIAN:
#ifdef JP
			sprintf(power_desc[num].name, "֥쥹 (᡼ %d)", lvl * 2);
#else
			sprintf(power_desc[num].name, "breath weapon (dam %d)", lvl * 2);
#endif
			power_desc[num].level = 1;
			power_desc[num].cost = lvl;
			power_desc[num].fail = 100 - racial_chance(1, A_CON, 12);
			power_desc[num++].number = -1;
			break;
		case RACE_MIND_FLAYER:
#ifdef JP
			sprintf(power_desc[num].name, " (᡼ %d)", lvl);
#else
			sprintf(power_desc[num].name, "mind blast (dam %d)", lvl);
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(15, A_INT, 14);
			power_desc[num++].number = -1;
			break;
		case RACE_IMP:
#ifdef JP
			sprintf(power_desc[num].name, "եܥ/ܡ (᡼ %d)", lvl);
#else
			sprintf(power_desc[num].name, "fire bolt/ball (dam %d)", lvl);
#endif
			power_desc[num].level = 9;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(9, A_WIS, 15);
			power_desc[num++].number = -1;
			break;
		case RACE_GOLEM:
#ifdef JP
			strcpy(power_desc[num].name, "ȩв ( 1d20+30)");
#else
			strcpy(power_desc[num].name, "stone skin (dur 1d20+30)");
#endif
			power_desc[num].level = 20;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(20, A_CON, 8);
			power_desc[num++].number = -1;
			break;
		case RACE_SKELETON:
		case RACE_ZOMBIE:
#ifdef JP
			strcpy(power_desc[num].name, "̿");
#else
			strcpy(power_desc[num].name, "restore life");
#endif
			power_desc[num].level = 30;
			power_desc[num].cost = 30;
			power_desc[num].fail = 100 - racial_chance(30, A_WIS, 18);
			power_desc[num++].number = -1;
			break;
		case RACE_VAMPIRE:
#ifdef JP
			strcpy(power_desc[num].name, "̿ϵۼ");
#else
			strcpy(power_desc[num].name, "drain life");
#endif
			power_desc[num].level = 2;
			power_desc[num].cost = 1 + (lvl / 3);
			power_desc[num].fail = 100 - racial_chance(2, A_CON, 9);
			power_desc[num++].number = -1;
			break;
		case RACE_SPRITE:
#ifdef JP
			strcpy(power_desc[num].name, "̲ʴ");
#else
			strcpy(power_desc[num].name, "sleeping dust");
#endif
			power_desc[num].level = 12;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(12, A_INT, 15);
			power_desc[num++].number = -1;
			break;
		default:
			break;
	}

	if ((num == 0) && !p_ptr->muta1)
	{
#ifdef JP
		msg_print("ȤüǽϤ⤢ޤ");
#else
		msg_print("You have no powers to activate.");
#endif
		energy_use = 0;
		return;
	}

	if (p_ptr->muta1)
	{
		if (p_ptr->muta1 & MUT1_SPIT_ACID)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "spit acid");
#endif
			power_desc[num].level = 9;
			power_desc[num].cost = 9;
			power_desc[num].fail = 100 - racial_chance(9, A_DEX, 15);
			power_desc[num++].number = MUT1_SPIT_ACID;
		}

		if (p_ptr->muta1 & MUT1_BR_FIRE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "Υ֥쥹");
#else
			strcpy(power_desc[num].name, "fire breath");
#endif
			power_desc[num].level = 20;
			power_desc[num].cost = lvl;
			power_desc[num].fail = 100 - racial_chance(20, A_CON, 18);
			power_desc[num++].number = MUT1_BR_FIRE;
		}

		if (p_ptr->muta1 & MUT1_HYPN_GAZE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "̲ˤ");
#else
			strcpy(power_desc[num].name, "hypnotic gaze");
#endif
			power_desc[num].level = 12;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(12, A_CHR, 18);
			power_desc[num++].number = MUT1_HYPN_GAZE;
		}

		if (p_ptr->muta1 & MUT1_TELEKINES)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ǰư");
#else
			strcpy(power_desc[num].name, "telekinesis");
#endif
			power_desc[num].level = 9;
			power_desc[num].cost = 9;
			power_desc[num].fail = 100 - racial_chance(9, A_WIS, 14);
			power_desc[num++].number = MUT1_TELEKINES;
		}

		if (p_ptr->muta1 & MUT1_VTELEPORT)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ƥݡ");
#else
			strcpy(power_desc[num].name, "teleport");
#endif
			power_desc[num].level = 7;
			power_desc[num].cost = 7;
			power_desc[num].fail = 100 - racial_chance(7, A_WIS, 15);
			power_desc[num++].number = MUT1_VTELEPORT;
		}

		if (p_ptr->muta1 & MUT1_MIND_BLST)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "mind blast");
#endif
			power_desc[num].level = 5;
			power_desc[num].cost = 3;
			power_desc[num].fail = 100 - racial_chance(5, A_WIS, 15);
			power_desc[num++].number = MUT1_MIND_BLST;
		}

		if (p_ptr->muta1 & MUT1_RADIATION)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ǽ");
#else
			strcpy(power_desc[num].name, "emit radiation");
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(15, A_CON, 14);
			power_desc[num++].number = MUT1_RADIATION;
		}

		if (p_ptr->muta1 & MUT1_VAMPIRISM)
		{
#ifdef JP
			strcpy(power_desc[num].name, "۷ɥ쥤");
#else
			strcpy(power_desc[num].name, "vampiric drain");
#endif
			power_desc[num].level = 2;
			power_desc[num].cost = (1 + (lvl / 3));
			power_desc[num].fail = 100 - racial_chance(2, A_CON, 9);
			power_desc[num++].number = MUT1_VAMPIRISM;
		}

		if (p_ptr->muta1 & MUT1_SMELL_MET)
		{
#ifdef JP
			strcpy(power_desc[num].name, "°̳");
#else
			strcpy(power_desc[num].name, "smell metal");
#endif
			power_desc[num].level = 3;
			power_desc[num].cost = 2;
			power_desc[num].fail = 100 - racial_chance(3, A_INT, 12);
			power_desc[num++].number = MUT1_SMELL_MET;
		}

		if (p_ptr->muta1 & MUT1_SMELL_MON)
		{
#ifdef JP
			strcpy(power_desc[num].name, "Ũ̳");
#else
			strcpy(power_desc[num].name, "smell monsters");
#endif
			power_desc[num].level = 5;
			power_desc[num].cost = 4;
			power_desc[num].fail = 100 - racial_chance(5, A_INT, 15);
			power_desc[num++].number = MUT1_SMELL_MON;
		}

		if (p_ptr->muta1 & MUT1_BLINK)
		{
#ifdef JP
			strcpy(power_desc[num].name, "硼ȡƥݡ");
#else
			strcpy(power_desc[num].name, "blink");
#endif
			power_desc[num].level = 3;
			power_desc[num].cost = 3;
			power_desc[num].fail = 100 - racial_chance(3, A_WIS, 12);
			power_desc[num++].number = MUT1_BLINK;
		}

		if (p_ptr->muta1 & MUT1_EAT_ROCK)
		{
#ifdef JP
			strcpy(power_desc[num].name, "俩");
#else
			strcpy(power_desc[num].name, "eat rock");
#endif
			power_desc[num].level = 8;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(8, A_CON, 18);
			power_desc[num++].number = MUT1_EAT_ROCK;
		}

		if (p_ptr->muta1 & MUT1_SWAP_POS)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ָ");
#else
			strcpy(power_desc[num].name, "swap position");
#endif
			power_desc[num].level = 15;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(15, A_DEX, 16);
			power_desc[num++].number = MUT1_SWAP_POS;
		}

		if (p_ptr->muta1 & MUT1_SHRIEK)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "shriek");
#endif
			power_desc[num].level = 20;
			power_desc[num].cost = 14;
			power_desc[num].fail = 100 - racial_chance(20, A_CON, 16);
			power_desc[num++].number = MUT1_SHRIEK;
		}

		if (p_ptr->muta1 & MUT1_ILLUMINE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "illuminate");
#endif
			power_desc[num].level = 3;
			power_desc[num].cost = 2;
			power_desc[num].fail = 100 - racial_chance(3, A_INT, 10);
			power_desc[num++].number = MUT1_ILLUMINE;
		}

		if (p_ptr->muta1 & MUT1_DET_CURSE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "detect curses");
#endif
			power_desc[num].level = 7;
			power_desc[num].cost = 14;
			power_desc[num].fail = 100 - racial_chance(7, A_WIS, 14);
			power_desc[num++].number = MUT1_DET_CURSE;
		}

		if (p_ptr->muta1 & MUT1_BERSERK)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ò");
#else
			strcpy(power_desc[num].name, "berserk");
#endif
			power_desc[num].level = 8;
			power_desc[num].cost = 8;
			power_desc[num].fail = 100 - racial_chance(8, A_STR, 14);
			power_desc[num++].number = MUT1_BERSERK;
		}

		if (p_ptr->muta1 & MUT1_POLYMORPH)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ѿ");
#else
			strcpy(power_desc[num].name, "polymorph");
#endif
			power_desc[num].level = 18;
			power_desc[num].cost = 20;
			power_desc[num].fail = 100 - racial_chance(18, A_CON, 18);
			power_desc[num++].number = MUT1_POLYMORPH;
		}

		if (p_ptr->muta1 & MUT1_MIDAS_TCH)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ߥμ");
#else
			strcpy(power_desc[num].name, "midas touch");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 5;
			power_desc[num].fail = 100 - racial_chance(10, A_INT, 12);
			power_desc[num++].number = MUT1_MIDAS_TCH;
		}

		if (p_ptr->muta1 & MUT1_GROW_MOLD)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ȯ");
#else
			strcpy(power_desc[num].name, "grow mold");
#endif
			power_desc[num].level = 1;
			power_desc[num].cost = 6;
			power_desc[num].fail = 100 - racial_chance(1, A_CON, 14);
			power_desc[num++].number = MUT1_GROW_MOLD;
		}

		if (p_ptr->muta1 & MUT1_RESIST)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "resist elements");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(10, A_CON, 12);
			power_desc[num++].number = MUT1_RESIST;
		}

		if (p_ptr->muta1 & MUT1_EARTHQUAKE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "Ͽ");
#else
			strcpy(power_desc[num].name, "earthquake");
#endif
			power_desc[num].level = 12;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(12, A_STR, 16);
			power_desc[num++].number = MUT1_EARTHQUAKE;
		}

		if (p_ptr->muta1 & MUT1_EAT_MAGIC)
		{
#ifdef JP
			strcpy(power_desc[num].name, "Ͽ");
#else
			strcpy(power_desc[num].name, "eat magic");
#endif
			power_desc[num].level = 17;
			power_desc[num].cost = 1;
			power_desc[num].fail = 100 - racial_chance(17, A_WIS, 15);
			power_desc[num++].number = MUT1_EAT_MAGIC;
		}

		if (p_ptr->muta1 & MUT1_WEIGH_MAG)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ϴ");
#else
			strcpy(power_desc[num].name, "weigh magic");
#endif
			power_desc[num].level = 6;
			power_desc[num].cost = 6;
			power_desc[num].fail = 100 - racial_chance(6, A_INT, 10);
			power_desc[num++].number = MUT1_WEIGH_MAG;
		}

		if (p_ptr->muta1 & MUT1_STERILITY)
		{
#ifdef JP
			strcpy(power_desc[num].name, "˻");
#else
			strcpy(power_desc[num].name, "sterilize");
#endif
			power_desc[num].level = 12;
			power_desc[num].cost = 23;
			power_desc[num].fail = 100 - racial_chance(12, A_CHR, 15);
			power_desc[num++].number = MUT1_STERILITY;
		}

		if (p_ptr->muta1 & MUT1_PANIC_HIT)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ҥåȡ");
#else
			strcpy(power_desc[num].name, "panic hit");
#endif
			power_desc[num].level = 10;
			power_desc[num].cost = 12;
			power_desc[num].fail = 100 - racial_chance(10, A_DEX, 14);
			power_desc[num++].number = MUT1_PANIC_HIT;
		}

		if (p_ptr->muta1 & MUT1_DAZZLE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "dazzle");
#endif
			power_desc[num].level = 7;
			power_desc[num].cost = 15;
			power_desc[num].fail = 100 - racial_chance(7, A_CHR, 8);
			power_desc[num++].number = MUT1_DAZZLE;
		}

		if (p_ptr->muta1 & MUT1_LASER_EYE)
		{
#ifdef JP
			strcpy(power_desc[num].name, "졼");
#else
			strcpy(power_desc[num].name, "laser eye");
#endif
			power_desc[num].level = 7;
			power_desc[num].cost = 10;
			power_desc[num].fail = 100 - racial_chance(7, A_WIS, 9);
			power_desc[num++].number = MUT1_LASER_EYE;
		}

		if (p_ptr->muta1 & MUT1_RECALL)
		{
#ifdef JP
			strcpy(power_desc[num].name, "");
#else
			strcpy(power_desc[num].name, "recall");
#endif
			power_desc[num].level = 17;
			power_desc[num].cost = 50;
			power_desc[num].fail = 100 - racial_chance(17, A_INT, 16);
			power_desc[num++].number = MUT1_RECALL;
		}

		if (p_ptr->muta1 & MUT1_BANISH)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ٰ");
#else
			strcpy(power_desc[num].name, "banish evil");
#endif
			power_desc[num].level = 25;
			power_desc[num].cost = 25;
			power_desc[num].fail = 100 - racial_chance(25, A_WIS, 18);
			power_desc[num++].number = MUT1_BANISH;
		}

		if (p_ptr->muta1 & MUT1_COLD_TOUCH)
		{
#ifdef JP
			strcpy(power_desc[num].name, "μ");
#else
			strcpy(power_desc[num].name, "cold touch");
#endif
			power_desc[num].level = 2;
			power_desc[num].cost = 2;
			power_desc[num].fail = 100 - racial_chance(2, A_CON, 11);
			power_desc[num++].number = MUT1_COLD_TOUCH;
		}

		if (p_ptr->muta1 & MUT1_LAUNCHER)
		{
#ifdef JP
			strcpy(power_desc[num].name, "ƥꤲ");
#else
			strcpy(power_desc[num].name, "throw object");
#endif
			power_desc[num].level = 1;
			power_desc[num].cost = lvl;
			power_desc[num].fail = 100 - racial_chance(1, A_STR, 6);
			/* XXX_XXX_XXX Hack! MUT1_LAUNCHER counts as negative... */
			power_desc[num++].number = 3;
		}
	}

	/* Nothing chosen yet */
	flag = FALSE;

	/* No redraw yet */
	redraw = FALSE;

	/* Build a prompt */
#ifdef JP
	(void) strnfmt(out_val, 78, "(üǽ %c-%c, *'ǰ, ESC) ɤüǽϤȤޤ",
#else
	(void)strnfmt(out_val, 78, "(Powers %c-%c, *=List, ESC=exit) Use which power? ",
#endif
		I2A(0), (num <= 26) ? I2A(num - 1) : '0' + num - 27);

#ifdef ALLOW_REPEAT
	if (!repeat_pull(&i) || i<0 || i>=num) {
#endif /* ALLOW_REPEAT */

	/* Get a spell from the user */
	choice = always_show_list ? ESCAPE : 1;
	while (!flag)
	{
		if( choice==ESCAPE ) choice = ' '; 
		else if( !get_com(out_val, &choice) )break; 

		/* Request redraw */
		if ((choice == ' ') || (choice == '*') || (choice == '?'))
		{
			/* Show the list */
			if (!redraw)
			{
				byte y = 1, x = 0;
				int ctr = 0;
				char dummy[80];
				char letter;
				int x1, y1;

				strcpy(dummy, "");

				/* Show list */
				redraw = TRUE;

				/* Save the screen */
				screen_save();

				/* Print header(s) */
				if (num < 17)
#ifdef JP
					prt("                            Lv   MP Ψ", y++, x);
#else
					prt("                            Lv Cost Fail", y++, x);
#endif
				else
#ifdef JP
					prt("                            Lv   MP Ψ                            Lv   MP Ψ", y++, x);
#else
					prt("                            Lv Cost Fail                            Lv Cost Fail", y++, x);
#endif

				/* Print list */
				while (ctr < num)
				{
					/* letter/number for power selection */
					if (ctr < 26)
						letter = I2A(ctr);
					else
						letter = '0' + ctr - 26;
					x1 = ((ctr < 17) ? x : x + 40);
					y1 = ((ctr < 17) ? y + ctr : y + ctr - 17);

					sprintf(dummy, " %c) %-23.23s %2d %4d %3d%%", letter, power_desc[ctr].name, power_desc[ctr].level, power_desc[ctr].cost, power_desc[ctr].fail);
					prt(dummy, y1, x1);
					ctr++;
				}
			}

			/* Hide the list */
			else
			{
				/* Hide list */
				redraw = FALSE;

				/* Restore the screen */
				screen_load();
			}

			/* Redo asking */
			continue;
		}

		if (choice == '\r' && num == 1)
		{
			choice = 'a';
		}

		if (isalpha(choice))
		{
			/* Note verify */
			ask = (isupper(choice));

			/* Lowercase */
			if (ask) choice = tolower(choice);

			/* Extract request */
			i = (islower(choice) ? A2I(choice) : -1);
		}
		else
		{
			ask = FALSE; /* Can't uppercase digits */

			i = choice - '0' + 26;
		}

		/* Totally Illegal */
		if ((i < 0) || (i >= num))
		{
			bell();
			continue;
		}

		/* Verify it */
		if (ask)
		{
			char tmp_val[160];

			/* Prompt */
#ifdef JP
			(void) strnfmt(tmp_val, 78, "%sȤޤ ", power_desc[i].name);
#else
			(void)strnfmt(tmp_val, 78, "Use %s? ", power_desc[i].name);
#endif

			/* Belay that order */
			if (!get_check(tmp_val)) continue;
		}

		/* Stop the loop */
		flag = TRUE;
	}

	/* Restore the screen */
	if (redraw) screen_load();

	/* Abort if needed */
	if (!flag)
	{
		energy_use = 0;
		return;
	}

#ifdef ALLOW_REPEAT
	repeat_push(i); } /* if (!repeat_pull(&i) || ...) */
#endif /* ALLOW_REPEAT */

	if (power_desc[i].number < 0)
	{
		cmd_racial_power_aux(power_desc[i].number);
	}
	else
	{
		mutation_power_aux(power_desc[i].number);
	}

	/* Success */
	return;
}
