'use strict';

var request = require('request');
const neo4j = require('neo4j-driver').v1;
var logger = require('../common/log-utils.js');

var graphDb = process.env.GRAPH_DB_IP || '52.11.150.230';
const uri = 'bolt://' + graphDb + ':7687';
const user = process.env.GRAPH_USER || '';
const password = process.env.GRAPH_PWD || '';
const url = 'http://' + graphDb + ':7474/db/data/transaction/commit';
const C_MODEL = '[xmail]';

/*
 * [UI-1] XMAILデータ一覧
 *・GrpahDBへ接続
 *・Cypher生成
 *・Cypher実行
 *・結果データセットからGraphJsonを生成
 *
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns graphJson - XMAILデータ一覧JSON
 * @throws {cypher error}
 */
exports.list = async function() {
	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	const session = driver.session();
	logger.app.debug(C_MODEL + 'Graph DB connected.');

	var cypher = `match (a:XMAIL)-[:XML_Root]->(d)-[:XML_Child]->(do {__tag: 'document'})
                    optional match (do)-[:XML_Child]->(uu {__tag: 'uuid'})-[:XML_Data]->(uud)
                    optional match (do)-[:XML_Child]->(na {__tag: 'name'})-[:XML_Data]->(nad)
                    optional match (do)-[:XML_Child]->(de {__tag: 'description'})-[:XML_Data]->(ded)
                    optional match (do)-[:XML_Child]->(cr {__tag: 'creator'})
                    optional match (do)-[:XML_Child]->(ve {__tag: 'vendor'})
                    optional match (do)-[:XML_Child]->(ow {__tag: 'owner'})
                    optional match (cr)-[:XML_Child]->(cruu {__tag: 'uuid'})-[:XML_Data]->(cruud)
                    optional match (cr)-[:XML_Child]->(crna {__tag: 'name'})-[:XML_Data]->(crnad)
                    optional match (cr)-[:XML_Child]->(crde {__tag: 'description'})-[:XML_Data]->(crded)
                    optional match (ve)-[:XML_Child]->(veuu {__tag: 'uuid'})-[:XML_Data]->(veuud)
                    optional match (ve)-[:XML_Child]->(vena {__tag: 'name'})-[:XML_Data]->(venad)
                    optional match (ve)-[:XML_Child]->(vede {__tag: 'description'})-[:XML_Data]->(veded)
                    optional match (ow)-[:XML_Child]->(owuu {__tag: 'uuid'})-[:XML_Data]->(owuud)
                    optional match (ow)-[:XML_Child]->(owna {__tag: 'name'})-[:XML_Data]->(ownad)
                    optional match (ow)-[:XML_Child]->(owde {__tag: 'description'})-[:XML_Data]->(owded)
                return
                    id(a) as nid,
                    a.file as file,
                    uud.value as xmail_uuid,
                    nad.value as xmail_name,
                    ded.value as xmail_description,
                    cruud.value as creator_uuid,
                    crnad.value as creator_name,
                    crded.value as creator_description,
                    veuud.value as vendor_uuid,
                    venad.value as vendor_name,
                    veded.value as vendor_description,
                    owuud.value as owner_uuid,
                    ownad.value as owner_name,
                    owded.value as owner_description;`;
	logger.app.debug(C_MODEL + 'Cypher : ' + cypher.toString());

	var graphJson = '[';
	var idx = 0;

	await session
		.run(cypher)
		.then(function(result) {
			result.records.forEach(function(record) {
				session.close();

				if (idx >= 1) {
					graphJson = graphJson + ',';
				}

				graphJson =
					graphJson +
					'{"del_link": "' +
					record.get('nid') +
					'", ' +
					'"nid": "' +
					record.get('nid') +
					'", ' +
					'"file": "' +
					record.get('file') +
					'", ' +
					'"xmail_uuid": "' +
					record.get('xmail_uuid') +
					'", ' +
					'"xmail_name": "' +
					record.get('xmail_name') +
					'", ' +
					'"xmail_description": "' +
					record.get('xmail_description') +
					'", ' +
					'"creator_uuid": "' +
					record.get('creator_uuid') +
					'", ' +
					'"creator_name": "' +
					record.get('creator_name') +
					'", ' +
					'"creator_description": "' +
					record.get('creator_description') +
					'", ' +
					'"vendor_uuid": "' +
					record.get('vendor_uuid') +
					'", ' +
					'"vendor_name": "' +
					record.get('vendor_name') +
					'", ' +
					'"vendor_description": "' +
					record.get('vendor_description') +
					'", ' +
					'"owner_uuid": "' +
					record.get('owner_uuid') +
					'", ' +
					'"owner_name": "' +
					record.get('owner_name') +
					'", ' +
					'"owner_description": "' +
					record.get('owner_description') +
					'"} ';
				idx = idx + 1;
			});

			graphJson = graphJson + ']';
		})
		.catch(function(error) {
			session.close();
			logger.app.error(C_MODEL + error.message);
			throw error;
		});

	return graphJson;
};

/*
 * [UI-2] PN図出力 - nodes & edges
 *・GrpahDBへ接続
 *・Cypher生成
 *・Cypher実行
 *・結果データセットからGraphJsonを生成
 *
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns graphJson - ペトリネットJSON
 * @throws {cypher error}
 */
exports.pn_nodes = async function(id) {
	var cypher =
		`match (a:XMAIL)
                    -[:XML_Root]->(d)
                    -[:XML_Child]->(pr {__tag: 'protocol'})
                    -[:XML_Child]->(me {__tag: 'method'})
                    -[:XML_Child]->(pg {__tag: 'program'})
                    -[:XML_Child]->(pn {__tag: 'pnml'})
                    -[:XML_Child]->(s {__tag: 'place'}),
                    m=(s)-[:PNarc*0..1]-(t)
                where id(a)=` +
		id +
		`
                unwind relationships(m) as edge
                return collect(distinct edge);
                `;
	logger.app.debug(C_MODEL + 'Cypher : ' + cypher.toString());

	var graphJson = '';
	var idx = 0;

	return new Promise(function(resolve, reject) {
		request.post(
			{
				uri: url,
				json: {
					statements: [
						{
							statement: cypher,
							parameters: {},
							resultDataContents: ['row', 'graph']
						}
					]
				}
			},
			function(error, response, body) {
				if (error) {
					logger.app.error(C_MODEL + error.message);
					reject(error);
				}

				/*
				 * nodes json
				 */
				graphJson = '{"nodes":[';
				body.results[0].data[0].graph.nodes.forEach(function(record) {
					if (idx >= 1) {
						graphJson = graphJson + ',';
					}

					graphJson =
						graphJson +
						'{"id": "' +
						record.id +
						'", ' +
						' "pid": "' +
						record.properties.id +
						'", ' +
						' "label": "[' +
						record.id +
						'] ' +
						record.properties.id +
						'", ' +
						' "nodeType": "' +
						record.properties.__tag +
						'"} ';

					idx = idx + 1;
				});
				graphJson = graphJson + '], ';

				/*
				 * edges json
				 */
				idx = 0;
				graphJson = graphJson + '"edges":[';
				body.results[0].data[0].graph.relationships.forEach(function(
					record
				) {
					if (idx >= 1) {
						graphJson = graphJson + ',';
					}

					graphJson =
						graphJson +
						'{"source": "' +
						record.startNode +
						'", ' +
						' "target": "' +
						record.endNode +
						'", ' +
						' "edgeType": "' +
						record.type +
						'"} ';

					idx = idx + 1;
				});
				graphJson = graphJson + ']} ';
				logger.app.debug(C_MODEL + 'GraphJson : ' + graphJson);

				resolve(graphJson);
			}
		);
	});
};

/*
 * [UI-3] node詳細情報取得（個別）
 *・GrpahDBへ接続
 *・Cypher生成
 *・Cypher実行
 *・結果データセットからGraphJsonを生成
 *
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns graphJson - node詳細情報JSON
 * @throws {cypher error}
 */
exports.node = async function(node_id) {
	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	const session = driver.session();
	logger.app.debug(C_MODEL + 'Graph DB connected.');

	var cypher =
		`match
                    (pr {__tag: 'protocol'})
                    -[:XML_Child]->(me {__tag: 'method'})
                    -[:XML_Child]->(pg {__tag: 'program'})
                    -[:XML_Child]->(pn {__tag: 'pnml'})
                    -[:XML_Child]->(pl {__tag: 'place'})
                where id(pl)=134
                optional match
                    (pr)-[:XML_Child*]->(te)-[:XML_Child]->(plR {__tag: 'placeRef'})
                where plR.ref = ` +
		node_id +
		`
                optional match
                    (te)-[:XML_Child]->(uu {__tag: 'uuid'})-[:XML_Data]->(uud),
                    (te)-[:XML_Child]->(na {__tag: 'name'})-[:XML_Data]->(nad),
                    (te)-[:XML_Child]->(de {__tag: 'description'})-[:XML_Data]->(ded)
                return
                    id(pl) as place_nid,
                    pl.id as place_id,
                    id(plR) as placeRef_nid,
                    plR.id as placeRef_id,
                    id(te) as template_nid,
                    te.__tag as template_tag,
                    te.id as template_id,
                    uud.value as template_uuid,
                    nad.value as template_name,
                    ded.value as template_description;`;
	logger.app.debug(C_MODEL + 'Cypher : ' + cypher.toString());

	var graphJson = '[';
	var idx = 0;

	await session
		.run(cypher)
		.then(function(result) {
			result.records.forEach(function(record) {
				session.close();

				if (idx >= 1) {
					graphJson = graphJson + ',';
				}

				graphJson =
					graphJson +
					'{"place_nid": "' +
					record.get('place_nid') +
					'", ' +
					'"place_id": "' +
					record.get('place_id') +
					'", ' +
					'"placeRef_nid": "' +
					record.get('placeRef_nid') +
					'", ' +
					'"placeRef_id": "' +
					record.get('placeRef_id') +
					'", ' +
					'"template_nid": "' +
					record.get('template_nid') +
					'", ' +
					'"template_tag": "' +
					record.get('template_tag') +
					'", ' +
					'"template_id": "' +
					record.get('template_id') +
					'", ' +
					'"template_uuid": "' +
					record.get('template_uuid') +
					'", ' +
					'"template_name": "' +
					record.get('template_name') +
					'", ' +
					'"template_description": "' +
					record.get('template_description') +
					'"} ';
				idx = idx + 1;
			});

			graphJson = graphJson + ']';
		})
		.catch(function(error) {
			session.close();
			logger.app.error(C_MODEL + error.message);
			throw error;
		});

	logger.app.debug(C_MODEL + 'GraphJson : ' + graphJson);
	return graphJson;
};

/*
 * [UI-3] node詳細情報取得（一括取得）
 *・GrpahDBへ接続
 *・Cypher生成
 *・Cypher実行
 *・結果データセットからGraphJsonを生成
 *
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns graphJson - node詳細情報JSON
 * @throws {cypher error}
 */
exports.node_all = async function(node_id) {
	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	const session = driver.session();
	logger.app.debug(C_MODEL + 'Graph DB connected.');

	var cypher =
		`match (a:XMAIL)
                    -[:XML_Root]->(d)
                    -[:XML_Child]->(pr {__tag: 'protocol'})
                    -[:XML_Child]->(me {__tag: 'method'})
                    -[:XML_Child]->(pg {__tag: 'program'})
                    -[:XML_Child]->(pn {__tag: 'pnml'})
                    -[:XML_Child]->(pl {__tag: 'place'})
                where id(a)=` +
		node_id +
		`
                optional match
					(pr)-[:XML_Child*]->(te)-[:XML_Child]->(plR {__tag: 'placeRef'})
				where plR.ref = pl.id
                optional match
                    (te)-[:XML_Child]->(uu {__tag: 'uuid'})-[:XML_Data]->(uud),
                    (te)-[:XML_Child]->(na {__tag: 'name'})-[:XML_Data]->(nad),
                    (te)-[:XML_Child]->(de {__tag: 'description'})-[:XML_Data]->(ded)
                return
                    id(pl) as place_nid,
                    pl.id as place_id,
                    id(plR) as placeRef_nid,
                    plR.id as placeRef_id,
                    id(te) as template_nid,
                    te.__tag as template_tag,
                    te.id as template_id,
                    uud.value as template_uuid,
                    nad.value as template_name,
                    ded.value as template_description;`;
	logger.app.debug(C_MODEL + 'Cypher : ' + cypher.toString());

	var graphJson = '[';
	var idx = 0;

	await session
		.run(cypher)
		.then(function(result) {
			result.records.forEach(function(record) {
				session.close();

				if (idx >= 1) {
					graphJson = graphJson + ',';
				}

				graphJson =
					graphJson +
					'{"place_nid": "' +
					record.get('place_nid') +
					'", ' +
					'"place_id": "' +
					record.get('place_id') +
					'", ' +
					'"placeRef_nid": "' +
					record.get('placeRef_nid') +
					'", ' +
					'"placeRef_id": "' +
					record.get('placeRef_id') +
					'", ' +
					'"template_nid": "' +
					record.get('template_nid') +
					'", ' +
					'"template_tag": "' +
					record.get('template_tag') +
					'", ' +
					'"template_id": "' +
					record.get('template_id') +
					'", ' +
					'"template_uuid": "' +
					record.get('template_uuid') +
					'", ' +
					'"template_name": "' +
					record.get('template_name') +
					'", ' +
					'"template_description": "' +
					record.get('template_description') +
					'"} ';
				idx = idx + 1;
			});

			graphJson = graphJson + ']';
		})
		.catch(function(error) {
			session.close();
			logger.app.error(C_MODEL + error.message);
			throw error;
		});

	logger.app.debug(C_MODEL + 'GraphJson : ' + graphJson);
	return graphJson;
};

/*
 * [UI-4] ペトリネットnode詳細情報取得
 *・GrpahDBへ接続
 *・Cypher生成
 *・Cypher実行
 *・結果データセットからGraphJsonを生成
 *
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns graphJson - node詳細情報JSON
 * @throws {cypher error}
 */
exports.node_material = async function(node_id) {
	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	const session = driver.session();
	logger.app.debug(C_MODEL + 'Graph DB connected.');

	var cypher =
		`match
			(pr {__tag: 'protocol'})
			-[:XML_Child]->(me {__tag: 'method'})
			-[:XML_Child]->(pg {__tag: 'program'})
			-[:XML_Child]->(pn {__tag: 'pnml'})
			-[:XML_Child]->(pl {__tag: 'place'}),
			(pr)-[:XML_Child*]->(te)-[:XML_Child]->(plR {__tag: 'placeRef'})
		where
			id(pl)=` +
		node_id +
		`
			and plR.ref = pl.id
		with te
		match
			(te)-[:XML_Child*]->(prop {__tag: 'property'})
		optional match
			(prop)-[:XML_Child]->(v {__tag: 'value'})-[:XML_Data]->(vd)
		optional match
			(prop)-[:XML_Child]->(d {__tag: 'description'})-[:XML_Data]->(dd)
		optional match
			(prop)<-[:XML_Child]-(parent {__tag: 'property'})
		return
			te.__tag as tag_name,
			te.id as template,
			id(prop) as nid,
			id(parent) as parent_nid,
			dd.value as description,
			vd.value as value,
			properties(prop) as attrib;
		`;
	logger.app.debug(C_MODEL + 'Cypher : ' + cypher.toString());

	var graphJson = '[';
	var idx = 0;

	await session
		.run(cypher)
		.then(function(result) {
			result.records.forEach(function(record) {
				session.close();

				if (idx >= 1) {
					graphJson = graphJson + ',';
				}

				var attrib = record.get('attrib');
				delete attrib.__tag;
/*
				var attrValue =
					'type : ' +
					attrib['type'] +
					'<br>__tag : ' +
					attrib['__tag'] +
					'<br>key : ' +
					attrib['key'];
*/
/*
				graphJson =
					graphJson +
					'{"nid": "' +
					record.get('nid') +
					'", ' +
					'"parent_nid": "' +
					record.get('parent_nid') +
					'", ' +
					'"description": "' +
					record.get('description') +
					'", ' +
					'"value": "' +
					record.get('value') +
					'", ' +
					'"attrib": "' +
					attrValue +
					'"} ';
*/
				var arrayTmp = {
					"nid": String(record.get('nid')),
					"parent_nid": String(record.get('parent_nid')),
					"description": record.get('description'),
					"value": record.get('value'),
					"attrib": JSON.stringify(attrib)
				};
				graphJson =
				  graphJson + JSON.stringify(arrayTmp);
				idx = idx + 1;
			});

			graphJson = graphJson + ']';
		})
		.catch(function(error) {
			session.close();
			logger.app.error(C_MODEL + error.message);
			throw error;
		});

	logger.app.debug(C_MODEL + 'GraphJson : ' + graphJson);
	return graphJson;
};
