﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using Const = Yubeshi.Constants.Wgs84;

namespace Yubeshi
{
    public struct Degree
    {
        #region fields
        private double degree;
        private static readonly double degToRad = Const.Pi / 180.0;
        private static readonly double radToDeg = 180.0 / Const.Pi;
        #endregion

        #region constructors
        public Degree(double degree)
        {
            this.degree = degree;
        }

        public Degree(int degree, int minute, int second)
        {
            if ((degree >= 0 && minute >= 0 && second >= 0) ||
                (degree <= 0 && minute <= 0 && second <= 0)) 
            {
                this.degree = (double)degree + minute / 60.0 + second / 3600.0;
            }
            else
            {
                this.degree =
                       -Math.Abs((double)degree)
                       - Math.Abs(minute / 60.0)
                       - Math.Abs(second / 3600.0);
            }
        }

        public Degree(bool negative, uint degree, uint minute, uint second)
        {
            this.degree = (double)degree + minute / 60.0 + second / 3600.0;
            if (negative)
            {
                this.degree = -degree;
            }
        }
        #endregion

        #region operators
        public static implicit operator Degree(double degree)
        {
            return new Degree(degree);
        }

        public static implicit operator double(Degree degree)
        {
            return degree.degree;
        }
        #endregion

        #region properties

        public int Deg
        {
            get
            {
                return (int)degree;
            }
        }

        public int Minute
        {
            get
            {
                return (int)(degree * 60) % 60;
            }
        }

        public int Second
        {
            get
            {
                return (int)(degree * 3600) % 60;
            }
        }

        public double Radian
        {
            get
            {
                return degree * degToRad;
            }
        }
 
        #endregion

        #region public methods

        public static Degree FromRadian(double rad)
        {
            return new Degree(rad * radToDeg);
        }

        public static Degree FromNmeaFormat(string angle)
        {
            return FromNmeaFormat(angle, null);
        }

        public static Degree FromNmeaFormat(string angle, string direction)
        {
            decimal raw;
            if (!Decimal.TryParse(angle, out raw))
            {
                return new Degree(Double.NaN);
            }
            double deg = (double)(Math.Floor(raw / 100m) + (raw % 100m) / 60m);
            if (direction == "S" || direction == "W")
            {
                return new Degree(-deg);
            }
            return new Degree(deg);
        }

        public override string ToString()
        {
            double s = Math.Abs(degree * 3600) % 60.0;
            if (degree < 0)
            {
                return String.Format("-{0}°{1}′{2:G6}″",
                                                -Deg, -Minute, s);
            }
            else
            {
                return String.Format("{0}°{1}′{2:G6}″", Deg, Minute, s);
            }
        }

        public string ToString(string format)
        {
            return degree.ToString(format);
        }

        #endregion
    }
}
