﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Gps
{
    public class Word
    {
        #region fields
        public static readonly Word Null = new Word();
        /// <summary>
        ///     (msb) D29* D30* D1 D2 ... D24 ... D28 D29 D30 (lsb)
        /// </summary>
        protected uint value;
        #endregion

        #region constructor
        public Word()
        {
            value = 0;
        }

        public Word(uint source)
            : this(source, Word.Null)
        {
        }

        public Word(uint source, Word previous)
        {
            if ((previous & 1) != 0)
            {
                source ^= 0xFFFFFF;
            }
            value = (source & 0xFFFFFF) << 6;
            value |= (previous & 3) << 30;
            
            value |= GenerateParity(this);
        }
        #endregion

        #region operator
        public static implicit operator Word(uint value)
        {
            Word w = new Word();
            w.value = value;
            return w;
        }

        public static implicit operator uint(Word word)
        {
            return (uint)word.value;
        }
        #endregion

        #region properties
        public uint Parity
        {
            get
            {
                return value & 0x3F;
            }
        }

        public uint Source
        {
            get
            {
                return (value & 0x3FFFFFC0) >> 6;
            }
        }

        public uint Value
        {
            get
            {
                return value;
            }
        }

        #endregion

        #region public methods
        public static uint GenerateParity(Word word)
        {
            // (msb) 0 0 d1 d2 ... d24 0 0 0 0 D29* D30* (lsb)
            uint s = (word & 0x3FFFFFC0) | (word >> 30);
            if ((word & 0x40000000) != 0)
            {
                s ^= 0x3FFFFFC0;
            }
            uint[] c = new uint[]{      //            111111111122222222223
                                        //   123456789012345678901234567890
                            0x0b7a89c2, // 0b001011011110101000100111000010
                            0x2bb1f341, // 0b101011101100011111001101000001
                            0x1763e681, // 0b010111011000111110011010000001
                            0x2ec7cd02, // 0b101110110001111100110100000010
                            0x1d8f9a41, // 0b011101100011111001101001000001
                            0x3b1f3482, // 0b111011000111110011010010000010
                            };
            for (int i = 0; i < c.Length; ++i)
            {
                c[i] &= s;
                c[i] ^= c[i] >> 16;
                c[i] ^= c[i] >> 8;
                c[i] ^= c[i] >> 4;
                c[i] ^= c[i] >> 2;
                c[i] ^= c[i] >> 1;
            }
            uint parity = 0;
            for (int i = 0; i < c.Length; ++i)
            {
                parity |= (c[i] & 1) << i;
            }
            
            return parity;
        }
        #endregion
    }
}
