/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Nmea
{
    /// <summary>
    ///     Datum Reference
    /// </summary>
    public class GpDtm : Packet
    {
        #region type definitions
        public enum Code
        { 
            Unknown = 0,
            Wgs84 = 84,
            Wgs72 = 72,
            UserDefined = 999,
        }

        #endregion

        #region fields
        private static Dictionary<string, Code> codes;
        private static readonly byte[] header;
        private const int elementNum = 8;

        #endregion

        #region constructors
        
        static GpDtm()
        {
            header = Encoding.ASCII.GetBytes("$GPDTM,");
            codes = new Dictionary<string,Code>();
            codes["W84"] = Code.Wgs84;
            codes["W72"] = Code.Wgs72;
            codes["999"] = Code.UserDefined;
        }
        
        public GpDtm()
        {
        }

        public GpDtm(byte[] sentence)
            : this(sentence, GetElements(sentence, elementNum))
        {
        }

        private GpDtm(byte[] sentence, Elements elements)
            : base(sentence, elements.PacketLength)
        {
            string[] v = elements.Values;
            LocalDatumCode = GetDatumCode(v[0]);
            LocalDatumSubCode = v[1];
            Offset = new GeodeticCoordinate(
                                    Degree.FromNmeaFormat(v[2], v[3]),
                                    Degree.FromNmeaFormat(v[4], v[5]),
                                    Double.Parse(v[6]));
            ReferenceDatumCode = GetDatumCode(v[7]);
            CheckSum = elements.CheckSum;
        }

        #endregion

        #region properties
        public Code LocalDatumCode
        {
            get;
            private set;
        }

        public string LocalDatumSubCode
        {
            get;
            private set;
        }

        public GeodeticCoordinate Offset
        {
            get;
            private set;
        }

        public Code ReferenceDatumCode
        {
            get;
            private set;
        }
        #endregion

        #region public method

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, header, elementNum, Build);
        }

        #endregion

        #region private method

        private static Packet Build(byte[] sentence, Elements elements)
        {
            return new GpDtm(sentence, elements);
        }

        private Code GetDatumCode(string datumCode)
        {
            Code c;
            if (codes.TryGetValue(datumCode, out c))
            {
                return c;
            }
            return Code.Unknown;
        }
        #endregion

    }
}
