/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Nmea
{
    /// <summary>
    ///     GNSS Satellite Fault Detection
    /// </summary>
    public class GpGbs : Packet
    {
        #region fields
        private static readonly byte[] header;
        private const int elementNum = 8;

        #endregion

        #region constructors
        
        static GpGbs()
        {
            header = Encoding.ASCII.GetBytes("$GPGBS,");
        }
        
        public GpGbs()
        {
        }

        public GpGbs(byte[] sentence)
            : this(sentence, GetElements(sentence, elementNum))
        {
        }

        private GpGbs(byte[] sentence, Elements elements)
            : base(sentence, elements.PacketLength)
        {
            string[] v = elements.Values;
            Utc = ParseTime(v[0]);
            LatitudeError = ParseDouble(v[1]);
            LongitudeError = ParseDouble(v[2]);
            AltitudeError = ParseDouble(v[3]);
            FailedSatellite = ParseInt(v[4], -1);
            ProbabilityOfMissedDetection = ParseDouble(v[5]);
            Bias = ParseDouble(v[6]);
            StandardDeviationOfBias = ParseDouble(v[7]);
            CheckSum = elements.CheckSum;
        }

        #endregion

        #region properties
        public TimeSpan Utc
        {
            get;
            private set;
        }

        public double LatitudeError
        {
            get;
            private set;
        }

        public double LongitudeError
        {
            get;
            private set;
        }

        public double AltitudeError
        {
            get;
            private set;
        }

        public int FailedSatellite
        {
            get;
            private set;
        }

        public double ProbabilityOfMissedDetection
        {
            get;
            private set;
        }

        public double Bias
        {
            get;
            private set;
        }

        public double StandardDeviationOfBias
        {
            get;
            private set;
        }
        #endregion

        #region public method

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, header, elementNum, Build);
        }

        #endregion

        #region private method

        private static Packet Build(byte[] sentence, Elements elements)
        {
            return new GpGbs(sentence, elements);
        }

        #endregion
    }
}