﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Nmea
{
    public abstract class Packet : Yubeshi.UnknownPacket
    {
        #region type definitions
        protected delegate Packet PacketBuilder(
                                        byte[] sentence, Elements elements);

        protected class Elements
        {
            public string[] Values;
            public int CheckSum;
            public int PacketLength;
        }

        public enum Status
        {
            Unknown,
            Invalid,
            Valid,
        }
        #endregion

        #region fields

        #endregion

        #region constructors

        public Packet()
        {
        }

        public Packet(byte[] sentence, int length)
            : base(sentence, length)
        {
        }

        #endregion

        #region properties

        public int CheckSum
        {
            get;
            protected set;
        }
        #endregion

        #region public method

        #endregion
        #region protected method

        protected static bool TryParse(byte[] sentence, out UnknownPacket packet,
                        byte[] header, int elementNum, PacketBuilder builder)
        {
            packet = null;
            if (!Match(sentence, header))
            {
                return false;
            }
            Elements elements = GetElements(sentence, elementNum);
            if (elements == null)
            {
                return false;
            }
            try
            {
                packet = builder(sentence, elements);
            }
            catch
            {
                packet = null;
                return false;
            }
            return true;
        }

        protected static Elements GetElements(byte[] sentence, int elementNum)
        {
            int elementIndex = 0;
            int lastCommaPos = 6;
            if (sentence[1] == 'G' && sentence[2] == 'P')
            {
                lastCommaPos = 7;
            }
            if (elementNum < 0)
            {
                elementNum = 0;
                for (int i = lastCommaPos; i < sentence.Length; ++i)
                {
                    if (sentence[i] == ',')
                    {
                        elementNum++;
                        continue;
                    }
                    if (sentence[i] == '*')
                    {
                        elementNum++;
                        break;
                    }
                }
            }
            if (elementNum < 1)
            {
                return null;
            }
            string[] values = new string[elementNum];
            for (int i = lastCommaPos; i < sentence.Length; ++i)
            {
                if (sentence[i] == ',' || sentence[i] == '*')
                {
                    if (elementIndex >= elementNum)
                    {
                        return null;
                    }
                    values[elementIndex++] = Encoding.ASCII.GetString(
                                    sentence, lastCommaPos, i - lastCommaPos);
                    lastCommaPos = i + 1;
                    if (sentence[i] == '*')
                    {
                        break;
                    }
                }
            }
            if (elementIndex != elementNum)
            {
                return null;
            }

            int length = lastCommaPos + 4; // CC, CC, '\r', '\n'

            if (sentence.Length < length)
            {
                return null;
            }

            if (sentence[lastCommaPos + 2] != (byte)'\r' ||
                sentence[lastCommaPos + 3] != (byte)'\n')
            {
                return null;
            }
            string csString = 
                        Encoding.ASCII.GetString(sentence, lastCommaPos, 2);
            Elements elements = new Elements();
            elements.Values = values;
            elements.CheckSum = Int32.Parse(
                        csString, System.Globalization.NumberStyles.HexNumber);
            elements.PacketLength = length;
            return elements;
        }

        protected static bool Match(byte[] sentence, byte[] header)
        {
            if (sentence.Length < header.Length)
            {
                return false;
            }
            for (int i = 0; i < header.Length; ++i)
            {
                if (header[i] != sentence[i])
                {
                    return false;
                }
            }
            return true;
        }

        protected TimeSpan ParseTime(string time)
        {
            int h = Int32.Parse(time.Substring(0, 2));
            int m = Int32.Parse(time.Substring(2, 2));
            decimal ds = Decimal.Parse(time.Substring(4));
            int sec = (int)Math.Floor(ds);
            int msec = (int)((ds - sec) * 1000m);
            return new TimeSpan(0, h, m, sec, msec);
        }

        protected static double GetLength(string value, string unit)
        { 
            double v;
            
            if (!Double.TryParse(value, out v))
            {
                return Double.NaN;
            }
            if (unit == "M")
            {
                return v;
            }
            else
            {
                return Double.NaN;
            }
        }

        protected static int ParseInt(string intString, int defaultValue)
        {
            int v;
            if (!Int32.TryParse(intString, out v))
            {
                return defaultValue;
            }
            return v;
        }

        protected static double ParseDouble(string doubleString)
        {
            return ParseDouble(doubleString, Double.NaN);
        }
        
        protected static double ParseDouble(string doubleString,
                                                        double defaultValue)
        {
            double v;
            if (!Double.TryParse(doubleString, out v))
            {
                return defaultValue;
            }
            return v;
        }

        protected static Status ParseStatus(string status)
        {
            switch (status)
            {
                case "V":
                    return Status.Invalid;
                case "A":
                    return Status.Valid;
                default:
                    return Status.Unknown;
            }
        }
        #endregion
    }
}
