﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public class NavStatus : Packet
    {
        #region type definition
        [Flags]
        public enum NavigationStatus : byte
        {
            GpsFixOk = 0x01,
            DgpsFix = 0x02,
            WeekNumberValid = 0x04,
            TimeOfWeekValid = 0x08,
        }

        public enum DifferentialStatus : byte
        { 
            None = 0,
            PrPrr = 1,
            PrPrrCp = 2,
            HighAccuracyPrPrrCp = 3,
        }
        #endregion

        #region fields

        #endregion
        
        #region constructors

        public NavStatus(byte[] sentence, int length)
            : base(sentence, length)
        {
        }

        #endregion

        #region properties

        public decimal TimeOfWeek
        {
            get
            {
                return BitConverter.ToUInt32(Raw, 6 + 0) * 1e-3m;
            }
        }

        public GpsFixType PositionFixType
        {
            get
            {
                return (GpsFixType)Raw[6 + 4];
            }
        }

        public NavigationStatus NavigationStatusFlags
        {
            get
            {
                return (NavigationStatus)Raw[6 + 5];
            }
        }

        public DifferentialStatus DgpsStatus
        {
            get
            {
                return (DifferentialStatus)(Raw[6 + 6] & 3);
            }
        }

        public TimeSpan TimeToFirstFix
        {
            get
            {
                long ms = BitConverter.ToUInt32(Raw, 6 + 8);
                return new TimeSpan(ms * 10000); // 1e-3 / 100e-9
            }
        }

        public TimeSpan TimeSinceStartup
        {
            get
            { 
                long ms = BitConverter.ToUInt32(Raw, 6 + 12);
                return new TimeSpan(ms * 10000); // 1e-3 / 100e-9
            }
        }

        #endregion

        #region public methods

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, 
                                            MessageID.NavStatus, 16, Build);
        }

        #endregion

        #region private methods

        private static Packet Build(byte[] sentence, int length)
        {
            return new NavStatus(sentence, length);
        }

        #endregion
    }
}
