﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;

namespace Yubeshi
{
    public class YumaAlmanacReader
    {
        #region type definitions
        private delegate void AlmanacSetter(Almanac almanac, string value);
        #endregion

        #region fields
        private string filePath;
        private static readonly Regex yuma = 
            new Regex(@"^([\w\s\(\)/]+):\s*([\d.E+-]+)$", 
                                                    RegexOptions.Compiled);
        
        private static readonly Dictionary<string, AlmanacSetter> setters;
        #endregion


        #region constructors
        public YumaAlmanacReader(string filePath)
        {
            this.filePath = filePath;
        }

        static YumaAlmanacReader()
        {
            setters = new Dictionary<string,AlmanacSetter>();
            setters["ID"] = delegate(Almanac a, string v)
                {
                    a.ID = Int32.Parse(v);  
                };
            setters["Health"] = delegate(Almanac a, string v)
                {
                    a.Health = (Almanac.HealthIndication)Int16.Parse(v);
                };
            setters["Eccentricity"] = delegate(Almanac a, string v)
                {
                    a.Eccentricity = Double.Parse(v);
                };
            setters["Time of Applicability(s)"] = delegate(Almanac a, string v)
                {
                    a.TimeOfApplicability = Decimal.Parse(v);
                };
            setters["Orbital Inclination(rad)"] = delegate(Almanac a, string v)
                {
                    a.OrbitalInclination = Degree.FromRadian(Double.Parse(v));
                };
            setters["Rate of Right Ascen(r/s)"] = delegate(Almanac a, string v)
                {
                    a.RateOfRightAscen = Double.Parse(v);
                };
            setters["SQRT(A)  (m 1/2)"] = delegate(Almanac a, string v)
                {
                    a.SqrtA = Double.Parse(v);
                };
            setters["Right Ascen at Week(rad)"] = delegate(Almanac a, string v)
                {
                    a.RightAscenAtWeek = Degree.FromRadian(Double.Parse(v));
                };
            setters["Argument of Perigee(rad)"] = delegate(Almanac a, string v)
                {
                    a.ArgumentOfPerigee = Degree.FromRadian(Double.Parse(v));
                };
            setters["Mean Anom(rad)"] = delegate(Almanac a, string v)
                {
                    a.MeanAnom = Degree.FromRadian(Double.Parse(v));
                };
            setters["Af0(s)"] = delegate(Almanac a, string v)
                {
                    a.Af0 = Double.Parse(v);
                };
            setters["Af1(s/s)"] = delegate(Almanac a, string v)
                {
                    a.Af1 = Double.Parse(v);
                };
            setters["week"] = delegate(Almanac a, string v)
            {
                a.Week = Int32.Parse(v);
            };
        }
        #endregion

        public Almanac[] Read()
        {
            Almanac[] list = null;
            try
            {
                using (Stream s = new FileStream(filePath,
                            FileMode.Open, FileAccess.Read, FileShare.Read))
                {
                    using (StreamReader sr = new StreamReader(s))
                    {
                        list = Parse(sr);
                    }
                }
            }
            catch
            {
                throw;
            }
            return list;
        }

        private Almanac[] Parse(StreamReader sr)
        {
            List<Almanac> list = new List<Almanac>();
            string line = null;
            Almanac a = null;
            while ((line = sr.ReadLine()) != null)
            {
                MatchCollection m = yuma.Matches(line);
                if (m.Count == 1)
                {
                    string key = m[0].Groups[1].Value;
                    if (setters.ContainsKey(key))
                    {
                        if (a == null)
                        {
                            a = new Almanac();
                        }
                        setters[key](a, m[0].Groups[2].Value);
                    }
                    else
                    {
                        a = null;
                    }
                }
                else if(a != null)
                {
                    list.Add(a);
                    a = null;
                }

            }
            if (a != null)
            {
                list.Add(a);
            }
            return list.ToArray();
        }

    }
}
