import sdl4gcj.Sys;
import sdl4gcj.SDLConstants;
import sdl4gcj.video.*;
import sdl4gcj.event.*;
import java.util.Random;
import java.util.HashMap;

class Sprite implements Rect
{
	private static final int MAX_SPEED = 1;
	private static final Random random = new Random();

	private Surface screen;
	private Surface image;
	private int movableW, movableH;

	private int x, y, w, h, dx = 0, dy = 0;

	/**
	* Call Screen.setVideoMode() before instanciation. 
	**/
	Sprite(Surface image)
	{
		this.screen = Screen.getVideoSurface();

		// init size
		this.image = image;
		this.w = image.getW();
		this.h = image.getH();
		this.movableW = screen.getW() - this.w;
		this.movableH = screen.getH() - this.h;

		// init location
		this.x = random.nextInt(movableW);
		this.y = random.nextInt(movableH);

		// init speed
		while(dx == 0 && dy == 0)
		{
			dx = random.nextInt(MAX_SPEED * 2 + 1) - MAX_SPEED;
			dy = random.nextInt(MAX_SPEED * 2 + 1) - MAX_SPEED;
		}
	}

	// for Rect interface 
	public final int getX() { return this.x; }
	public final int getY() { return this.y; }
	public final int getW() { return this.w; }
	public final int getH() { return this.h; }

	public final void move()
	{
		x += dx;
		if((x < 0) || (x >= movableW))
		{
			dx = -dx;
			x += dx;
		}

		y += dy;
		if((y < 0) || (y >= movableH))
		{
			dy = -dy;
			y += dy;
		}
	}

	public final void draw()
	{
		this.screen.blitSurface(image, x, y);
	}
}

public class TestSprite implements SDLConstants
{
	public static final int DEFAULT_NUMBER_OF_SPRITE = 100;

	private int screenWidth = 640;
	private int screenHeight = 480;
	private int bitsPerPixel = 8;
	private int videoFlags = SDL_SWSURFACE;
	private int numberOfSprite = DEFAULT_NUMBER_OF_SPRITE;

	private Screen screen;
	private Surface icon;

	private int bgColor;
	private Sprite[] sprites;
	private long ticks = 0;
	private int frames = 0;

	public int getScreenWidth() { return screenWidth; }
	public void setScreenWidth(int screenWidth) { this.screenWidth = screenWidth; }
	public int getScreenHeight() { return screenHeight; }
	public void setScreenHeight(int screenHeight) { this.screenHeight = screenHeight; }
	public int getBitsPerPixel() { return bitsPerPixel; }
	public void setBitsPerPixel(int bitsPerPixel) { this.bitsPerPixel = bitsPerPixel; }
	public int getVideoFlags() { return videoFlags; }
	public void setVideoFlags(int videoFlags) { this.videoFlags = videoFlags; }
	public int getNumberOfSprite() { return numberOfSprite; }
	public void setNumberOfSprite(int numberOfSprite) { this.numberOfSprite = numberOfSprite; }

	public void run()
	{
		this.initScreen();
		this.initIcon();
		this.initSprites();
		this.printInfo();
		this.mainLoop();
		this.freeIcon();
	}

	public double getFPS()
	{
		return ((double)frames * 1000)/ticks;
	}

	protected void initScreen()
	{
		int width = this.getScreenWidth();
		int height = this.getScreenHeight();
		int bpp = this.getBitsPerPixel();
		int flags = this.getVideoFlags();
		this.screen = Screen.setVideoMode(width, height, bpp, flags);

		this.bgColor = this.screen.mapRGB(Color.BLACK); 
	}

	protected void initIcon()
	{
		// load icon image and change pixel format
		this.icon = Surface.loadBMP("images/icon.bmp");
		this.icon.setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL);
		this.icon.displayFormat();

		// Run a sample blit to trigger blit acceleration 
		this.screen.blitSurface(this.icon);
		this.screen.fillRect(this.bgColor);
	}

	protected void freeIcon()
	{
		this.icon.freeSurface();
		this.icon = null;
	}

	protected void initSprites()
	{
		this.sprites = new Sprite[this.getNumberOfSprite()];
		for (int i = this.sprites.length-1;i >= 0;i--)
			this.sprites[i] = new Sprite(this.icon);
	}

	protected void updateSprites()
	{
		for (int i = sprites.length-1;i >= 0 ;i--)
		{
			this.sprites[i].move();
			this.sprites[i].draw();
		}
	}

	protected void updateScreen()
	{
		if ((this.screen.getFlags() & SDL_DOUBLEBUF) == SDL_DOUBLEBUF)
			this.screen.flip();
		else
			this.screen.updateRects(sprites);
	}

	protected void mainLoop()
	{
		boolean done = false;
		EventManager event = new EventManager();
		frames = 0;
		System.gc();
		ticks = Sys.getTicks();
		while (!done)
		{
			frames++;
			while(event.pollEvent() > 0)
			{
				switch (event.type)
				{
					case SDL_KEYDOWN:
					case SDL_QUIT:
						done = true;
						break;
					default:
						break;
				}
			}

			this.screen.fillRect(bgColor);
			this.updateSprites();
			this.updateScreen();
		}

		ticks = Sys.getTicks() - ticks;
	}

	protected void printInfo ()
	{
		if (this.screen != null)
		{
			System.out.println("Screen is at " + this.screen.getPixelFormat().getBitsPerPixel() + " bits per pixel");
			int flags = this.screen.getFlags();
			if ((flags & SDL_HWSURFACE) == SDL_HWSURFACE)
				System.out.println("Screen is in video memory");
			else
				System.out.println("Screen is in system memory");

			if ((flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF)
				System.out.println("Screen has double-buffering enabled");
		}

		if (this.icon != null)
		{
			int iconFlags = this.icon.getFlags();
			if ((iconFlags & SDL_HWSURFACE) == SDL_HWSURFACE)
				System.out.println("Sprite is in video memory");
			else
				System.out.println("Sprite is in system memory");

			if ((iconFlags & SDL_HWACCEL) == SDL_HWACCEL)
				System.out.println("Sprite blit uses hardware acceleration");

			if ((iconFlags & SDL_RLEACCEL) == SDL_RLEACCEL)
				System.out.println("Sprite blit uses RLE acceleration");
		}
	}

	public static void main(String[] args)
	{
		
		if (Sys.init(SDL_INIT_VIDEO) != 0)
		{
			System.out.println("init fail.");
			System.exit(0);
		}

		try
		{
			SDLOptionParser optionParser = new SDLOptionParser(640, 480, 8, SDL_SWSURFACE);
			String[] opts = optionParser.parse(args);

			TestSprite testSprite = new TestSprite();
			testSprite.setScreenWidth(optionParser.getScreenWidth());
			testSprite.setScreenHeight(optionParser.getScreenHeight());
			testSprite.setBitsPerPixel(optionParser.getBitsPerPixel());
			testSprite.setVideoFlags(optionParser.getVideoFlags());
			if (opts.length > 0)
			{
				try 
				{
					testSprite.setNumberOfSprite(Integer.parseInt(opts[0]));
				}
				catch (NumberFormatException e) { }
			}
			testSprite.run();

			System.out.println(testSprite.getFPS() + " frames per second");
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			Sys.quit();
		}
	}
}
