/************************************************************
Copyright (C) 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/
#include "sdl4gcj/ttf/Font.h"

#include "sdl4gcj/ttf/FontConstants.h"
#include "sdl4gcj/video/Surface.h"
#include "sdl4gcj/video/Color.h"
#include "sdl4gcj/SDLException.h"

#include <java/lang/CharSequence.h>
#include <gcj/cni.h>
#include <gcj/javaprims.h>
#include <stdlib.h>
#include <string.h>

#include <SDL.h>
#include <SDL_ttf.h>


using namespace sdl4gcj::ttf;
using sdl4gcj::video::Surface;
using sdl4gcj::video::Color;
using java::lang::CharSequence;

// function
static
inline
sdl4gcj::SDLException*
createFreedFontException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Font was already freed."));
}

static
inline
sdl4gcj::SDLException*
createTTFFontException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1(TTF_GetError()));
}


static
inline
SDL_Color
convertToNative(Color* color)
{
	SDL_Color nativeColor;
	nativeColor.r = color->r;
	nativeColor.g = color->g;
	nativeColor.b = color->b;
	return nativeColor;
}

static
inline
void
sizeText(Font* font, CharSequence* text, int* w, int* h)
{
	if (font->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)font->implementation;

	int textLength = text->length();
	Uint16 nativeText[textLength+1];
	for(int i = textLength-1;i >= 0;i--)
	{
		nativeText[i] = text->charAt(i);
	}
	nativeText[textLength] = '\0';

	if(TTF_SizeUNICODE(nativeFont, nativeText, w, h) != 0)
		throw createTTFFontException();
}


// class methods


Font*
Font::openFont(jstring fontFilePath, jint ptSize)
{
	JvInitClass(&Font::class$);

	jbyteArray byteArray = fontFilePath->getBytes();
	int length = byteArray->length;
	jbyte* bytes = elements(byteArray);
	char buffer[length+1];
	memcpy(buffer, bytes, length);
	buffer[length] = '\0';

	TTF_Font* nativeFont = TTF_OpenFont(buffer, ptSize);
	if (nativeFont == NULL) throw createTTFFontException();

	Font* font = new Font();;
	font->implementation = (::gnu::gcj::RawData*)nativeFont;

	return font;
}

Font*
Font::openFontIndex(jstring fontFilePath, jint ptSize, jlong index)
{
	JvInitClass(&Font::class$);

	jbyteArray byteArray = fontFilePath->getBytes();
	int length = byteArray->length;
	jbyte* bytes = elements(byteArray);
	char buffer[length+1];
	memcpy(buffer, bytes, length);
	buffer[length] = '\0';

	TTF_Font* nativeFont = TTF_OpenFontIndex(buffer, ptSize, index);
	if (nativeFont == NULL) throw createTTFFontException();

	Font* font = new Font();;
	font->implementation = (::gnu::gcj::RawData*)nativeFont;

	return font;
}


// instance methods


jint
Font::getFontStyle()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_GetFontStyle(nativeFont);
}


void
Font::setFontStyle(jint fontStyle)
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	TTF_SetFontStyle(nativeFont, fontStyle);
}

jint
Font::fontHeight()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_FontHeight(nativeFont);
}

jint
Font::fontAscent()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_FontAscent(nativeFont);
}

jint
Font::fontDescent()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_FontDescent(nativeFont);
}

jint
Font::fontLineSkip()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_FontLineSkip(nativeFont);
}

jint
Font::fontFaces()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return TTF_FontFaces(nativeFont);
}

jboolean
Font::fontFaceIsFixedWidth()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	return (TTF_FontFaceIsFixedWidth(nativeFont) > 0) ? true : false;
}

::java::lang::String*
Font::fontFaceFamilyName()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	char* nativeFamilyName = TTF_FontFaceFamilyName(nativeFont);

	::java::lang::String* familyName = JvNewStringLatin1(nativeFamilyName);

	return familyName;
}

::java::lang::String*
Font::fontFaceStyleName()
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	char* nativeStyleName = TTF_FontFaceStyleName(nativeFont);

	::java::lang::String* styleName = JvNewStringLatin1(nativeStyleName);;

	return styleName;
}

jint
Font::widthText(CharSequence* text)
{
	int width;
	sizeText(this, text, &width, NULL);
	return width;
}

jint
Font::heightText(CharSequence* text)
{
	int height;
	sizeText(this, text, NULL, &height);
	return height;
}

Surface*
Font::renderTextSolid(CharSequence* text, Color* foregroundColor)
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	int textLength = text->length();
	Uint16 nativeText[textLength+1];
	for(int i = textLength-1;i >= 0;i--)
	{
		nativeText[i] = text->charAt(i);
	}
	nativeText[textLength] = '\0';

	SDL_Color nativeForegroundColor = convertToNative(foregroundColor);
	SDL_Surface* nativeSurface = TTF_RenderUNICODE_Solid(nativeFont, nativeText, nativeForegroundColor);
	if (nativeSurface == NULL) throw createTTFFontException();

	Surface* surface = new Surface();
	surface->implementation = (::gnu::gcj::RawData*)nativeSurface;
	
	return surface;
}

Surface*
Font::renderTextShaded(CharSequence* text, Color* foregroundColor, Color* backgroundColor)
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	int textLength = text->length();
	Uint16 nativeText[textLength+1];
	for(int i = textLength-1;i >= 0;i--)
	{
		nativeText[i] = text->charAt(i);
	}
	nativeText[textLength] = '\0';

	SDL_Color nativeForegroundColor = convertToNative(foregroundColor);
	SDL_Color nativeBackgroundColor = convertToNative(backgroundColor);
	SDL_Surface* nativeSurface = TTF_RenderUNICODE_Shaded(nativeFont,
		nativeText, nativeForegroundColor, nativeBackgroundColor);
	if (nativeSurface == NULL) throw createTTFFontException();

	Surface* surface = new Surface();
	surface->implementation = (::gnu::gcj::RawData*)nativeSurface;
	
	return surface;
}

Surface*
Font::renderTextBlended(CharSequence* text, Color* foregroundColor)
{
	if (this->implementation == NULL) throw createFreedFontException();
	TTF_Font* nativeFont = (TTF_Font*)this->implementation;

	int textLength = text->length();
	Uint16 nativeText[textLength+1];
	for(int i = textLength-1;i >= 0;i--)
	{
		nativeText[i] = text->charAt(i);
	}
	nativeText[textLength] = '\0';

	SDL_Color nativeForegroundColor = convertToNative(foregroundColor);
	SDL_Surface* nativeSurface = TTF_RenderUNICODE_Blended(nativeFont,
		nativeText, nativeForegroundColor);
	if (nativeSurface == NULL) throw createTTFFontException();

	Surface* surface = new Surface();
	surface->implementation = (::gnu::gcj::RawData*)nativeSurface;
	
	return surface;
}

void
Font::closeFont()
{
	if (this->implementation != NULL) 
	{
		TTF_CloseFont((TTF_Font*)this->implementation);
		this->implementation = NULL;
	}
}
