#include "sdl4gcj/video/Surface.h"
#include "sdl4gcj/video/PixelFormat.h"
#include "sdl4gcj/video/Color.h"
#include "sdl4gcj/video/Palette.h"
#include "sdl4gcj/SDLException.h"

#include <gcj/cni.h>
#include <java/lang/String.h>
#include <stdlib.h>
#include <string.h>
#include <SDL.h>


using namespace sdl4gcj::video;

// function
static
inline
sdl4gcj::SDLException*
createFreedSurfaceException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Surface was already freed."));
}

// class methods
::java::lang::String*
Surface::getVideoDriverName()
{
	JvInitClass(&Surface::class$);
	::java::lang::String* result = NULL;
	char buffer[32];

	if (SDL_VideoDriverName(buffer, 32) != NULL)
		result = JvNewStringLatin1(buffer);
	
	return result;
}

Surface *
Surface::loadBMP(::java::lang::String* filePath)
{
	JvInitClass(&Surface::class$);
	Surface* result = NULL;

	jbyteArray byteArray = filePath->getBytes();
	int length = byteArray->length;
	jbyte* bytes = elements(byteArray);
	char buffer[length+1];

	memcpy(buffer, bytes, length);
	buffer[length] = '\0';

	SDL_Surface* image = SDL_LoadBMP(buffer);
	if (image != NULL)
	{
		result = new Surface();
		result->implementation = (::gnu::gcj::RawData*)image;
	}

	return result;
}

Surface *
Surface::loadBMP(JArray<jbyte>* byteArray, jint offset, jint length)
{
	JvInitClass(&Surface::class$);
	Surface* result = NULL;
	jbyte* bytes = elements(byteArray) + offset;

	SDL_Surface* image = SDL_LoadBMP_RW(SDL_RWFromMem(bytes, length), 1);
	if (image != NULL)
	{
		result = new Surface();
		result->implementation = (::gnu::gcj::RawData*)image;
	}

	return result;
}


// instance methods

jint 
Surface::getFlags ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)implementation;
	return (jint)nativeSurface->flags;
}

jint 
Surface::getW ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	jint w = (jint)((SDL_Surface*)this->implementation)->w;
	return w;
}

jint 
Surface::getH ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	jint h = (jint)((SDL_Surface*)this->implementation)->h;
	return h;
}

jint 
Surface::getPitch ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	jint pitch = (jint)((SDL_Surface*)this->implementation)->pitch;
	return pitch;
}

void
Surface::saveBMP(::java::lang::String* filePath)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	jbyteArray byteArray = filePath->getBytes();
	int length = byteArray->length;
	jbyte* bytes = elements(byteArray);
	char buffer[length+1];

	memcpy(buffer, bytes, length);
	buffer[length] = '\0';
	int result = SDL_SaveBMP((SDL_Surface*)this->implementation, buffer);
	if (result == -1) throw new SDLException();
}

jint
Surface::blitSurface(Surface* src, 
	jint sx, jint sy, jint sw, jint sh, 
	jint dx, jint dy)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* dstImplementation = (SDL_Surface*)this->implementation;

	if (src->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* srcImplementation = (SDL_Surface*)src->implementation;

	SDL_Rect srect;
	SDL_Rect drect;

	srect.x = sx;
	srect.y = sy;
	srect.w = sw;
	srect.h = sh;
	drect.x = dx;
	drect.y = dy;
	int result = SDL_BlitSurface(srcImplementation, &srect, dstImplementation, &drect);
	return result;
}

jint
Surface::blitSurface(Surface* src, 
	jint dx, jint dy)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* dstImplementation = (SDL_Surface*)this->implementation;

	if (src->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* srcImplementation = (SDL_Surface*)src->implementation;

	SDL_Rect drect;

	drect.x = dx;
	drect.y = dy;
	int result = SDL_BlitSurface(srcImplementation, NULL, dstImplementation, &drect);
	return result;
}

jint
Surface::blitSurface(Surface* src)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* dstImplementation = (SDL_Surface*)this->implementation;

	if (src->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* srcImplementation = (SDL_Surface*)src->implementation;

	int result = SDL_BlitSurface(srcImplementation, NULL, dstImplementation, NULL);
	return result;
}

void
Surface::fillRect(
	jint x, jint y, jint w, jint h, jint color)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
	SDL_Rect rect;

	rect.x = x;
	rect.y = y;
	rect.w = w;
	rect.h = h;
	int result = SDL_FillRect(nativeSurface, &rect, (Uint32)color);
	if (result == -1) throw new SDLException();
}

void
Surface::fillRect(jint color)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	int result = SDL_FillRect(nativeSurface, NULL, (Uint32)color);
	if (result == -1) throw new SDLException();
}

void
Surface::setColorKey (jint flags, jint colorkey)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	int result = SDL_SetColorKey(nativeSurface, (Uint32)flags, (Uint32)colorkey);
	if (result == -1) throw new SDLException();

}

void 
Surface::setColorKey (jint flags)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	int result = SDL_SetColorKey(nativeSurface, flags, *(Uint8*)nativeSurface->pixels);
	if (result == -1) throw new SDLException();
}

jboolean 
Surface::setClipRect (jint x, jint y, jint w, jint h)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
	int result = SDL_FALSE;
	SDL_Rect rect;

	rect.x = x;
	rect.y = y;
	rect.w = w;
	rect.h = h;
	result = SDL_SetClipRect(nativeSurface, &rect);

	return (result == SDL_TRUE);
}

jboolean 
Surface::setClipRect ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
	SDL_bool result = SDL_FALSE;

	result = SDL_SetClipRect(nativeSurface, NULL);

	return (result == SDL_TRUE);
}

jintArray 
Surface::getClipRect ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	jintArray clipRectArray = JvNewIntArray(4);
	jint* xywh = elements(clipRectArray);
	SDL_Rect rect;

	SDL_GetClipRect(nativeSurface, &rect);
	xywh[0] = rect.x;
	xywh[1] = rect.y;
	xywh[2] = rect.w;
	xywh[3] = rect.h;

	return clipRectArray;
}

jint 
Surface::setAlpha (jint flags, jint alpha)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
	int result = 0;

	result = SDL_SetAlpha(nativeSurface, (Uint32)flags, (Uint8)alpha);

	return result;
}


void
Surface::displayFormat ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	SDL_Surface* newNativeSurface = SDL_DisplayFormat(nativeSurface);
	if (newNativeSurface == NULL) throw new SDLException();

	SDL_FreeSurface(nativeSurface);
	this->implementation = (::gnu::gcj::RawData*)newNativeSurface;
	this->format = NULL;
}

void
Surface::displayForamtAlpha ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	SDL_Surface* newNativeSurface = SDL_DisplayFormatAlpha(nativeSurface);
	if (newNativeSurface == NULL) throw new SDLException();

	SDL_FreeSurface(nativeSurface);
	this->implementation = (::gnu::gcj::RawData*)newNativeSurface;
	this->format = NULL;
}

void
Surface::convertSurface (PixelFormat* format, jint flags)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	if (format == NULL) throw createFreedSurfaceException();
	if (format->implementation == NULL) throw createFreedSurfaceException();
	SDL_PixelFormat* nativeFormat = (SDL_PixelFormat*)format->implementation;

	SDL_Surface* newNativeSurface = SDL_ConvertSurface(nativeSurface, nativeFormat, flags);
	if (newNativeSurface == NULL) throw new SDLException();

	SDL_FreeSurface(nativeSurface);
	this->implementation = (::gnu::gcj::RawData*)newNativeSurface;
	this->format = NULL;
}

Surface*
Surface::cloneSurface ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
	SDL_PixelFormat* nativeFormat = nativeSurface->format;

	SDL_Surface* newNativeSurface = SDL_ConvertSurface(nativeSurface, nativeFormat, nativeSurface->flags);
	if (newNativeSurface == NULL) throw new SDLException();

	Surface* result = new Surface();
	result->implementation = (::gnu::gcj::RawData*)newNativeSurface;

	return result;
}

PixelFormat*
Surface::getPixelFormat ()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	if (this->format == NULL)
	{
		SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;
		this->format = new PixelFormat();
		this->format->implementation = (::gnu::gcj::RawData*)nativeSurface->format;
	}
	return this->format;
}

void 
Surface::setPalette (JArray< Color *> *palette)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	if (nativeSurface->format->palette != NULL)
	{
		Color** colors = elements(palette);
		int ncolors = JvGetArrayLength(palette);
		ncolors = ncolors <= 256 ? ncolors : 256;
		SDL_Color nativeColors[ncolors];
		for (int i = 0;i < ncolors;i++)
		{
			nativeColors[i].r = colors[i]->r;
			nativeColors[i].g = colors[i]->g;
			nativeColors[i].b = colors[i]->b;
		}
		SDL_SetColors(nativeSurface, nativeColors, 0, ncolors);
	}
	return ;
}

void 
Surface::setPalette (Palette *palette)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	if (nativeSurface->format->palette != NULL)
	{
		SDL_Color* nativeColors = (SDL_Color*)palette->colors;
		int ncolors = palette->getSize();
		ncolors = ncolors <= 256 ? ncolors : 256;
		SDL_SetColors(nativeSurface, nativeColors, 0, ncolors);
	}
	return ;
}

JArray<jbyte>*
Surface::getPixels()
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	int size = nativeSurface->h * nativeSurface->pitch;
	if (this->pixelArray == NULL)
	{
		this->pixelArray = JvNewByteArray(size);
	}
	jbyte* pixels = elements(this->pixelArray);
	memcpy(pixels, nativeSurface->pixels, size);

	return this->pixelArray;
}

void 
Surface::setPixels(JArray<jbyte>* pixelArray)
{
	if (this->implementation == NULL) throw createFreedSurfaceException();
	SDL_Surface* nativeSurface = (SDL_Surface*)this->implementation;

	if ( SDL_MUSTLOCK(nativeSurface) ) 
	{
		if ( SDL_LockSurface(nativeSurface) < 0 ) 
		{
			throw new SDLException();
		}
	}

	int nativeSize = nativeSurface->h * nativeSurface->pitch;
	int size = JvGetArrayLength(pixelArray);
	if (nativeSize < size) size = nativeSize;

	jbyte* pixels = elements(pixelArray);
	memcpy(nativeSurface->pixels, pixels, size);

	if ( SDL_MUSTLOCK(nativeSurface) ) 
	{
		SDL_UnlockSurface(nativeSurface);
	}

	return ;
}

void
Surface::freeSurface()
{
	if (this->implementation != NULL) 
	{
		SDL_FreeSurface((SDL_Surface*)this->implementation);
		this->implementation = NULL;
	}

	return ;
}

void
Surface::initWithRGBSurface( jint flags, 
		jint width, jint height, jint bpp,
		jint rmask, jint gmask, jint bmask, jint amask)
{
	SDL_Surface* nativeSurface = NULL;

	nativeSurface = SDL_CreateRGBSurface(flags, width, height, bpp,
		rmask, gmask, bmask, amask);
	if (nativeSurface != NULL)
	{
		this->implementation = (::gnu::gcj::RawData*)nativeSurface;
	}

	return ;
}
