/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/


#include<stdio.h>
#include<stdlib.h>
#include<string.h>

#include"SDL.h"
#include"SDL_rwops.h"
#include"SDL_endian.h"

#include"zlib.h"
#include"SDL_archive.h"


#define	GZ_BUF	1024
#define	UN_BUF	65536


enum
{
	METHOD_STORED = 0, /* no compression */
	METHOD_DEFLATED = 8 /* zlib */
};


typedef struct __LocalFileHeaderBase__
{
	unsigned char signature[4];
	Uint16 versionNeededToExtract;
	Uint16 generalPurposeBitFlags;
	Uint16 compressionMethod;
	Uint16 lastModFileTime;
	Uint16 lastModFileDate;
	Uint32 crc32;
	Uint32 compressedSize;
	Uint32 uncompressedSize;
	Uint16 fileNameLength;
	Uint16 extraFieldLength;
} LocalFileHeaderBase;


typedef struct __CentralDirHeaderBase__
{
	unsigned char signature[4];
	Uint16 versionMadeBy;
	Uint16 versionNeededToExtract;
	Uint16 generalPurposeBitFlags;
	Uint16 compressionMethod;
	Uint16 lastModFileTime;
	Uint16 lastModFileDate;
	Uint32 crc32;
	Uint32 compressedSize;
	Uint32 uncompressedSize;
	Uint16 fileNameLength;
	Uint16 extraFieldLength;
	Uint16 fileCommentLength;
	Uint16 diskNumberStart;
	Uint16 internalFileAttributes;
	Uint32 externalFileAttributes;
	Uint32 relativeOffsetOfLocalHeader;
} CentralDirHeaderBase;


typedef struct __EndOfCentralDirRecordBase__
{
	unsigned char signature[4];
	Uint16 numberOfDisk;
	Uint16 numberOfDiskWithStartOfCentralDir;
	Uint16 totalNumOfEntriesOnThisDisk;
	Uint16 totalNumOfEntries;
	Uint32 sizeOfCentralDir;
	Uint32 offsetOfStartOfCentralDir;
	Uint16 commentLength;
} EndOfCentralDirRecordBase;


enum
{
	HEADER_SUCCESS,
	HEADER_BROKEN,
	HEADER_NO_MAGIC
};


enum
{
	FILE_GZIP,
	FILE_UNCOMPRESS
};


typedef struct inner_file
{
	long seek;
//	long originalSize;
//	long compressedSize;
	char *fileName;
} INNER_FILE;

typedef struct pkzip_data
{
	SDL_RWops *rw;
	
	int openNumber;
	
	int isStreamFinish;
	long fileSeek;
	long originalSize;
	
	int memSeek;
	int flush;
	char compressData[GZ_BUF];
	char uncompressData[UN_BUF];
	long restDataSize;
	z_stream z;
	
	int fileNum;
	int allocatedChain;
	INNER_FILE *chain;
} PKZIP_DATA;


static PKZIP_DATA *PKZip_Create(const char *file);

static int PKZip_Open(SDL_Archive *archive, const char *file);
static int PKZip_NumOpen(SDL_Archive *archive, const int num);
static int PKZip_NameToIndex(SDL_Archive *archive, const char *name);
static int PKZip_FileNumber(SDL_Archive *archive);
static void PKZip_Finish(SDL_Archive *archive);
static int PKZip_LoadNextHeader(PKZIP_DATA *pkzip);
static int PKZip_GetEndOfCentralDirectory(
	PKZIP_DATA *pkzip,
	EndOfCentralDirRecordBase *endOfCentralDir);

static void Stored_SetMethod(SDL_Archive *archive);
static long Stored_Size(SDL_Archive *archive);
static long Stored_Tell(SDL_Archive *archive);
static int Stored_GetChar(SDL_Archive *archive);
static int Stored_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Stored_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Stored_EOF(SDL_Archive *archive);
static int Stored_Close(SDL_Archive *archive);

static void Deflate_SetMethod(SDL_Archive *archive);
static long Deflate_Size(SDL_Archive *archive);
static long Deflate_Tell(SDL_Archive *archive);
static int Deflate_GetChar(SDL_Archive *archive);
static int Deflate_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Deflate_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Deflate_EOF(SDL_Archive *archive);
static int Deflate_Close(SDL_Archive *archive);

static int Deflate_GetCharWithoutSeek(SDL_Archive *archive);
static void Deflate_InCheck(PKZIP_DATA *gzt);
static void Deflate_OutCheck(PKZIP_DATA *gzt);
static int Deflate_IsEOF(PKZIP_DATA *gzt);


int Archive_IsPKZip(const char *file)
{
	int ret;
	SDL_RWops *rw;
	unsigned char compareSignature[4];
	const unsigned char pkSignature[4] = {'P', 'K', 0x03, 0x04};
	
	rw = SDL_RWFromFile(file, "rb");
	if (rw == NULL) {
		return 0;
	}
	
	ret = SDL_RWread(rw, compareSignature, 1, 4);
	if (ret != 4 || memcmp(compareSignature, pkSignature, 4) != 0) {
		SDL_RWclose(rw);
		return 0;
	}
	SDL_RWclose(rw);
	
	return 1;
}


SDL_Archive *Archive_FromPKZip(const char *file)
{
	SDL_Archive *archive;
	PKZIP_DATA *pkzip;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	pkzip = PKZip_Create(file);
	if (pkzip == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	pkzip->openNumber = -1;
	archive->data = pkzip;
	
	archive->open = PKZip_Open;
	archive->numopen = PKZip_NumOpen;
	archive->name2index = PKZip_NameToIndex;
	archive->filenum = PKZip_FileNumber;
	archive->finish = PKZip_Finish;
	
	archive->get_char = NULL;
	archive->read = NULL;
	archive->seek = NULL;
	archive->size = NULL;
	archive->tell = NULL;
	archive->eof = NULL;
	archive->close = NULL;
	
	return archive;
}


void Deflate_SetMethod(SDL_Archive *archive)
{
	archive->get_char = Deflate_GetChar;
	archive->read = Deflate_Read;
	archive->seek = Deflate_Seek;
	archive->size = Deflate_Size;
	archive->tell = Deflate_Tell;
	archive->eof = Deflate_EOF;
	archive->close = Deflate_Close;
}


int PKZip_GetEndOfCentralDirectory(
	PKZIP_DATA *pkzip,
	EndOfCentralDirRecordBase *endOfCentralDir)
{
	const unsigned char pkSignature[4] = {'P', 'K', 0x05, 0x06};
	unsigned char compareSignature[4];
	int count;
	
	SDL_RWseek(
		pkzip->rw,
		-1 * (int)sizeof(EndOfCentralDirRecordBase),
		SEEK_END);
	SDL_RWread(pkzip->rw, compareSignature, 4, 1);
	SDL_RWseek(pkzip->rw, -5, SEEK_END);
	count = 0;
	while (memcmp(compareSignature, pkSignature, 4) != 0) {
		if (SDL_RWtell(pkzip->rw) <= 0) {
			return -1;
		}
		compareSignature[3] = compareSignature[2];
		compareSignature[2] = compareSignature[1];
		compareSignature[1] = compareSignature[0];
		SDL_RWread(pkzip->rw, &compareSignature[0], 1, 1);
		SDL_RWseek(pkzip->rw, -2, SEEK_CUR);
		count++;
		if (count > 8192) {
			return -1;
		}
	}
	if (SDL_RWtell(pkzip->rw) <= 0) {
		return -1;
	}
	
	SDL_RWseek(pkzip->rw, 5, SEEK_CUR);
	memmove(endOfCentralDir->signature, compareSignature, 4);
	endOfCentralDir->numberOfDisk = SDL_ReadLE16(pkzip->rw);
	endOfCentralDir->numberOfDiskWithStartOfCentralDir = SDL_ReadLE16(pkzip->rw);
	endOfCentralDir->totalNumOfEntriesOnThisDisk = SDL_ReadLE16(pkzip->rw);
	endOfCentralDir->totalNumOfEntries = SDL_ReadLE16(pkzip->rw);
	endOfCentralDir->sizeOfCentralDir = SDL_ReadLE32(pkzip->rw);
	endOfCentralDir->offsetOfStartOfCentralDir = SDL_ReadLE32(pkzip->rw);
	endOfCentralDir->commentLength = SDL_ReadLE16(pkzip->rw);
	
	return 0;
}


int PKZip_LoadNextHeader(PKZIP_DATA *pkzip)
{
	CentralDirHeaderBase centralDir;
	unsigned char pkSignature[4] = {'P', 'K', 0x01, 0x02};
	unsigned char compareSignature[4];
	char *fileName;
	
	SDL_RWread(pkzip->rw, compareSignature, 4, 1);
	if (memcmp(compareSignature, pkSignature, 4) != 0) {
		return -1;
	}
/*
	if (SDL_RWtell(pkzip->rw) <= 0) {
		return -1;
	}
*/
	
	memmove(centralDir.signature, compareSignature, 4);
	centralDir.versionMadeBy = SDL_ReadLE16(pkzip->rw);
	centralDir.versionNeededToExtract = SDL_ReadLE16(pkzip->rw);
	centralDir.generalPurposeBitFlags = SDL_ReadLE16(pkzip->rw);
	centralDir.compressionMethod = SDL_ReadLE16(pkzip->rw);
	centralDir.lastModFileTime = SDL_ReadLE16(pkzip->rw);
	centralDir.lastModFileDate = SDL_ReadLE16(pkzip->rw);
	centralDir.crc32 = SDL_ReadLE32(pkzip->rw);
	centralDir.compressedSize = SDL_ReadLE32(pkzip->rw);
	centralDir.uncompressedSize = SDL_ReadLE32(pkzip->rw);
	centralDir.fileNameLength = SDL_ReadLE16(pkzip->rw);
	centralDir.extraFieldLength = SDL_ReadLE16(pkzip->rw);
	centralDir.fileCommentLength = SDL_ReadLE16(pkzip->rw);
	centralDir.diskNumberStart = SDL_ReadLE16(pkzip->rw);
	centralDir.internalFileAttributes = SDL_ReadLE16(pkzip->rw);
	centralDir.externalFileAttributes = SDL_ReadLE32(pkzip->rw);
	centralDir.relativeOffsetOfLocalHeader = SDL_ReadLE32(pkzip->rw);
	fileName = (char*)malloc(centralDir.fileNameLength + 1);
	if (fileName == NULL) {
		return -1;
	}
	SDL_RWread(pkzip->rw, fileName, centralDir.fileNameLength, 1);
	fileName[centralDir.fileNameLength] = '\0';
	SDL_RWseek(
		pkzip->rw,
		centralDir.extraFieldLength + centralDir.fileCommentLength,
		SEEK_CUR);
	
	pkzip->chain[pkzip->fileNum].seek = centralDir.relativeOffsetOfLocalHeader;
//	pkzip->chain[pkzip->fileNum].originalSize = centralDir.uncompressedSize;
//	pkzip->chain[pkzip->fileNum].compressedSize = centralDir.compressedSize;
	pkzip->chain[pkzip->fileNum].fileName = fileName;
	
	pkzip->fileNum++;
	
	return 0;
}


PKZIP_DATA *PKZip_Create(const char *file)
{
	int ret;
	long fileSize;
	PKZIP_DATA *pkzip;
	EndOfCentralDirRecordBase endOfCentralDir;
	void *tempptr;
	
	pkzip = (PKZIP_DATA*)malloc(sizeof(PKZIP_DATA));
	if (pkzip == NULL) {
		return NULL;
	}
	
	pkzip->rw = SDL_RWFromFile(file, "rb");
	if (pkzip->rw == NULL) {
		free(pkzip);
		return NULL;
	}
	
	pkzip->fileNum = 0;
	
	ret = PKZip_GetEndOfCentralDirectory(
		pkzip,
		&endOfCentralDir);
	if (ret == -1) {
		free(pkzip);
		return NULL;
	}
	
	SDL_RWseek(pkzip->rw, 0, SEEK_END);
	fileSize = SDL_RWtell(pkzip->rw);;
	SDL_RWseek(
		pkzip->rw,
		endOfCentralDir.offsetOfStartOfCentralDir,
		SEEK_SET);
	
	pkzip->chain = malloc(sizeof(INNER_FILE));
	if (pkzip->chain == NULL) {
		SDL_RWclose(pkzip->rw);
		free(pkzip);
		return NULL;
	}
	pkzip->allocatedChain = 1;
	while (SDL_RWtell(pkzip->rw) < fileSize) {
		if (pkzip->fileNum + 1 > pkzip->allocatedChain) {
			pkzip->allocatedChain = pkzip->allocatedChain * 2;
			tempptr = realloc(pkzip->chain,
				sizeof(INNER_FILE) * pkzip->allocatedChain);
			if (tempptr == NULL) {
				break;
			}
			pkzip->chain = (INNER_FILE*)tempptr;
		}
		ret = PKZip_LoadNextHeader(pkzip);
		if (ret != 0) {
			break;
		}
	}
	
	return pkzip;
}


int PKZip_Open(SDL_Archive *archive, const char *file)
{
	return PKZip_NumOpen(archive, PKZip_NameToIndex(archive, file));
}


int PKZip_NumOpen(SDL_Archive *archive, const int num)
{
	int flag;
	int readSize;
	LocalFileHeaderBase localFileHeader;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	if (num < 0 || pkzip->fileNum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (pkzip->openNumber != -1) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	SDL_RWseek(pkzip->rw, pkzip->chain[num].seek, SEEK_SET);
	SDL_RWread(pkzip->rw, localFileHeader.signature, 4, 1);
	localFileHeader.versionNeededToExtract = SDL_ReadLE16(pkzip->rw);
	localFileHeader.generalPurposeBitFlags = SDL_ReadLE16(pkzip->rw);
	localFileHeader.compressionMethod = SDL_ReadLE16(pkzip->rw);
	localFileHeader.lastModFileTime = SDL_ReadLE16(pkzip->rw);
	localFileHeader.lastModFileDate = SDL_ReadLE16(pkzip->rw);
	localFileHeader.crc32 = SDL_ReadLE32(pkzip->rw);
	localFileHeader.compressedSize = SDL_ReadLE32(pkzip->rw);
	localFileHeader.uncompressedSize = SDL_ReadLE32(pkzip->rw);
	localFileHeader.fileNameLength = SDL_ReadLE16(pkzip->rw);
	localFileHeader.extraFieldLength = SDL_ReadLE16(pkzip->rw);
	SDL_RWseek(
		pkzip->rw,
		localFileHeader.fileNameLength + localFileHeader.extraFieldLength,
		SEEK_CUR);
	
	pkzip->restDataSize = localFileHeader.compressedSize;
	pkzip->originalSize = localFileHeader.uncompressedSize;
//	pkzip->restDataSize = pkzip->chain[num].compressedSize;
//	pkzip->originalSize = pkzip->chain[num].originalSize;
	
	switch (localFileHeader.compressionMethod) {
	case METHOD_STORED:
		Stored_SetMethod(archive);
		pkzip->fileSeek = 0L;
		pkzip->openNumber = num;
		return ARCHIVE_SUCCESS;
	case METHOD_DEFLATED:
		Deflate_SetMethod(archive);
		break;
	default:
		return ARCHIVE_ERROR_BROKEN_DATA;
	}
	
	readSize = GZ_BUF;
	pkzip->flush = Z_NO_FLUSH;
	if (pkzip->restDataSize < GZ_BUF) {
		readSize = (int)(pkzip->restDataSize);
		pkzip->flush = Z_FINISH;
	}
	readSize = SDL_RWread(pkzip->rw, pkzip->compressData, 1, readSize);
	pkzip->restDataSize -= readSize;
	
	pkzip->z.opaque = Z_NULL;
	
	pkzip->z.next_in = (char*)pkzip->compressData;
	pkzip->z.avail_in = readSize;
	pkzip->z.next_out = (char*)pkzip->uncompressData;
	pkzip->z.avail_out = UN_BUF;
	
	flag = inflateInit2(&pkzip->z, -MAX_WBITS);
	if (flag != Z_OK) {
		SDL_RWclose(pkzip->rw);
		return ARCHIVE_ERROR_BROKEN_DATA;
	}
	
	pkzip->fileSeek = 0L;
	pkzip->memSeek = 0L;
	pkzip->isStreamFinish = 0;
	
	flag = inflate(&pkzip->z, pkzip->flush);
	if (flag == Z_OK) {
		pkzip->openNumber = num;
		return ARCHIVE_SUCCESS;
	}
	if (flag == Z_STREAM_END) {
		pkzip->openNumber = num;
		pkzip->isStreamFinish = 1;
		return ARCHIVE_SUCCESS;
	}
	
	return ARCHIVE_ERROR_UNKNOWN;
}


int PKZip_NameToIndex(SDL_Archive *archive, const char *name)
{
	int i;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	for (i = 0; i < pkzip->fileNum; i++) {
		if (strcmp(name, pkzip->chain[i].fileName) == 0) {
			return i;
		}
	}
	
	return -1;
}


int PKZip_FileNumber(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	return pkzip->fileNum;
}


void PKZip_Finish(SDL_Archive *archive)
{
	int i;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	if (pkzip->openNumber != -1) {
		Archive_Close(archive);
	}
	
	if (pkzip != NULL) {
		SDL_RWclose(pkzip->rw);
		for (i = 0; i < pkzip->fileNum; i++) {
			free(pkzip->chain[i].fileName);
		}
		free(pkzip->chain);
		free(pkzip);
	}
	
	Archive_FreeMainContext(archive);
}





long Deflate_Size(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	return pkzip->originalSize;
}


long Deflate_Tell(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	return pkzip->fileSeek;
}


static void Deflate_InCheck(PKZIP_DATA *pkzip)
{
	int readSize;
	
	if (pkzip->restDataSize > 0 && pkzip->z.avail_in == 0) {
		readSize = GZ_BUF;
		if (pkzip->restDataSize < GZ_BUF) {
			pkzip->flush = Z_FINISH;
			readSize = (int)pkzip->restDataSize;
		}
		SDL_RWread(pkzip->rw, pkzip->compressData, readSize, 1);
		pkzip->restDataSize -= readSize;
		pkzip->z.next_in = pkzip->compressData;
		pkzip->z.avail_in = readSize;
	}
}


static void Deflate_OutCheck(PKZIP_DATA *pkzip)
{
	int status;
	
	if (pkzip->isStreamFinish == 0 && pkzip->memSeek >= UN_BUF - (int)pkzip->z.avail_out) {
		pkzip->z.next_out = pkzip->uncompressData;
		pkzip->z.avail_out = UN_BUF;
		pkzip->memSeek = 0;
		status = inflate(&pkzip->z, pkzip->flush);
		if (status == Z_STREAM_END) {
			pkzip->isStreamFinish = 1;
		}
	}
}


int Deflate_GetCharWithoutSeek(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	Deflate_InCheck(pkzip);
	Deflate_OutCheck(pkzip);
	if (Deflate_IsEOF(pkzip)) {
		return EOF;
	}
	
	return pkzip->uncompressData[pkzip->memSeek++];
}


int Deflate_GetChar(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	Deflate_InCheck(pkzip);
	Deflate_OutCheck(pkzip);
	if (Deflate_IsEOF(pkzip)) {
		return EOF;
	}
	pkzip->fileSeek++;
	
	return pkzip->uncompressData[pkzip->memSeek++];
}


int Deflate_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	int num;
	int check;
	long position;
	long i;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	switch (whence) {
	case SEEK_SET:
		position = offset;
		break;
	case SEEK_CUR:
		position = pkzip->fileSeek + offset;
		break;
	case SEEK_END:
		position = pkzip->originalSize + offset;
		break;
	default:
		return ARCHIVE_ERROR_WHENCE;
	}
	
	if (position > pkzip->originalSize) {
		position = pkzip->originalSize;
	}
	if (position < 0) {
		position = 0;
	}
	
	if (position > pkzip->fileSeek) {
		for (i = pkzip->fileSeek; i < position; i++) {
			Deflate_GetChar(archive);
		}
	} else if (position < pkzip->fileSeek) {
		num = pkzip->openNumber;
		Deflate_Close(archive);
		check = PKZip_NumOpen(archive, num);
		if (check != ARCHIVE_SUCCESS) {
			return check;
		}
		for (i = 0; i < position; i++) {
			Deflate_GetChar(archive);
		}
	}
	
	return ARCHIVE_SUCCESS;
}


int Deflate_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	int i;
	int count;
	long position;
	
	char *memc;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	memc = (char*)mem;
	
	position = 0L;
	count = 0;
	while (count < maxnum && pkzip->fileSeek + position < pkzip->originalSize) {
		for (i = 0; i < size; i++) {
			memc[count * size + i] = Deflate_GetCharWithoutSeek(archive);
		}
		position += size;
		count++;
	}
	
	pkzip->fileSeek += size *count;
	
	return count;
}


int Deflate_IsEOF(PKZIP_DATA *pkzip)
{
	if (pkzip->isStreamFinish == 1 && pkzip->memSeek >= UN_BUF - (int)pkzip->z.avail_out) {
		return 1;
	} else {
		if (pkzip->memSeek > UN_BUF) {
			return 1;
		} else {
			return 0;
		}
	}
}


int Deflate_EOF(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	Deflate_InCheck(pkzip);
	Deflate_OutCheck(pkzip);
	
	return Deflate_IsEOF(pkzip);
}


int Deflate_Close(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	if (pkzip->openNumber != -1) {
		inflateEnd(&pkzip->z);
		pkzip->openNumber = -1;
	}
	return ARCHIVE_SUCCESS;
}





void Stored_SetMethod(SDL_Archive *archive)
{
	archive->get_char = Stored_GetChar;
	archive->read = Stored_Read;
	archive->seek = Stored_Seek;
	archive->size = Stored_Size;
	archive->tell = Stored_Tell;
	archive->eof = Stored_EOF;
	archive->close = Stored_Close;
}


long Stored_Size(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	return pkzip->originalSize;
}


long Stored_Tell(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	return pkzip->fileSeek;
}


int Stored_GetChar(SDL_Archive *archive)
{
	char c;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	if (pkzip->fileSeek >= pkzip->originalSize) {
		return EOF;
	} else {
		SDL_RWread(pkzip->rw, &c, 1, 1);
		pkzip->fileSeek++;
		return c;
	}
}


int Stored_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	long position;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	switch (whence) {
	case SEEK_SET:
		position = offset;
		break;
	case SEEK_CUR:
		position = pkzip->fileSeek + offset;
		break;
	case SEEK_END:
		position = pkzip->originalSize + offset;
		break;
	default:
		return ARCHIVE_ERROR_WHENCE;
	}
	
	if (position > pkzip->originalSize) {
		position = pkzip->originalSize;
	}
	if (position < 0) {
		position = 0;
	}
	
	SDL_RWseek(pkzip->rw, position - pkzip->fileSeek, SEEK_CUR);
	pkzip->fileSeek = position;
	
	return ARCHIVE_SUCCESS;
}


int Stored_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	int i;
	int ret;
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	for (i = 0; i < maxnum; i++) {
		ret = SDL_RWread(pkzip->rw, (char*)mem + size * i, size, 1);
		if (ret == 0) {
			break;
		}
	}
	pkzip->fileSeek += size * i;
	
	return i;
}


int Stored_EOF(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	
	if (pkzip->fileSeek >= pkzip->originalSize) {
		return 1;
	} else {
		return 0;
	}
}


int Stored_Close(SDL_Archive *archive)
{
	PKZIP_DATA *pkzip;
	
	pkzip = (PKZIP_DATA*)archive->data;
	pkzip->openNumber = -1;
	
	return ARCHIVE_SUCCESS;
}
