/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#ifndef _SDL_archive_H_
#define _SDL_archive_H_


#include <stdio.h>
#include "SDL_version.h"
#include "SDL_rwops.h"

#include "begin_code.h"



#define ARCHIVE_MAJOR_VERSION	0
#define ARCHIVE_MINOR_VERSION	3
#define ARCHIVE_PATCHLEVEL		1

/* This macro can be used to fill a version structure with the compile-time
 * version of the SDL_mixer library.
 */
#define ARCHIVE_VERSION(X)							\
{									\
	(X)->major = ARCHIVE_MAJOR_VERSION;					\
	(X)->minor = ARCHIVE_MINOR_VERSION;					\
	(X)->patch = ARCHIVE_PATCHLEVEL;					\
}


/* Set up for C function definitions, even when using C++ */
#ifdef __cplusplus
extern "C" {
#endif


typedef struct _SDL_Archive
{
	/* public section */
		/* ! must set up before open ! */
	int (*getChildDirNum)(struct _SDL_Archive *data);
	char *(*getChildDirName)(struct _SDL_Archive *data, int dirNum);
	int (*goDownDir)(struct _SDL_Archive *data, const char *dirName);
	int (*goUpDir)(struct _SDL_Archive *data);
	int (*moveDirByPath)(struct _SDL_Archive *data, const char *path);
	char *(*getCurrentDirName)(struct _SDL_Archive *data);
	char *(*getCurrentDirPath)(struct _SDL_Archive *data);
	
	int (*getFileNum)(struct _SDL_Archive *data);
	char *(*getFileName)(struct _SDL_Archive *data, int fileNum);
	int (*getFileNumByName)(struct _SDL_Archive *data, const char *fileName);
	int (*openFile)(struct _SDL_Archive *data, const char *fileName);
	
	void (*finish)(struct _SDL_Archive *data);
	
	struct _SDL_Archive *(*clone)(struct _SDL_Archive *data);
	
		/* ! not have to set up before open ! */
	int (*getChar)(struct _SDL_Archive *data);
	int (*read)(struct _SDL_Archive *data, void *mem, const int size, const int maxnum);
	int (*seek)(struct _SDL_Archive *data, const long offset, const int whence);
	long (*size)(struct _SDL_Archive *data);
	long (*tell)(struct _SDL_Archive *data);
	int (*eof)(struct _SDL_Archive *data);
	int (*close)(struct _SDL_Archive *data);
	
	/* protected section */
	void *data;
} SDL_Archive;

typedef struct _ArchiverList ArchiverList;


enum
{
	ARCHIVE_SUCCESS = 0,
	ARCHIVE_ERROR_WHENCE,
	ARCHIVE_ERROR_BROKEN_DATA,
	ARCHIVE_ERROR_UNKNOWN,
	ARCHIVE_ERROR_NO_INIT,
	ARCHIVE_ERROR_NO_EXIST_FILE,
	ARCHIVE_ERROR_NO_EXIST_DIRECTORY,
	ARCHIVE_ERROR_INVALID_NAME,
	ARCHIVE_ERROR_ALLOCATE,
	ARCHIVE_ERROR_ALREADY_ROOT,
	ARCHIVE_ERROR_ALREADY_OPEN
};


extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_Alloc(void);
extern DECLSPEC void SDLCALL Archive_FreeMainContext(struct _SDL_Archive *archive);

#define Archive_GetChildDirNum(ctx)	(ctx)->getChildDirNum(ctx)
#define Archive_GetChildDirName(ctx, dirNum)	(ctx)->getChildDirName(ctx, dirNum)
#define Archive_GoDownDir(ctx, dirName)	(ctx)->goDownDir(ctx, dirName)
#define Archive_GoUpDir(ctx)	(ctx)->goUpDir(ctx)
#define Archive_GetCurrentDirName(ctx)	(ctx)->getCurrentDirName(ctx)
#define Archive_GetCurrentDirPath(ctx)	(ctx)->getCurrentDirPath(ctx)

#define Archive_MoveDirByPath(ctx, path)	(ctx)->moveDirByPath(ctx, path)

#define Archive_GetFileNum(ctx)	(ctx)->getFileNum(ctx)
#define Archive_GetFileName(ctx, fileNum)	(ctx)->getFileName(ctx, fileNum)
#define Archive_GetFileNumByName(ctx, fileName)	(ctx)->getFileNumByName(ctx, fileName)

#define Archive_Open(ctx, fileName)	(ctx)->openFile(ctx, fileName)
#define Archive_GetChar(ctx)	(ctx)->getChar(ctx)
#define Archive_Size(ctx)	(ctx)->size(ctx)
#define Archive_Tell(ctx)	(ctx)->tell(ctx)
#define Archive_Seek(ctx, offset, whence)	(ctx)->seek(ctx, offset, whence)
#define Archive_Read(ctx, mem, size, maxnum)	(ctx)->read(ctx, mem, size, maxnum)
#define Archive_EOF(ctx)	(ctx)->eof(ctx)
#define Archive_Close(ctx)	(ctx)->close(ctx)
#define Archive_Finish(ctx)	(ctx)->finish(ctx)

#define Archive_Clone(ctx)	(ctx)->clone(ctx)

/* backward compatibility */
#define Archive_FileNum(ctx)	(ctx)->getFileNum(ctx)
#define Archive_NameToIndex(ctx, fileName)	(ctx)->getFileNumByName(ctx, fileName)
#define Archive_NumOpen(ctx, fileNum)	(ctx)->openFile(ctx, (ctx)->getFileName(ctx, fileNum))



extern DECLSPEC int SDLCALL Archive_IsPKZip(const char *file);
extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_FromPKZip(const char *file);

extern DECLSPEC int SDLCALL Archive_IsTar(const char *file);
extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_FromTar(const char *file);

extern DECLSPEC int SDLCALL Archive_IsDirectory(const char *file);
extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_FromDirectory(const char *file);


typedef int (*Archive_IsOpenable)(const char *file);
typedef struct _SDL_Archive* (*Archive_FromFormat)(const char *file);

extern DECLSPEC ArchiverList * SDLCALL ArchiverList_Create(void);

extern DECLSPEC int SDLCALL ArchiverList_AddArchiver(
	ArchiverList *archiverList,
	Archive_IsOpenable isOpenable,
	Archive_FromFormat fromFormat);
extern DECLSPEC void SDLCALL ArchiverList_Free(ArchiverList *archiverList);

extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_FromArchiverList(
	ArchiverList *archiverList,
	const char *file);


extern DECLSPEC struct _SDL_Archive * SDLCALL Archive_FromGZFilter(
	ArchiverList *archiverList,
	const char *file);

extern DECLSPEC SDL_RWops * SDLCALL SDL_RWFromArchive(struct _SDL_Archive *arc, const char *file);
extern DECLSPEC SDL_RWops * SDLCALL SDL_RWFromArchiveWithClone(struct _SDL_Archive *arc, const char *file);





/* for directory container */

typedef struct _DirContainer DirContainer;

extern DECLSPEC DirContainer * SDLCALL DirContainer_Create(void);
extern DECLSPEC void SDLCALL DirContainer_Free(DirContainer *dirList);
extern DECLSPEC DirContainer * SDLCALL DirContainer_Clone(DirContainer *dirList);

extern DECLSPEC int SDLCALL DirContainer_AddDirectory(DirContainer *dirList, const char *name);
extern DECLSPEC int DirContainer_AddDirectoryToDirIndex(
	DirContainer *dirList, const char *name, int index);
extern DECLSPEC int SDLCALL DirContainer_AddFile(DirContainer *dirList, int fileIndex);
extern DECLSPEC int SDLCALL DirContainer_AddFileToDirIndex(
	DirContainer *dirList, int fileIndex, int dirIndex);

typedef enum {
	DIRECTORY_NON = 0x0000,
	DIRECTORY_ADD_DIR = 0x0001,
	DIRECTORY_MOVE_DIR = 0x0002
} DirPathParseFlag;
/* path delimiter is '/' */
extern DECLSPEC int SDLCALL DirContainer_ParsePath(
	DirContainer *dirList, const char *path,
	int flags, int *returnDirIndex);

extern DECLSPEC int SDLCALL DirContainer_GetChildDirNum(DirContainer *dirList);
extern DECLSPEC char * SDLCALL DirContainer_GetChildDirName(DirContainer *dirList, int num);
extern DECLSPEC int SDLCALL DirContainer_GetFileNum(DirContainer *dirList);
extern DECLSPEC int SDLCALL DirContainer_GetFileIndex(DirContainer *dirList, int num);
extern DECLSPEC char * SDLCALL DirContainer_GetCurrentDirName(DirContainer *dirList);
extern DECLSPEC char * SDLCALL DirContainer_GetCurrentPath(DirContainer *dirList);
extern DECLSPEC int SDLCALL DirContainer_GoUpDir(DirContainer *dirList);
extern DECLSPEC int SDLCALL DirContainer_GoDownDir(DirContainer *dirList, const char *dirName);



/* Ends C function definitions when using C++ */
#ifdef __cplusplus
}
#endif

#include "close_code.h"



#endif
