/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32
#  include <windows.h>
#else
#  include <dirent.h>
#endif

#include "SDL_archive.h"


typedef struct _file_chain
{
	long size;
	char *filename;
} FILE_CHAIN;


typedef struct _directory_DATA
{
	FILE *fp;
	int chainNum;
	FILE_CHAIN *chain;
	int chainAllocated;
	
	int tempStringAllocated;
	char *tempString;
	
	int openNumber;
	char *dirName;
	
	DirContainer *dirContainer;
} DIRECTORY_DATA;


static DIRECTORY_DATA *Dir_Create(const char *file);
static int Dir_GetFileNumByName(SDL_Archive *archive, const char *filename);
static int Dir_OpenFile(SDL_Archive *archive, const char *fileName);
static int Dir_GetChildDirNum(SDL_Archive *archive);
static char *Dir_GetChildDirName(SDL_Archive *archive, int dirNum);
static int Dir_GoDownDir(SDL_Archive *archive, const char *dirName);
static int Dir_GoUpDir(SDL_Archive *archive);
static int Dir_MoveDirByPath(SDL_Archive *archive, const char *path);
static char *Dir_GetCurrentDirName(SDL_Archive *archive);
static char *Dir_GetCurrentDirPath(SDL_Archive *archive);
static long Dir_Size(SDL_Archive *archive);
static int Dir_GetFileNumber(SDL_Archive *archive);
static char *Dir_GetFileName(SDL_Archive *archive, int num);
static void Dir_Finish(SDL_Archive *archive);
static int Dir_GetChar(SDL_Archive *archive);
static long Dir_Tell(SDL_Archive *archive);
static int Dir_EOF(SDL_Archive *archive);
static int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Dir_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Dir_Close(SDL_Archive *archive);
static int Dir_LoadDirectory(DIRECTORY_DATA *data);
static int Dir_SetTempPath(DIRECTORY_DATA *data, const char *fileName);
static SDL_Archive *Dir_Clone(SDL_Archive *archive);

static int Dir_ReallocChain(DIRECTORY_DATA *data);
static int Dir_AddFile(DIRECTORY_DATA *data, const char *fileName, int fileSize);
static int Dir_AddDirectory(DIRECTORY_DATA *data, const char *dirName);
static int Dir_IsValidDir(const char *name);



int Archive_IsDirectory(const char *file)
#ifdef WIN32
{
	char *temp;
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return SDL_FALSE;
	}
	
	temp = malloc(strlen(file) + strlen("\\*.*") + 1);
	if (temp == NULL) {
		return SDL_FALSE;
	}
	
	strcpy(temp, file);
	strcat(temp, "\\*.*");
	handle = FindFirstFile(temp, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		free(temp);
		return SDL_FALSE;
	}
	FindClose(handle);
	free(temp);
	
	return SDL_TRUE;
}
#else
{
	DIR *dir;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return SDL_FALSE;
	}
	
	dir = opendir(file);
	if (dir == NULL) {
		return SDL_FALSE;
	} else {
		closedir(dir);
		return SDL_TRUE;
	}
}
#endif


SDL_Archive *Archive_FromDirectory(const char *file)
{
	SDL_Archive *archive;
	
	if (Dir_IsValidDir(file) == SDL_FALSE) {
		return NULL;
	}
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	archive->data = Dir_Create(file);
	if (archive->data == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	
	archive->getChildDirNum = Dir_GetChildDirNum;
	archive->getChildDirName = Dir_GetChildDirName;
	archive->goDownDir = Dir_GoDownDir;
	archive->goUpDir = Dir_GoUpDir;
	archive->moveDirByPath = Dir_MoveDirByPath;
	archive->getCurrentDirName = Dir_GetCurrentDirName;
	archive->getCurrentDirPath = Dir_GetCurrentDirPath;
	
	archive->openFile = Dir_OpenFile;
	archive->getFileNum = Dir_GetFileNumber;
	archive->getFileName = Dir_GetFileName;
	archive->getFileNumByName = Dir_GetFileNumByName;
	archive->finish = Dir_Finish;
	
	archive->clone = Dir_Clone;
	
	archive->getChar = Dir_GetChar;
	archive->read = Dir_Read;
	archive->seek = Dir_Seek;
	archive->size = Dir_Size;
	archive->tell = Dir_Tell;
	archive->eof = Dir_EOF;
	archive->close = Dir_Close;
	
	return archive;
}



int Dir_IsValidDir(const char *name)
{
	int i;
	int length;
	
	if (strcmp(name, "..") == 0 || strcmp(name, ".") == 0) {
		return SDL_FALSE;
	}
	
	length = strlen(name);
	for (i = 0; i < length; i++) {
		if (name[i] == '/' || name[i] == '\\') {
			return SDL_FALSE;
		}
	}
	
	return SDL_TRUE;
}


int Dir_SetTempPath(DIRECTORY_DATA *data, const char *fileName)
{
	int allocSize;
	int needSize;
	char *tempMem;
	char *currentPath;
	int fileNameLength;
	
	currentPath = DirContainer_GetCurrentPath(data->dirContainer);
	currentPath = currentPath + 1; /* next '/' */
	
	if (fileName) {
		fileNameLength = strlen(fileName);
	} else {
		fileNameLength = 0;
	}
	/*
	  data->dirName + delimiter +
	  dirName + delimiter +
	  fileName + '\0'
	 */
	needSize =
		strlen(data->dirName) + 1 +
		strlen(currentPath) + 1 +
		fileNameLength + 1;
	allocSize = data->tempStringAllocated;
	while (allocSize < needSize) {
		allocSize *= 2;
	}
	
	if (allocSize > data->tempStringAllocated) {
		tempMem = realloc(data->tempString, allocSize);
		if (tempMem == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		data->tempStringAllocated = allocSize;
		data->tempString = tempMem;
	}
	
	strcpy(data->tempString, data->dirName);
	
	if (currentPath != NULL && currentPath[0] != '\0') {
		strcat(data->tempString, "/");
		strcat(data->tempString, currentPath);
	}
	
	if (fileName) {
		strcat(data->tempString, "/");
		strcat(data->tempString, fileName);
	}
	
#ifdef WIN32
	{
		int i;
		int stringLength;
		stringLength = strlen(data->tempString);
		for (i = 0; i < stringLength; i++) {
			if (data->tempString[i] == '/') {
				data->tempString[i] = '\\';
			}
		}
	}
#endif
	
	return ARCHIVE_SUCCESS;
}


int Dir_ReallocChain(DIRECTORY_DATA *data)
{
	int needSize;
	void *tempptr;
	
	if (data->chainNum >= data->chainAllocated) {
		needSize = data->chainAllocated * 2;
		tempptr = realloc(data->chain,
			sizeof(FILE_CHAIN) * needSize);
		if (tempptr == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		data->chain = (FILE_CHAIN*)tempptr;
		data->chainAllocated = needSize;
	}
	
	return ARCHIVE_SUCCESS;
}


int Dir_AddFile(DIRECTORY_DATA *data, const char *fileName, int fileSize)
{
	int result;
	
	result = Dir_ReallocChain(data);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	data->chain[data->chainNum].filename =
		malloc(strlen(fileName) + 1);
	if (data->chain[data->chainNum].filename == NULL) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	strcpy(data->chain[data->chainNum].filename, fileName);
	data->chain[data->chainNum].size = fileSize;
	DirContainer_AddFile(data->dirContainer, data->chainNum);
	data->chainNum++;
	
	return ARCHIVE_SUCCESS;
}


int Dir_AddDirectory(DIRECTORY_DATA *data, const char *dirName)
{
	int result;
	
	result = DirContainer_AddDirectory(
		data->dirContainer, dirName);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = DirContainer_GoDownDir(
		data->dirContainer, dirName);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = Dir_LoadDirectory(data);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	result = DirContainer_GoUpDir(data->dirContainer);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	return ARCHIVE_SUCCESS;
}


int Dir_LoadDirectory(DIRECTORY_DATA *data)
#ifdef WIN32
{
	WIN32_FIND_DATA findFile;
	HANDLE handle;
	int result;
	
	result = Dir_SetTempPath(data, "*");
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	handle = FindFirstFile(data->tempString, &findFile);
	if (handle == INVALID_HANDLE_VALUE) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	do {
		if (
			strcmp(findFile.cFileName, "..") == 0 ||
			strcmp(findFile.cFileName, ".") == 0
		) {
			continue;
		}
		if (findFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			result = Dir_AddDirectory(data, findFile.cFileName);
			if (result != ARCHIVE_SUCCESS) {
				return result;
			}
		} else {
			result = Dir_SetTempPath(data, findFile.cFileName);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
			
			result = Dir_AddFile(data, findFile.cFileName, findFile.nFileSizeLow);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
		}
	} while(FindNextFile(handle, &findFile));
	
	FindClose(handle);
	
	return ARCHIVE_SUCCESS;
}
#else
{
	DIR *dir;
	DIR *tempDir;
	FILE *fp;
	struct dirent *ent;
	long fileSize;
	int result;
	
	result = Dir_SetTempPath(data, NULL);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	dir = opendir(data->tempString);
	if (dir == NULL) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	while ((ent = readdir(dir)) != NULL) {
		if (
			strcmp(ent->d_name, "..") == 0 ||
			strcmp(ent->d_name, ".") == 0
		) {
			continue;
		}
		
		result = Dir_SetTempPath(data, ent->d_name);
		if (result != ARCHIVE_SUCCESS) {
			return result;
		}
		tempDir = opendir(data->tempString);
		if (tempDir) {
			closedir(tempDir);
			result = Dir_AddDirectory(data, ent->d_name);
			if (result != ARCHIVE_SUCCESS) {
				return result;
			}
		} else {
			result = Dir_SetTempPath(data, ent->d_name);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
			fp = fopen(data->tempString, "rb");
			if (fp == NULL) {
				continue;
			}
			fseek(fp, 0L, SEEK_END);
			fileSize = ftell(fp);
			fclose(fp);
			
			result = Dir_AddFile(data, ent->d_name, fileSize);
			if (result != ARCHIVE_SUCCESS) {
				break;
			}
		}
	}
	
	closedir(dir);
	
	return ARCHIVE_SUCCESS;
}
#endif


DIRECTORY_DATA *Dir_Create(const char *file)
{
	DIRECTORY_DATA *data;
	int needSize;
	int result;
	
	data = (DIRECTORY_DATA*)malloc(sizeof(DIRECTORY_DATA));
	if (data == NULL) {
		return NULL;
	}
	
	data->dirName = (char*)malloc(strlen(file) + 1);
	if (data->dirName == NULL) {
		free(data);
		return NULL;
	}
	strcpy(data->dirName, file);
	
	/* directory name + delimiter + '\0' */
	needSize = strlen(data->dirName) + 1 + 1;
	data->tempString = (char*)malloc(needSize);
	if (data->tempString == NULL) {
		free(data->dirName);
		free(data);
		return NULL;
	}
	data->tempStringAllocated = needSize;
	
	data->dirContainer = DirContainer_Create();
	if (data->dirContainer == NULL) {
		free(data->tempString);
		free(data->dirName);
		free(data);
		return NULL;
	}
	
	data->chain = (FILE_CHAIN*)malloc(sizeof(FILE_CHAIN) * 1);
	if (data->chain == NULL) {
		DirContainer_Free(data->dirContainer);
		free(data->tempString);
		free(data->dirName);
		free(data);
		return NULL;
	}
	data->fp = NULL;
	data->chainNum = 0;
	data->chainAllocated = 1;
	
	result = Dir_LoadDirectory(data);
	if (result != ARCHIVE_SUCCESS) {
		DirContainer_Free(data->dirContainer);
		free(data->chain);
		free(data->tempString);
		free(data->dirName);
		free(data);
		return NULL;
	}
	
	return data;
}


SDL_Archive *Dir_Clone(SDL_Archive *orgArchive)
{
	SDL_Archive *archive;
	DIRECTORY_DATA *dir;
	DIRECTORY_DATA *orgDir;
	int iDir, iOrgDir;
	int needSize;
	
	orgDir = (DIRECTORY_DATA*)orgArchive->data;
	
	dir = malloc(sizeof(DIRECTORY_DATA));
	if (dir == NULL) {
		return NULL;
	}
	
	dir->dirName = (char*)malloc(strlen(orgDir->dirName) + 1);
	if (dir->dirName == NULL) {
		free(dir);
		return NULL;
	}
	strcpy(dir->dirName, orgDir->dirName);
	
	/* directory name + delimiter + '\0' */
	needSize = strlen(dir->dirName) + 1 + 1;
	dir->tempString = (char*)malloc(needSize);
	if (dir->tempString == NULL) {
		free(dir->dirName);
		free(dir);
		return NULL;
	}
	dir->tempStringAllocated = needSize;
	
	dir->chain = (FILE_CHAIN*)malloc(
		sizeof(FILE_CHAIN) * orgDir->chainNum);
	if (dir->chain == NULL) {
		free(dir->tempString);
		free(dir->dirName);
		free(dir);
		return NULL;
	}
	dir->chainAllocated = orgDir->chainNum;
	
	dir->fp = NULL;
	
	iDir = 0;
	for (iOrgDir = 0; iOrgDir < orgDir->chainNum; iOrgDir++) {
		needSize = strlen(orgDir->chain[iOrgDir].filename) + 1;
		dir->chain[iDir].filename = (char*)malloc(needSize);
		if (dir->chain[iDir].filename == NULL) {
			continue;
		}
		strcpy(dir->chain[iDir].filename,
			orgDir->chain[iOrgDir].filename);
		
		dir->chain[iDir].size = orgDir->chain[iOrgDir].size;
		
		iDir++;
	}
	dir->chainNum = iDir;
	
	dir->dirContainer = DirContainer_Clone(orgDir->dirContainer);
	if (dir->dirContainer == NULL) {
		free(dir->chain);
		free(dir->tempString);
		free(dir->dirName);
		free(dir);
		return NULL;
	}
	
	dir->openNumber = -1;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		DirContainer_Free(dir->dirContainer);
		free(dir->chain);
		free(dir->tempString);
		free(dir->dirName);
		free(dir);
		return NULL;
	}
	
	archive->data = dir;
	
	archive->getChildDirNum = Dir_GetChildDirNum;
	archive->getChildDirName = Dir_GetChildDirName;
	archive->goDownDir = Dir_GoDownDir;
	archive->goUpDir = Dir_GoUpDir;
	archive->moveDirByPath = Dir_MoveDirByPath;
	archive->getCurrentDirName = Dir_GetCurrentDirName;
	archive->getCurrentDirPath = Dir_GetCurrentDirPath;
	
	archive->openFile = Dir_OpenFile;
	archive->getFileNum = Dir_GetFileNumber;
	archive->getFileName = Dir_GetFileName;
	archive->getFileNumByName = Dir_GetFileNumByName;
	archive->finish = Dir_Finish;
	
	archive->clone = Dir_Clone;
	
	archive->getChar = Dir_GetChar;
	archive->read = Dir_Read;
	archive->seek = Dir_Seek;
	archive->size = Dir_Size;
	archive->tell = Dir_Tell;
	archive->eof = Dir_EOF;
	archive->close = Dir_Close;
	
	return archive;
}


void Dir_Finish(SDL_Archive *archive)
{
	int i;
	DIRECTORY_DATA *data;
	
	if (archive != NULL) {
		data = archive->data;
		
		for (i = 0; i < data->chainNum; i++) {
			free(data->chain[i].filename);
		}
		
		DirContainer_Free(data->dirContainer);
		free(data->tempString);
		free(data->dirName);
		
		free(data->chain);
		if (data->fp != NULL) {
			fclose(data->fp);
		}
		free(data);
	}
	Archive_FreeMainContext(archive);
}


int Dir_GetFileNumByName(SDL_Archive *archive, const char *filename)
{
	DIRECTORY_DATA *data;
	int i;
	
	data = archive->data;
	
	for (i = 0; i < data->chainNum; i++) {
		if (strcmp(data->chain[i].filename, filename) == 0) {
			return i;
		}
	}
	
	return -1;
}


int Dir_OpenFile(SDL_Archive *archive, const char *fileName)
{
	DIRECTORY_DATA *data;
	int result;
	int fileIndex;
	int num;
	
	data = archive->data;
	
	num = Dir_GetFileNumByName(archive, fileName);
	
	if (num < 0 || data->chainNum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (data->fp != NULL) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	fileIndex = DirContainer_GetFileIndex(data->dirContainer, num);
	result = Dir_SetTempPath(data, data->chain[fileIndex].filename);
	if (result != 0) {
		return result;
	}
	data->fp = fopen(data->tempString, "rb");
	if (data->fp == NULL) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	data->openNumber = num;
	
	return ARCHIVE_SUCCESS;
}


int Dir_GetChildDirNum(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetChildDirNum(dir->dirContainer);
}


char *Dir_GetChildDirName(SDL_Archive *archive, int dirNum)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetChildDirName(dir->dirContainer, dirNum);
}


int Dir_GoDownDir(SDL_Archive *archive, const char *dirName)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GoDownDir(dir->dirContainer, dirName);
}


int Dir_GoUpDir(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GoUpDir(dir->dirContainer);
}


int Dir_MoveDirByPath(SDL_Archive *archive, const char *path)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_ParsePath(dir->dirContainer, path, DIRECTORY_MOVE_DIR, NULL);
}


char *Dir_GetCurrentDirName(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetCurrentDirName(dir->dirContainer);
}


char *Dir_GetCurrentDirPath(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetCurrentPath(dir->dirContainer);
}


int Dir_GetFileNumber(SDL_Archive *archive)
{
	DIRECTORY_DATA *dir;
	
	dir = (DIRECTORY_DATA*)archive->data;
	
	return DirContainer_GetFileNum(dir->dirContainer);
}


char *Dir_GetFileName(SDL_Archive *archive, int num)
{
	DIRECTORY_DATA *dir;
	int fileIndex;
	
	dir = (DIRECTORY_DATA*)archive->data;
	fileIndex = DirContainer_GetFileIndex(dir->dirContainer, num);
	
	return dir->chain[fileIndex].filename;
}


long Dir_Size(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return data->chain[data->openNumber].size;
}


int Dir_GetChar(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fgetc(data->fp);
}


long Dir_Tell(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return ftell(data->fp);
}


int Dir_EOF(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return feof(data->fp);
}


int Dir_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fread(mem, size, maxnum, data->fp);
}


int Dir_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	return fseek(data->fp, offset, whence);
}


int Dir_Close(SDL_Archive *archive)
{
	DIRECTORY_DATA *data;
	
	data = archive->data;
	
	fclose(data->fp);
	data->fp = NULL;
	data->openNumber = -1;
	
	return ARCHIVE_SUCCESS;
}
