/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "SDL_archive.h"


#define	TAR_BLOCK	512

/* typeflag */
#define REGTYPE  '0'   /* Regular file  */
#define AREGTYPE '_0' /* Regular file  */
#define LNKTYPE  '1'   /* Link          */
#define SYMTYPE  '2'   /* Reserved      */
#define CHRTYPE  '3'   /* Character special */
#define BLKTYPE  '4'   /* Block special */
#define DIRTYPE  '5'   /* Directory     */
#define FIFOTYPE '6'   /* FIFO special  */
#define CONTTYPE '7'   /* Reserved      */

typedef struct _tar_header
{
	char name[100];
	char mode[8];
	char uid[8];
	char gid[8];
	char size[12];
	char mtime[12];
	char chksum[8];
	char typeflag;
	char linkname[100];
	char magic[6];
	char version[2];
	char uname[32];
	char gname[32];
	char devmajor[8];
	char devminor[8];
	char prefix[167];
} TAR_HEADER;


typedef struct _tar_chain {
	long seek;
	long size;
	TAR_HEADER tar;
} TAR_CHAIN;


typedef struct _tar_data {
	int allFileNum;
	int allocatedChain;
	TAR_CHAIN *chain;
	
	FILE *fp;
	char *fileName;
	
	int openNumber;
	long seekInArchive;
	long nowSize;
	long nowSeek;
	
	DirContainer *dirContainer;
} TAR_DATA;


static TAR_DATA *Tar_Create(const char *file);
static void Tar_Finish(SDL_Archive *archive);
static int Tar_GetFileNumByName(SDL_Archive *archive, const char *fileName);
static int Tar_OpenFile(SDL_Archive *archive, const char *filename);
static int Tar_GetChildDirNum(SDL_Archive *archive);
static char *Tar_GetChildDirName(SDL_Archive *archive, int dirNum);
static int Tar_GoDownDir(SDL_Archive *archive, const char *dirName);
static int Tar_GoUpDir(SDL_Archive *archive);
static int Tar_MoveDirByPath(SDL_Archive *archive, const char *path);
static char *Tar_GetCurrentDirName(SDL_Archive *archive);
static char *Tar_GetCurrentDirPath(SDL_Archive *archive);
static int Tar_GetFileNumber(SDL_Archive *archive);
static char *Tar_GetFileName(SDL_Archive *archive, int num);
static int Tar_GetChar(SDL_Archive *archive);
static long Tar_Size(SDL_Archive *archive);
static long Tar_Tell(SDL_Archive *archive);
static int Tar_EOF(SDL_Archive *archive);
static int Tar_Close(SDL_Archive *archive);
static int Tar_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Tar_Seek(SDL_Archive *archive, const long offset, const int whence);
static SDL_Archive *Tar_Clone(SDL_Archive *archive);

static int Tar_GetFileData(TAR_DATA *tar, int fileSeek);


int Archive_IsTar(const char *file)
{
	FILE *fp;
	
	fp = fopen(file, "rb");
	if (fp == NULL) {
		return SDL_FALSE;
	}
	fclose(fp);
	
	return SDL_TRUE;
}


SDL_Archive *Archive_FromTar(const char *file)
{
	SDL_Archive *archive;
	TAR_DATA *tar;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	tar = Tar_Create(file);
	if (tar == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	tar->openNumber = -1;
	archive->data = tar;
	
	archive->getChildDirNum = Tar_GetChildDirNum;
	archive->getChildDirName = Tar_GetChildDirName;
	archive->goDownDir = Tar_GoDownDir;
	archive->goUpDir = Tar_GoUpDir;
	archive->moveDirByPath = Tar_MoveDirByPath;
	archive->getCurrentDirName = Tar_GetCurrentDirName;
	archive->getCurrentDirPath = Tar_GetCurrentDirPath;
	
	archive->openFile = Tar_OpenFile;
	archive->getFileNum = Tar_GetFileNumber;
	archive->getFileName = Tar_GetFileName;
	archive->getFileNumByName = Tar_GetFileNumByName;
	archive->finish = Tar_Finish;
	
	archive->clone = Tar_Clone;
	
	archive->getChar = Tar_GetChar;
	archive->read = Tar_Read;
	archive->seek = Tar_Seek;
	archive->size = Tar_Size;
	archive->tell = Tar_Tell;
	archive->eof = Tar_EOF;
	archive->close = Tar_Close;
	
	return archive;
}


int Tar_GetFileData(TAR_DATA *tar, int fileSeek)
{
	TAR_CHAIN *newChain;
	int dirIndex;
	void *tempptr;
	char *fileNameTemp;
	int fileNameLength;
	char sizetemp[2];
	int blockNum;
	int i;
	
	if (tar->allFileNum >= tar->allocatedChain) {
		tar->allocatedChain = tar->allocatedChain * 2;
		tempptr = realloc(tar->chain,
			sizeof(TAR_CHAIN) * tar->allocatedChain);
		if (tempptr == NULL) {
			return -1;
		}
		tar->chain = (TAR_CHAIN *)tempptr;
	}
	newChain = &tar->chain[tar->allFileNum];
	fread(&newChain->tar, TAR_BLOCK, 1, tar->fp);
	blockNum = 1;
	if (newChain->tar.typeflag == DIRTYPE) {
		DirContainer_ParsePath(
			tar->dirContainer, "/",
			DIRECTORY_MOVE_DIR, NULL);
		DirContainer_ParsePath(
			tar->dirContainer, newChain->tar.name,
			DIRECTORY_ADD_DIR, NULL);
		return blockNum;
	}
	fileNameTemp = strrchr(newChain->tar.name, '/');
	if (fileNameTemp == NULL) {
		DirContainer_ParsePath(
			tar->dirContainer, "/", DIRECTORY_NON, &dirIndex);
	} else {
		*fileNameTemp = '\0';
		DirContainer_ParsePath(
			tar->dirContainer, "/",
			DIRECTORY_MOVE_DIR, NULL);
		DirContainer_ParsePath(
			tar->dirContainer, newChain->tar.name,
			DIRECTORY_ADD_DIR, &dirIndex);
		fileNameLength = strlen(fileNameTemp + 1);
		memmove(newChain->tar.name, fileNameTemp + 1, fileNameLength);
		newChain->tar.name[fileNameLength] = '\0';
	}
	if (strlen(newChain->tar.name) == 0) {
		return blockNum;
	}
	DirContainer_AddFileToDirIndex(
		tar->dirContainer, tar->allFileNum, dirIndex);
	newChain->size = 0L;
	newChain->seek = fileSeek + TAR_BLOCK;
	/* convert octadecimal to decimal */
	for (i = 0; i < 12; i++) {
		sizetemp[0] = newChain->tar.size[i - 1];
		sizetemp[1] = '\0';
		newChain->size +=
			(long)(atoi(sizetemp) * pow(8, 11 - i));
	}
	blockNum += newChain->size / TAR_BLOCK;
	if (newChain->size % TAR_BLOCK != 0) { /* odd */
		blockNum++;
	}
	
	return blockNum;
}


TAR_DATA *Tar_Create(const char *file)
{
	int blockNum;
	long fileSeek;
	long fileSize;
	TAR_DATA *tar;
	
	tar = malloc(sizeof(TAR_DATA));
	if (tar == NULL) {
		return NULL;
	}
	
	tar->fileName = malloc(strlen(file) + 1);
	if (tar->fileName == NULL) {
		free(tar);
		return NULL;
	}
	strcpy(tar->fileName, file);
	
	tar->dirContainer = DirContainer_Create();
	if (tar->dirContainer == NULL) {
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	
	tar->fp = fopen(file, "rb");
	if (tar->fp == NULL) {
		DirContainer_Free(tar->dirContainer);
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	
	fseek(tar->fp, 0L, SEEK_END);
	fileSize = ftell(tar->fp);
	rewind(tar->fp);
	
	tar->allFileNum = 0;
	fileSeek = 0L;
	
	tar->chain = malloc(sizeof(TAR_CHAIN));
	if (tar->chain == NULL) {
		DirContainer_Free(tar->dirContainer);
		free(tar->fileName);
		fclose(tar->fp);
		free(tar);
		return NULL;
	}
	tar->allocatedChain = 1;
	while (fileSize - fileSeek >= TAR_BLOCK) {
		blockNum = Tar_GetFileData(tar, fileSeek);
		if (blockNum < 0) {
			break;
		}
		
		fileSeek += blockNum * TAR_BLOCK;
		fseek(tar->fp, fileSeek, SEEK_SET);
		tar->allFileNum++;
	}
	
	return tar;
}


SDL_Archive *Tar_Clone(SDL_Archive *orgArchive)
{
	DirContainer *dirContainer;
	SDL_Archive *archive;
	TAR_DATA *tar;
	TAR_DATA *orgTar;
	
	orgTar = (TAR_DATA*)orgArchive->data;
	
	tar = malloc(sizeof(TAR_DATA));
	if (tar == NULL) {
		return NULL;
	}
	tar->openNumber = -1;
	
	tar->fileName = malloc(strlen(orgTar->fileName) + 1);
	if (tar->fileName == NULL) {
		free(tar);
		return NULL;
	}
	strcpy(tar->fileName, orgTar->fileName);
	
	tar->chain = malloc(sizeof(TAR_CHAIN) * orgTar->allFileNum);
	if (tar->chain == NULL) {
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	memcpy(tar->chain, orgTar->chain, sizeof(TAR_CHAIN) * orgTar->allFileNum);
	
	tar->fp = fopen(orgTar->fileName, "rb");
	if (tar->fp == NULL) {
		free(tar->chain);
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	
	dirContainer = DirContainer_Clone(orgTar->dirContainer);
	if (dirContainer == NULL) {
		fclose(tar->fp);
		free(tar->chain);
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		DirContainer_Free(dirContainer);
		fclose(tar->fp);
		free(tar->chain);
		free(tar->fileName);
		free(tar);
		return NULL;
	}
	tar->openNumber = -1;
	archive->data = tar;
	tar->dirContainer = dirContainer;
	
	archive->getChildDirNum = Tar_GetChildDirNum;
	archive->getChildDirName = Tar_GetChildDirName;
	archive->goDownDir = Tar_GoDownDir;
	archive->goUpDir = Tar_GoUpDir;
	archive->moveDirByPath = Tar_MoveDirByPath;
	archive->getCurrentDirName = Tar_GetCurrentDirName;
	archive->getCurrentDirPath = Tar_GetCurrentDirPath;
	
	archive->openFile = Tar_OpenFile;
	archive->getFileNum = Tar_GetFileNumber;
	archive->getFileName = Tar_GetFileName;
	archive->getFileNumByName = Tar_GetFileNumByName;
	archive->finish = Tar_Finish;
	
	archive->clone = Tar_Clone;
	
	archive->getChar = Tar_GetChar;
	archive->read = Tar_Read;
	archive->seek = Tar_Seek;
	archive->size = Tar_Size;
	archive->tell = Tar_Tell;
	archive->eof = Tar_EOF;
	archive->close = Tar_Close;
	
	return archive;
}


void Tar_Finish(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	DirContainer_Free(tar->dirContainer);
	
	fclose(tar->fp);
	free(tar->chain);
	free(tar->fileName);
	free(tar);
}


int Tar_GetFileNumByName(SDL_Archive *archive, const char *fileName)
{
	int i;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	for (i = 0; i < tar->allFileNum; i++) {
		if (strcmp(tar->chain[i].tar.name, fileName) == 0) {
			return i;
		}
	}
	
	return -1;
}


int Tar_OpenFile(SDL_Archive *archive, const char *filename)
{
	TAR_DATA *tar;
	int num;
	
	tar = (TAR_DATA*)archive->data;
	
	num = Tar_GetFileNumByName(archive, filename);
	if (num < 0 || tar->allFileNum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (tar->openNumber != -1) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	fseek(tar->fp, tar->chain[num].seek, SEEK_SET);
	tar->openNumber = num;
	tar->nowSeek = 0L;
	tar->nowSize = tar->chain[num].size;
	tar->seekInArchive = tar->chain[num].seek;
	
	return ARCHIVE_SUCCESS;
}


int Tar_GetChildDirNum(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GetChildDirNum(tar->dirContainer);
}


char *Tar_GetChildDirName(SDL_Archive *archive, int dirNum)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GetChildDirName(tar->dirContainer, dirNum);
}


int Tar_GoDownDir(SDL_Archive *archive, const char *dirName)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GoDownDir(tar->dirContainer, dirName);
}


int Tar_GoUpDir(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GoUpDir(tar->dirContainer);
}


int Tar_MoveDirByPath(SDL_Archive *archive, const char *path)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_ParsePath(tar->dirContainer, path, DIRECTORY_MOVE_DIR, NULL);
}


char *Tar_GetCurrentDirName(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GetCurrentDirName(tar->dirContainer);
}


char *Tar_GetCurrentDirPath(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GetCurrentPath(tar->dirContainer);
}


int Tar_GetFileNumber(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return DirContainer_GetFileNum(tar->dirContainer);
}


char *Tar_GetFileName(SDL_Archive *archive, int num)
{
	int absoluteIndex;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	absoluteIndex = DirContainer_GetFileIndex(tar->dirContainer, num);
	
	return tar->chain[absoluteIndex].tar.name;
}


int Tar_GetChar(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	if (tar->nowSeek >= tar->nowSize) {
		return EOF;
	}
	tar->nowSeek++;
	return fgetc(tar->fp);
}


long Tar_Size(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return tar->nowSize;
}


long Tar_Tell(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return tar->nowSeek;
}


int Tar_EOF(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	if (tar->nowSeek >= tar->nowSize) {
		return 1;
	} else {
		return 0;
	}
}


int Tar_Close(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	tar->openNumber = -1;
	tar->seekInArchive = 0L;
	tar->nowSize = 0L;
	tar->nowSeek = 0L;
	
	return ARCHIVE_SUCCESS;
}


int Tar_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	int count;
	long position;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	count = 0;
	position = 0L;
	while (count < maxnum && tar->nowSeek + position < tar->nowSize) {
		fread(((char*)mem) + position, size, 1, tar->fp);
		position += size;
		count++;
	}
	
	tar->nowSeek += size * count;
	
	return count;
}


int Tar_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	long position;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	switch (whence) {
	case SEEK_SET:
		position = offset;
		break;
	case SEEK_CUR:
		position = tar->nowSeek + offset;
		break;
	case SEEK_END:
		position = tar->nowSize + offset;
		break;
	default:
		return ARCHIVE_ERROR_WHENCE;
	}
	
	if (position > tar->nowSize) {
		position = tar->nowSize;
	}
	if (position < 0) {
		position = 0;
	}
	
	fseek(tar->fp, tar->seekInArchive + position, SEEK_SET);
	
	tar->nowSeek = position;
	
	return ARCHIVE_SUCCESS;
}
