/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SDL_archive.h"
#include "SDL_version.h"


#define DEFAULT_ARCHIVERS_ALLOCATE 16


typedef struct _Archiver
{
	Archive_IsOpenable isOpenable;
	Archive_FromFormat fromFormat;
} Archiver;


struct _ArchiverList
{
	int archiversNum;
	int archiversAllocated;
	Archiver *archivers;
};


const SDL_version *Archive_Linked_Version(void)
{
	static SDL_version linked_version;
	ARCHIVE_VERSION(&linked_version);
	return(&linked_version);
}


SDL_Archive *Archive_Alloc(void)
{
	SDL_Archive *archive;
	
	archive = malloc(sizeof(SDL_Archive));
	if (archive == NULL) {
		return NULL;
	}
	
	archive->getChildDirNum = NULL;
	archive->getChildDirName = NULL;
	archive->goDownDir = NULL;
	archive->goUpDir = NULL;
	archive->moveDirByPath = NULL;
	archive->getCurrentDirName = NULL;
	archive->getCurrentDirPath = NULL;
	
	archive->getFileNum = NULL;
	archive->getFileName = NULL;
	archive->openFile = NULL;
	archive->finish = NULL;
	
	archive->clone = NULL;
	
	archive->getChar = NULL;
	archive->read = NULL;
	archive->seek = NULL;
	archive->size = NULL;
	archive->tell = NULL;
	archive->eof = NULL;
	archive->close = NULL;
	
	archive->data = NULL;
	
	return archive;
}


void Archive_FreeMainContext(SDL_Archive *archive)
{
	free(archive);
}


ArchiverList *ArchiverList_Create(void)
{
	ArchiverList *archiverList;
	
	archiverList = (ArchiverList*)malloc(sizeof(ArchiverList));
	
	archiverList->archivers =
		(Archiver*)malloc(sizeof(Archiver) * DEFAULT_ARCHIVERS_ALLOCATE);
	archiverList->archiversAllocated = DEFAULT_ARCHIVERS_ALLOCATE;
	archiverList->archiversNum = 0;
	
	return archiverList;
}


int ArchiverList_AddArchiver(
	ArchiverList *archiverList,
	Archive_IsOpenable isOpenable,
	Archive_FromFormat fromFormat)
{
	Archiver *allocateTemp;
	
	if (archiverList->archiversNum >= archiverList->archiversAllocated) {
		allocateTemp = (Archiver*)realloc(
			archiverList->archivers,
			sizeof(Archiver) * archiverList->archiversAllocated * 2);
		if (allocateTemp == NULL) {
			return -1;
		}
		archiverList->archivers = allocateTemp;
		archiverList->archiversAllocated *= 2;
	}
	
	archiverList->archivers[archiverList->archiversNum].isOpenable =
		isOpenable;
	archiverList->archivers[archiverList->archiversNum].fromFormat =
		fromFormat;
	
	archiverList->archiversNum++;
	
	return 0;
}


SDL_Archive *Archive_FromArchiverList(
	ArchiverList *archiverList,
	const char *file)
{
	int i;
	SDL_Archive *archive;
	
	for (i = 0; i < archiverList->archiversNum; i++) {
		if (archiverList->archivers[i].isOpenable(file)) {
			archive = archiverList->archivers[i].fromFormat(file);
			if (archive != NULL) {
				return archive;
			}
		}
	}
	return NULL;
}


void ArchiverList_Free(ArchiverList *archiverList)
{
	if (archiverList != NULL) {
		free(archiverList->archivers);
		free(archiverList);
	}
}







typedef struct _PieceOfDirectory PieceOfDirectory;

struct _PieceOfDirectory
{
	char *dirName;
	
	int parentDirIndex;
	
	int childDirIndexNum;
	int childDirIndexAllocated;
	int *childDirIndex;
	
	int fileIndexNum;
	int fileIndexAllocated;
	int *fileIndex;
};


struct _DirContainer
{
	int dirNum;
	int dirAllocated;
	PieceOfDirectory *piecesOfDir;
	
	int rootDirIndex;
	
	int currentDirIndex;
	
	int currentPathAllocated;
	char *currentPath;
};


static int DirContainer_ReallocFileIndex(PieceOfDirectory *dir);
static int DirContainer_ReallocChildDirIndex(PieceOfDirectory *dir);
static int DirContainer_ReallocPiecesOfDir(DirContainer *dirContainer);
static void DirContainer_FreePiecesOfDirectory(DirContainer *dirContainer);
static int DirContainer_CopyPiecesOfDirectory(
	DirContainer *dstDirList, DirContainer *srcDirList);
static int DirContainer_CopyDirName(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir);
static int DirContainer_CopyChildDirIndex(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir);
static int DirContainer_CopyFileIndex(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir);


DirContainer *DirContainer_Create(void)
{
	DirContainer *dirContainer;
	
	dirContainer = (DirContainer*)malloc(sizeof(DirContainer));
	if (dirContainer == NULL) {
		return NULL;
	}
	
	dirContainer->piecesOfDir = (PieceOfDirectory*)malloc(
		sizeof(PieceOfDirectory) * DEFAULT_ARCHIVERS_ALLOCATE);
	if (dirContainer->piecesOfDir == NULL) {
		free(dirContainer);
		return NULL;
	}
	dirContainer->dirAllocated = DEFAULT_ARCHIVERS_ALLOCATE;
	dirContainer->dirNum = 0;
	
	dirContainer->currentPath = (char*)malloc(2);
	if (dirContainer->currentPath == NULL) {
		free(dirContainer->piecesOfDir);
		free(dirContainer);
		return NULL;
	}
	dirContainer->currentPath[0] = '/';
	dirContainer->currentPath[1] = '\0';
	dirContainer->currentPathAllocated = 2;
	
	dirContainer->rootDirIndex = 0;
	dirContainer->currentDirIndex = 0;
	dirContainer->piecesOfDir[0].dirName = NULL;
	dirContainer->piecesOfDir[0].parentDirIndex = -1;
	dirContainer->piecesOfDir[0].childDirIndexNum = 0;
	dirContainer->piecesOfDir[0].childDirIndexAllocated = 0;
	dirContainer->piecesOfDir[0].childDirIndex = NULL;
	dirContainer->piecesOfDir[0].fileIndexNum = 0;
	dirContainer->piecesOfDir[0].fileIndexAllocated = 0;
	dirContainer->piecesOfDir[0].fileIndex = NULL;
	dirContainer->dirNum++;
	
	return dirContainer;
}


void DirContainer_Free(DirContainer *dirContainer)
{
	int i;
	
	for (i = 0; i < dirContainer->dirNum; i++) {
		free(dirContainer->piecesOfDir[i].dirName);
		free(dirContainer->piecesOfDir[i].childDirIndex);
		free(dirContainer->piecesOfDir[i].fileIndex);
	}
	free(dirContainer->currentPath);
	free(dirContainer->piecesOfDir);
	free(dirContainer);
}


void DirContainer_FreePiecesOfDirectory(DirContainer *dirContainer)
{
	int i;
	
	for (i = 0; i < dirContainer->dirNum; i++) {
		if (dirContainer->piecesOfDir[i].dirName) {
			free(dirContainer->piecesOfDir[i].dirName);
			dirContainer->piecesOfDir[i].dirName = NULL;
		}
		if (dirContainer->piecesOfDir[i].childDirIndex) {
			free(dirContainer->piecesOfDir[i].childDirIndex);
			dirContainer->piecesOfDir[i].childDirIndex = NULL;
		}
		if (dirContainer->piecesOfDir[i].fileIndex) {
			free(dirContainer->piecesOfDir[i].fileIndex);
			dirContainer->piecesOfDir[i].fileIndex = NULL;
		}
	}
}


int DirContainer_CopyDirName(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir)
{
	int length;
	char *dirNameTemp;
	
	if (srcDir->dirName) {
		length = strlen(srcDir->dirName);
		dirNameTemp = (char*)malloc(length + 1);
		if (dirNameTemp == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dstDir->dirName = dirNameTemp;
		strcpy(dstDir->dirName, srcDir->dirName);
	} else {
		dstDir->dirName = NULL;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_CopyChildDirIndex(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir)
{
	int needSize;
	int *childDirIndexTemp;
	
	if (srcDir->childDirIndex) {
		needSize = sizeof(int) * srcDir->childDirIndexAllocated;
		childDirIndexTemp = (int*)malloc(needSize);
		if (childDirIndexTemp == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dstDir->childDirIndex = childDirIndexTemp;
		memcpy(
			dstDir->childDirIndex, srcDir->childDirIndex,
			sizeof(int) * srcDir->childDirIndexNum);
		dstDir->childDirIndexNum = srcDir->childDirIndexNum;
		dstDir->childDirIndexAllocated = srcDir->childDirIndexAllocated;
	} else {
		dstDir->childDirIndex = NULL;
		dstDir->childDirIndexNum = srcDir->childDirIndexNum;
		dstDir->childDirIndexAllocated = srcDir->childDirIndexAllocated;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_CopyFileIndex(
	PieceOfDirectory *dstDir, PieceOfDirectory *srcDir)
{
	int needSize;
	int *fileIndexTemp;
	
	if (srcDir->fileIndex) {
		needSize = sizeof(int) * srcDir->fileIndexAllocated;
		fileIndexTemp = (int*)malloc(needSize);
		if (fileIndexTemp == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dstDir->fileIndex = fileIndexTemp;
		memcpy(
			dstDir->fileIndex, srcDir->fileIndex,
			sizeof(int) * srcDir->fileIndexNum);
		dstDir->fileIndexNum = srcDir->fileIndexNum;
		dstDir->fileIndexAllocated = srcDir->fileIndexAllocated;
	} else {
		dstDir->fileIndex = NULL;
		dstDir->fileIndexNum = srcDir->fileIndexNum;
		dstDir->fileIndexAllocated = srcDir->fileIndexAllocated;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_CopyPiecesOfDirectory(
	DirContainer *dstDirList, DirContainer *srcDirList)
{
	int i;
	int result;
	
	for (i = 0; i < dstDirList->dirNum; i++) {
		dstDirList->piecesOfDir[i].dirName = NULL;
		dstDirList->piecesOfDir[i].childDirIndex = NULL;
		dstDirList->piecesOfDir[i].fileIndex = NULL;
	}
	for (i = 0; i < srcDirList->dirNum; i++) {
		result = DirContainer_CopyDirName(
			&dstDirList->piecesOfDir[i],
			&srcDirList->piecesOfDir[i]);
		if (result != ARCHIVE_SUCCESS) {
			DirContainer_FreePiecesOfDirectory(dstDirList);
			return result;
		}
		
		result = DirContainer_CopyChildDirIndex(
			&dstDirList->piecesOfDir[i],
			&srcDirList->piecesOfDir[i]);
		if (result != ARCHIVE_SUCCESS) {
			DirContainer_FreePiecesOfDirectory(dstDirList);
			return result;
		}
		
		result = DirContainer_CopyFileIndex(
			&dstDirList->piecesOfDir[i],
			&srcDirList->piecesOfDir[i]);
		if (result != ARCHIVE_SUCCESS) {
			DirContainer_FreePiecesOfDirectory(dstDirList);
			return result;
		}
	}
	
	return ARCHIVE_SUCCESS;
}


DirContainer *DirContainer_Clone(DirContainer *orgDirList)
{
	DirContainer *dirContainer;
	int result;
	
	dirContainer = (DirContainer*)malloc(sizeof(DirContainer));
	if (dirContainer == NULL) {
		return NULL;
	}
	
	dirContainer->currentPath = (char*)malloc(
		sizeof(char) * orgDirList->currentPathAllocated);
	if (dirContainer->currentPath == NULL) {
		free(dirContainer);
		return NULL;
	}
	strcpy(dirContainer->currentPath, orgDirList->currentPath);
	dirContainer->currentPath[strlen(dirContainer->currentPath)] = '\0';
	dirContainer->currentPathAllocated = orgDirList->currentPathAllocated;
	
	dirContainer->piecesOfDir = (PieceOfDirectory*)malloc(
		sizeof(PieceOfDirectory) * orgDirList->dirAllocated);
	if (dirContainer->piecesOfDir == NULL) {
		free(dirContainer->currentPath);
		free(dirContainer);
		return NULL;
	}
	memcpy(
		dirContainer->piecesOfDir, orgDirList->piecesOfDir,
		sizeof(PieceOfDirectory) * orgDirList->dirNum);
	dirContainer->dirAllocated = orgDirList->dirAllocated;
	dirContainer->dirNum = orgDirList->dirNum;
	
	result = DirContainer_CopyPiecesOfDirectory(dirContainer, orgDirList);
	if (result != ARCHIVE_SUCCESS) {
		return NULL;
	}
	
	return dirContainer;
}


int DirContainer_ReallocCurrentPath(DirContainer *dirContainer, int length)
{
	char *tempString;
	int needSize;
	
	needSize = dirContainer->currentPathAllocated;
	while (needSize < length + 1) {
		needSize *= 2;
	}
	if (needSize == dirContainer->currentPathAllocated) {
		return ARCHIVE_SUCCESS;
	}
	tempString = (char*)realloc(
		dirContainer->currentPath, needSize);
	if (tempString == NULL) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	dirContainer->currentPath = tempString;
	dirContainer->currentPathAllocated = needSize;
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_(
	DirContainer *dirContainer, const char *path, int flags,
	int *nowDirIndex, int *readLength)
{
	char *dirName;
	int dirNameLength;
	int i;
	int absoluteIndex;
	char *compareDirName;
	int childDirIndexNum;
	int result;
	
	dirNameLength = 0;
	while (path[dirNameLength] != '\0' && path[dirNameLength] != '/') {
		dirNameLength++;
	}
	*readLength = dirNameLength;
	if (dirNameLength == 0) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	dirName = (char*)malloc(dirNameLength + 1);
	if (dirName == NULL) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	strncpy(dirName, path, dirNameLength);
	dirName[dirNameLength] = '\0';
	childDirIndexNum =
		dirContainer->piecesOfDir[*nowDirIndex].childDirIndexNum;
	for (i = 0; i < childDirIndexNum; i++) {
		absoluteIndex =
			dirContainer->piecesOfDir[*nowDirIndex].childDirIndex[i];
		compareDirName =
			dirContainer->piecesOfDir[absoluteIndex].dirName;
		if (strcmp(dirName, compareDirName) == 0) {
			break;
		}
	}
	if (i >= dirContainer->piecesOfDir[*nowDirIndex].childDirIndexNum) {
		/* not found */
		if (flags & DIRECTORY_ADD_DIR) {
			result = DirContainer_AddDirectoryToDirIndex(
				dirContainer, dirName, *nowDirIndex);
			if (result != ARCHIVE_SUCCESS) {
				free(dirName);
				return result;
			}
		} else {
			free(dirName);
			return ARCHIVE_ERROR_NO_EXIST_DIRECTORY;
		}
	}
	free(dirName);
	*nowDirIndex =
		dirContainer->piecesOfDir[*nowDirIndex].childDirIndex[i];
	
	return ARCHIVE_SUCCESS;
}


/* path delimiter is '/' */
int DirContainer_ParsePath(
	DirContainer *dirContainer, const char *path,
	int flags, int *returnDirIndex)
{
	int position;
	int lastDirNamePosition;
	int nowDirIndex;
	int result;
	int isFromRoot;
	int readLength;
	
	if (path[0] == '/') {
		position = 1;
		lastDirNamePosition = 1;
		nowDirIndex = dirContainer->rootDirIndex;
		isFromRoot = SDL_TRUE;
	} else {
		position = 0;
		lastDirNamePosition = 0;
		nowDirIndex = dirContainer->currentDirIndex;
		isFromRoot = SDL_FALSE;
	}
	while (path[lastDirNamePosition] != '\0') {
		result = DirContainer_(
			dirContainer, path + lastDirNamePosition, flags,
			&nowDirIndex, &readLength);
		if (result != ARCHIVE_SUCCESS) {
			return result;
		}
		position += readLength;
		if (path[position] == '\0') {
			break;
		}
		position++;
		lastDirNamePosition = position;
	}
	if (flags & DIRECTORY_MOVE_DIR) {
		dirContainer->currentDirIndex = nowDirIndex;
		result = DirContainer_ReallocCurrentPath(dirContainer, strlen(path));
		if (result != ARCHIVE_SUCCESS) {
			if (isFromRoot) {
				strcpy(dirContainer->currentPath, path);
			} else {
				strcat(dirContainer->currentPath, "/");
				strcat(dirContainer->currentPath, path);
			}
		}
	}
	if (returnDirIndex) {
		*returnDirIndex = nowDirIndex;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_ReallocChildDirIndex(PieceOfDirectory *dir)
{
	int needSize;
	int *tempIndex;
	
	if (dir->childDirIndexNum >= dir->childDirIndexAllocated) {
		if (dir->childDirIndex == NULL) {
			needSize = DEFAULT_ARCHIVERS_ALLOCATE;
		} else {
			needSize = dir->childDirIndexAllocated * 2;
		}
		tempIndex = (int*)realloc(
			dir->childDirIndex, sizeof(int) * needSize);
		if (tempIndex == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dir->childDirIndex = tempIndex;
		dir->childDirIndexAllocated = needSize;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_ReallocFileIndex(PieceOfDirectory *dir)
{
	int needSize;
	int *tempIndex;
	
	if (dir->fileIndexNum >= dir->fileIndexAllocated) {
		if (dir->fileIndex == NULL) {
			needSize = DEFAULT_ARCHIVERS_ALLOCATE;
		} else {
			needSize = dir->fileIndexAllocated * 2;
		}
		tempIndex = (int*)realloc(
			dir->fileIndex, sizeof(int) * needSize);
		if (tempIndex == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dir->fileIndex = tempIndex;
		dir->fileIndexAllocated = needSize;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_ReallocPiecesOfDir(DirContainer *dirContainer)
{
	PieceOfDirectory *tempDir;
	
	if (dirContainer->dirNum >= dirContainer->dirAllocated) {
		tempDir = realloc(
			dirContainer->piecesOfDir,
			sizeof(PieceOfDirectory) * dirContainer->dirAllocated * 2);
		if (tempDir == NULL) {
			return ARCHIVE_ERROR_ALLOCATE;
		}
		dirContainer->piecesOfDir = tempDir;
		dirContainer->dirAllocated *= 2;
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_AddDirectory(DirContainer *dirContainer, const char *name)
{
	return DirContainer_AddDirectoryToDirIndex(
		dirContainer, name, dirContainer->currentDirIndex);
}


int DirContainer_AddDirectoryToDirIndex(
	DirContainer *dirContainer, const char *name, int dirIndex)
{
	int result;
	PieceOfDirectory *currentDir;
	PieceOfDirectory *newDir;
	char *tempString;
	
	if (strcmp(name, ".") == 0 || strcmp(name, "..") == 0) {
		return ARCHIVE_ERROR_INVALID_NAME;
	}
	
	result = DirContainer_ReallocPiecesOfDir(dirContainer);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	currentDir = &dirContainer->piecesOfDir[dirIndex];
	
	result = DirContainer_ReallocChildDirIndex(currentDir);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	tempString = malloc(strlen(name) + 1);
	if (tempString == NULL) {
		return ARCHIVE_ERROR_ALLOCATE;
	}
	strcpy(tempString, name);
	
	newDir = &dirContainer->piecesOfDir[dirContainer->dirNum];
	newDir->dirName = tempString;
	newDir->parentDirIndex = dirIndex;
	newDir->childDirIndexNum = 0;
	newDir->childDirIndexAllocated = 0;
	newDir->childDirIndex = NULL;
	newDir->fileIndexNum = 0;
	newDir->fileIndexAllocated = 0;
	newDir->fileIndex = NULL;
	
	currentDir->childDirIndex[currentDir->childDirIndexNum] =
		dirContainer->dirNum;
	currentDir->childDirIndexNum++;
	
	dirContainer->dirNum++;
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_AddFile(DirContainer *dirContainer, int fileIndex)
{
	return DirContainer_AddFileToDirIndex(
		dirContainer, fileIndex, dirContainer->currentDirIndex);
}


int DirContainer_AddFileToDirIndex(
	DirContainer *dirContainer, int fileIndex, int dirIndex)
{
	PieceOfDirectory *basisDir;
	int result;
	
	basisDir = &dirContainer->piecesOfDir[dirIndex];
	result = DirContainer_ReallocFileIndex(basisDir);
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	
	basisDir->fileIndex[basisDir->fileIndexNum] = fileIndex;
	basisDir->fileIndexNum++;
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_GetChildDirNum(DirContainer *dirContainer)
{
	int currentDirIndex;
	
	currentDirIndex = dirContainer->currentDirIndex;
	
	return dirContainer->piecesOfDir[currentDirIndex].childDirIndexNum;
}


char *DirContainer_GetChildDirName(DirContainer *dirContainer, int num)
{
	PieceOfDirectory *currentDir;
	int currentDirIndex;
	int absoluteDirIndex;
	
	currentDirIndex = dirContainer->currentDirIndex;
	currentDir = &dirContainer->piecesOfDir[currentDirIndex];
	absoluteDirIndex = currentDir->childDirIndex[num];
	
	return dirContainer->piecesOfDir[absoluteDirIndex].dirName;
}


int DirContainer_GetFileNum(DirContainer *dirContainer)
{
	int currentDirIndex;
	
	currentDirIndex = dirContainer->currentDirIndex;
	
	return dirContainer->piecesOfDir[currentDirIndex].fileIndexNum;
}


int DirContainer_GetFileIndex(DirContainer *dirContainer, int num)
{
	PieceOfDirectory *currentDir;
	
	currentDir = &dirContainer->piecesOfDir[dirContainer->currentDirIndex];
	
	return currentDir->fileIndex[num];
}


char *DirContainer_GetCurrentDirName(DirContainer *dirContainer)
{
	PieceOfDirectory *currentDir;
	
	currentDir = &dirContainer->piecesOfDir[dirContainer->currentDirIndex];
	
	if (currentDir->dirName) {
		return currentDir->dirName;
	} else {
		return "";
	}
}


char *DirContainer_GetCurrentPath(DirContainer *dirContainer)
{
	return dirContainer->currentPath;
}


int DirContainer_GoUpDir(DirContainer *dirContainer)
{
	PieceOfDirectory *currentDir;
	char *lastDelimiter;
	
	if (dirContainer->currentDirIndex == 0) {
		return ARCHIVE_ERROR_ALREADY_ROOT;
	}
	
	currentDir = &dirContainer->piecesOfDir[dirContainer->currentDirIndex];
	
	dirContainer->currentDirIndex = currentDir->parentDirIndex;
	
	lastDelimiter = strrchr(dirContainer->currentPath, '/');
	if (lastDelimiter) {
		*lastDelimiter = '\0';
	} else {
		dirContainer->currentPath[0] = '\0';
	}
	
	/* dirContainer->currentPath[0] is always '/' */
	if (dirContainer->currentPath[0] == '\0') {
		dirContainer->currentPath[0] = '/';
		dirContainer->currentPath[1] = '\0';
	}
	
	return ARCHIVE_SUCCESS;
}


int DirContainer_GoDownDir(DirContainer *dirContainer, const char *dirName)
{
	PieceOfDirectory *currentDir;
	int i;
	int absoluteIndex;
	char *compareDirName;
	int result;
	
	if (dirContainer->currentDirIndex == -1) {
		return ARCHIVE_ERROR_ALREADY_ROOT;
	}
	
	currentDir = &dirContainer->piecesOfDir[dirContainer->currentDirIndex];
	
	for (i = 0; i < currentDir->childDirIndexNum; i++) {
		absoluteIndex = currentDir->childDirIndex[i];
		compareDirName = dirContainer->piecesOfDir[absoluteIndex].dirName;
		if (strcmp(dirName, compareDirName) == 0) {
			break;
		}
	}
	if (i >= currentDir->childDirIndexNum) {
		/* not found */
		return ARCHIVE_ERROR_NO_EXIST_DIRECTORY;
	}
	dirContainer->currentDirIndex = absoluteIndex;
	
	result =DirContainer_ReallocCurrentPath(
		dirContainer, strlen(dirContainer->currentPath) + 1 + strlen(dirName));
	if (result != ARCHIVE_SUCCESS) {
		return result;
	}
	/* dirContainer->currentPath[0] is always '/' */
	if (dirContainer->currentPath[1] != '\0') {
		strcat(dirContainer->currentPath, "/");
	}
	strcat(dirContainer->currentPath, dirName);
	
	return ARCHIVE_SUCCESS;
}



