package org.maachang.jni.io;

import java.io.IOException;
import java.io.InputStream;

/**
 * NativeInputStream.
 * 
 * @version 2010/06/04
 * @author  masahito suzuki
 * @since   SeabassNativeIO-1.0.0
 */
public class NativeInputStream extends InputStream {
    private static final int ACCESS_MODE = NativeIODefine.MODE_SEQ ;
    private static final int IO_OPT = NativeIODefine.OPT_READ ;
    private static final int TEMP_LEN = 512 ;
    private final FastMemoryBuffer temp = new FastMemoryBuffer( TEMP_LEN ) ;
    private int position = 0 ;
    private int length = 0 ;
    private long handle = -1L ;
    private long fileSize = 0L ;
    private long allPos = 0L ;
    
    private NativeInputStream() {
        
    }
    
    /**
     * コンストラクタ.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public NativeInputStream( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( NativeIODefine.useFile( name ) == false ) {
            throw new IllegalArgumentException( "指定ファイル[" + name + "]は存在しません" ) ;
        }
        name = NativeIODefine.fullPath( name ) ;
        this.handle = NativeIO.open( ACCESS_MODE,IO_OPT,NativeIODefine.nativeString( name ) ) ;
        if( this.handle == -1L ) {
            throw new IOException( "指定ファイル[" + name + "]のオープンに失敗しました" ) ;
        }
        this.fileSize = NativeIO.getLength( this.handle ) ;
    }
    
    protected void finalize() throws Exception {
        try {
            close() ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * ファイルクローズ.
     */
    public void close() {
        this.temp.clear() ;
        if( this.handle > -1L ) {
            NativeIO.close( this.handle ) ;
            this.handle = -1L ;
            this.position = 0 ;
            this.length = -1 ;
            this.fileSize = -1L ;
            this.allPos = -1L ;
        }
    }
    
    /**
     * 残り容量を取得.
     * @return int 残り容量が返されます.
     * @exception IOException I/O例外.
     */
    public int available() throws IOException {
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        return ( int )( ( fileSize - allPos ) & 0x000000007fffffffL ) ;
    }
    
    /**
     * 読み込み処理.
     * @return int 読み込まれた情報が返されます.
     * @exception IOException I/O例外.
     */
    public int read() throws IOException {
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        if( length <= -1 ) {
            return -1 ;
        }
        if( position >= length ) {
            length = NativeIO.read( handle,-1L,temp.address,0,TEMP_LEN ) ;
            position = 0 ;
            if( length <= 0 ) {
                length = -1 ;
                return -1 ;
            }
        }
        int ret = ( int )( temp.getByte( position ) & 0x000000ff ) ;
        position ++ ;
        allPos ++ ;
        return ret ;
    }
    
    /**
     * markサポート.
     * @return boolean [false]が返されます.
     */
    public boolean markSupported() {
        return false ;
    }
}
