package org.maachang.jni.io ;

import java.nio.ByteBuffer;

/**
 * 高速DirectByteBufferをNativeBuffer操作オブジェクト.
 * <p>このバッファは、速度重視を考慮した、チェック処理なしで、
 *    Nativeメモリにアクセスするオブジェクトです.
 *    欠点として、ダイレクトにNativeメモリにアクセスするので、
 *    このオブジェクトによって操作した結果によって、処理異常が
 *    発生した場合、最悪JavaVMが異常停止してしまいます.</p>
 * 
 * @version 2010/06/04
 * @author  masahito suzuki
 * @since   SeabassNativeIO-1.0.0
 */
public class FastMemoryByteBuffer
    extends AbstractFastNativeBuffer implements NativeByteBuffer {
    private ByteBuffer buf = null ;
    
    /**
     * コンストラクタ.
     */
    public FastMemoryByteBuffer() {
        
    }
    
    /**
     * コンストラクタ.
     * @param byteBuffer 対象のByteBufferを設定します.
     * @exception Exception 例外.
     */
    public FastMemoryByteBuffer( ByteBuffer buf )
        throws Exception {
        this.create( buf ) ;
    }
    
    /**
     * コンストラクタ.
     * @param byteBuffer 対象のByteBufferを設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @exception Exception 例外.
     */
    public FastMemoryByteBuffer( ByteBuffer buf,int offset,int length )
        throws Exception {
        this.create( buf,offset,length ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        clear() ;
    }
    
    /**
     * 情報生成.
     * @param byteBuffer 対象のByteBufferを設定します.
     * @exception Exception 例外.
     */
    public void create( ByteBuffer buf )
        throws Exception {
        clear() ;
        this.address = Unsafe.getDirectByteBufferNativeAddress( buf ) ;
        this.length = buf.capacity() ;
        this.buf = buf ;
    }
    
    /**
     * 情報生成.
     * @param byteBuffer 対象のByteBufferを設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @exception Exception 例外.
     */
    public void create( ByteBuffer buf,int offset,int length )
        throws Exception {
        clear() ;
        this.address = Unsafe.getDirectByteBufferNativeAddress( buf ) + offset ;
        this.length = length ;
        this.buf = buf ;
    }
    
    /**
     * メモリクリア.
     */
    public void clear() {
        buf = null ;
        address = 0L ;
        length = 0 ;
    }
    
    /**
     * ByteBufferを取得.
     * @return ByteBuffer ByteBufferオブジェクトが返されます.
     */
    public ByteBuffer getByteBuffer() {
        return buf ;
    }
    
    /**
     * positionを取得.
     * @return int ByteBufferのポジションが返されます.
     */
    public int position() {
        return buf.position() ;
    }
    
    /**
     * positionを設定.
     * @param pos ByteBufferのポジションを設定します.
     */
    public void position( int pos ) {
        buf.position( pos ) ;
    }
    
    /**
     * limitを取得.
     * @return int ByteBufferのリミットが返されます.
     */
    public int limit() {
        return buf.limit() ;
    }
    
    /**
     * limitを設定.
     * @param lim ByteBufferのリミットを設定します.
     */
    public void limit( int lim ) {
        buf.limit( lim ) ;
    }
}

