package org.maachang.mimdb.core;

import org.maachang.mimdb.MimdbException;
import org.maachang.mimdb.core.impl.EqualsNoList;
import org.maachang.mimdb.core.impl.MimdbUtils;
import org.maachang.mimdb.core.impl.SqlAnalyzer;
import org.maachang.mimdb.core.impl.WhereBlock;
import org.maachang.mimdb.core.util.ObjectList;

/**
 * ステートメント.
 * 
 * @version 2013/10/14
 * @author masahito suzuki
 * @since MasterInMemDB 1.00
 */
public class MimdbStatement implements MimdbBase {
    
    /** データベースID. **/
    protected long dbId = -1 ;
    
    /** テーブル名. **/
    private String tableName = null ;
    
    /** 表示カラム名. **/
    private final ObjectList<String> columns = new ObjectList<String>() ;
    
    /** 全表示カラム. **/
    private boolean allColumn = true ;
    
    /** 件数取得. **/
    private boolean countFlag = false ;
    
    /** 検索条件. **/
    private final ObjectList<MimdbSearchElement> where = new ObjectList<MimdbSearchElement>() ;
    
    /** ソートカラム. **/
    private final ObjectList<String> sortList = new ObjectList<String>() ;
    
    /** ソート条件. **/
    private final ObjectList<Boolean> descList = new ObjectList<Boolean>() ;
    
    
    /** コンパイル済みオブジェクト. **/
    private MimdbPreparedStatement compile = null ;
    
    /** オフセット位置. **/
    private int offset = -1 ;
    
    /** リミット位置. **/
    private int limit = -1 ;
    
    /**
     * コンストラクタ.
     */
    public MimdbStatement() {
        
    }
    
    /**
     * コンストラクタ.
     * @param table 対象のテーブル名を設定します.
     * @exception Exception 例外.
     */
    public MimdbStatement( String table ) throws Exception {
        this.setTable( table ) ;
    }
    
    /**
     * SQL文を指定して、MimdbStatementを生成.
     * @param sql 対象のSQL文を設定します.
     * @return MimdbStatement ステートメントオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbStatement createStatement( String sql ) throws Exception {
        MimdbStatement ret = new MimdbStatement() ;
        SqlAnalyzer.analysis( ret,sql ) ;
        return ret ;
    }
    
    /**
     * SQL文を指定して、MimdbStatementを生成.
     * @param out MimdbStatementオブジェクトを設定します.
     * @param sql 対象のSQL文を設定します.
     * @exception Exception 例外.
     */
    public static final void createStatement( MimdbStatement out,String sql ) throws Exception {
        SqlAnalyzer.analysis( out,sql ) ;
    }
    
    /**
     * SQL文を指定して、MimdbPreparedStatementを生成.
     * @param sql 対象のSQL文を設定します.
     * @return MimdbStatement ステートメントオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbPreparedStatement createPrepared( String sql ) throws Exception {
        MimdbStatement stmt = new MimdbStatement() ;
        SqlAnalyzer.analysis( stmt,sql ) ;
        return stmt.compile() ;
    }
    
    /**
     * SQL文を指定して、MimdbPreparedStatementを生成.
     * @param out MimdbStatementオブジェクトを設定します.
     * @param sql 対象のSQL文を設定します.
     * @exception Exception 例外.
     */
    public static final void createPrepared( MimdbPreparedStatement out,String sql ) throws Exception {
        MimdbStatement stmt = new MimdbStatement() ;
        SqlAnalyzer.analysis( stmt,sql ) ;
        stmt.compile( out ) ;
    }
    
    /**
     * 情報クリア.
     */
    public void clear() {
        dbId = -1 ;
        tableName = null ;
        columns.clear() ;
        allColumn = true ;
        countFlag = false ;
        where.clear() ;
        sortList.clear() ;
        descList.clear() ;
        compile = null ;
        offset = -1 ;
        limit = -1 ;
    }
    
    /**
     * テーブル名を設定.
     * @param table 対象のテーブル名を設定します.
     * @exception Exception 例外.
     */
    public void setTable( String table ) throws Exception {
        if( table == null || ( table = table.trim().toLowerCase() ).length() <= 0 ) {
            throw new MimdbException( "テーブル名が設定されていません" ) ;
        }
        this.tableName = table ;
    }
    
    /**
     * DB更新IDを取得.
     * この情報が、結果データと一致しない場合は、その結果データは古くなっています.
     * またこのオブジェクトでは、コンパイル済みの状態になったときに、DB更新IDが確定されます.
     * それ以前の状態では[-1]が返却されます.
     * @return int DB更新IDが返却されます.
     */
    public long getDbId() {
        return dbId ;
    }
    
    /**
     * テーブル名を取得.
     * @return String テーブル名が返却されます.
     */
    public String getName() {
        return tableName ;
    }
    
    /**
     * 表示カラム名をセット.
     * @param name 対象カラム名を設定します.
     */
    public void setViewColumn( String name ) {
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new MimdbException( "指定カラム名は存在しません" ) ;
        }
        if( "*".equals( name ) ) {
            allColumn = true ;
            countFlag = false ;
        }
        else {
            columns.add( name ) ;
            allColumn = false ;
            countFlag = false ;
        }
    }
    
    /**
     * 件数取得条件をセット.
     */
    public void setViewCount() {
        countFlag = true ;
    }
    
    /**
     * 表示カラム名一覧を取得.
     * @return ObjectList<String> 表示カラム名一覧が返却されます.
     */
    public ObjectList<String> getViewColumn() {
        return columns ;
    }
    
    /**
     * 全カラムが表示対象となっているかチェック.
     * @return boolean [true]の場合、全カラムが表示対象となっています.
     */
    public boolean isAllView() {
        return allColumn ;
    }
    
    /**
     * 検索件数が表示対象となっているかチェック.
     * @return int [true]の場合、検索件数が表示対象となっています.
     */
    public boolean isCountView() {
        return countFlag ;
    }
    
    /**
     * 検索条件を設定.
     * @param search 検索条件を設定します.
     * @exception Exception 例外.
     */
    public void addWhere( MimdbSearchElement search ) throws Exception {
        if( search == null ) {
            throw new MimdbException( "検索条件は無効です" ) ;
        }
        where.add( search ) ;
    }
    
    /**
     * 検索条件を取得.
     * @return ObjectList<MimdbSearchElement> 検索条件が返却されます.
     */
    public ObjectList<MimdbSearchElement> getWhere() {
        return where ;
    }
    
    /**
     * ソート順を設定.
     * @param name ソートカラム名を設定します.
     * @param mode [true]の場合は降順(3,2,1,0)、[false]の場合は昇順(0,1,2,3)となります.
     * @exception Exception 例外.
     */
    public void addSortColumn( String name,boolean desc ) throws Exception {
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new MimdbException( "指定カラム名は存在しません" ) ;
        }
        sortList.add( name ) ;
        descList.add( desc ) ;
    }
    
    /**
     * ソート順の情報を取得.
     * @return ObjectList<String> ソート順の情報が返却されます.
     */
    public ObjectList<String> getSortColumn() {
        return sortList ;
    }
    
    /**
     * ソート順を決定させる.
     */
    public ObjectList<Boolean> getDesc() {
        return descList ;
    }
    
    /**
     * SQL文に変換.
     * @return String SQL文に変換されます.
     * @exception Exception 例外.
     */
    public String getSql() throws Exception {
        if( tableName == null ) {
            throw new MimdbException( "テーブル名が設定されていません" ) ;
        }
        StringBuilder buf = new StringBuilder( 1024 ) ;
        buf.append( "SELECT " ) ;
        if( countFlag ) {
            buf.append( "COUNT(*) " ) ;
        }
        else if( allColumn ) {
            buf.append( "* " ) ;
        }
        else {
            int len = columns.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) {
                    buf.append( ", " ) ;
                }
                buf.append( columns.get( i ) ).append( " " ) ;
            }
        }
        buf.append( "FROM " ).append( tableName ).append( " " ) ;
        if( where.size() > 0 ) {
            buf.append( "WHERE " ) ;
            int len = where.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( where.get( i ) ).append( " " ) ;
            }
        }
        if( sortList.size() > 0 ) {
            buf.append( "ORDER BY " ) ;
            int len = sortList.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( sortList.get( i ) ).append( " " ) ;
                if( descList.get( i ) ) {
                    buf.append( "DESC " ) ;
                }
                else {
                    buf.append( "ASC " ) ;
                }
            }
        }
        
        return buf.append( ";" ).toString() ;
    }
    
    /**
     * コンパイル処理.
     * @return MimdbPreparedStatement コンパイル結果が返却されます.
     * @exception Exception 例外.
     */
    public MimdbPreparedStatement compile() throws Exception {
        MimdbPreparedStatement ret = new MimdbPreparedStatement() ;
        compile( ret ) ;
        return ret ;
    }
    
    /**
     * コンパイル処理.
     * @param out コンパイル結果が返却されます.
     * @exception Exception 例外.
     */
    public void compile( MimdbPreparedStatement out ) throws Exception {
        // 前回のコンパイル結果をクリア.
        out.clear() ;
        
        // 設定データチェック.
        checkCompile() ;
        
        // テーブルオブジェクトの取得.
        MimdbTable table = MimdbTableManager.getInstance().get( tableName ) ;
        if( table == null ) {
            throw new MimdbException( "指定テーブル名[" + tableName + "]のテーブルオブジェクトは存在しません" ) ;
        }
        else if( !table.isFix() ) {
            throw new MimdbException( "指定テーブル名[" + tableName + "]のテーブルオブジェクトはFIXされていません" ) ;
        }
        
        // 表示カラムチェック.
        if( !allColumn && !countFlag ) {
            setViewColumns( out,table,columns ) ;
        }
        else if( countFlag ) {
            out.columns = null ;
            out.countFlag = true ;
        }
        else if( allColumn ) {
            out.columns = null ;
            out.countFlag = false ;
        }
        
        // ソート条件を設定.
        if( sortList.size() > 0 ) {
            setSortColumns( out,table,sortList,descList ) ;
        }
        else {
            out.sort = null ;
        }
        
        // 判別条件を設定.
        if( where.size() > 0 ) {
            setWhere( out,table,where ) ;
        }
        else {
            out.block = null ;
            out.preparendParams = null ;
        }
        out.name = table.getName() ;
        out.dbId = table.getDbId() ;
        out.src = this ;
        
        // 自DB更新IDを設定.
        this.dbId = out.dbId ;
    }
    
    /**
     * コンパイルを実施させずに、クエリー実行.
     * @return MimdbResult 実行結果が返却されます.
     * @exception Exception 例外.
     */
    public MimdbResult executeQuery() throws Exception {
        if( compile == null ) {
            MimdbPreparedStatement c = new MimdbPreparedStatement() ;
            compile( c ) ;
            if( c.preparendParamsSize > 0 ) {
                throw new MimdbException( "パラメータが必要な条件なので、この処理は実行できません" ) ;
            }
            compile = c ;
        }
        compile.setOffset( offset ) ;
        compile.setLimit( limit ) ;
        offset = -1 ;
        limit = -1 ;
        return compile.executeQuery( false ) ;
    }
    
    /**
     * オフセット、リミットのクリア.
     */
    public void clearOffset() {
        offset = -1 ;
        limit = -1 ;
    }
    
    /**
     * 表示オフセット値を設定.
     * この条件により、表示位置を確定できます.
     * @param off 表示オフセット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbStatement このオブジェクトが返却されます.
     */
    public MimdbStatement setOffset( int off ) {
        offset = off ;
        return this ;
    }
    
    /**
     * 表示リミット値を設定.
     * この条件により、表示位置を確定できます.
     * @param limit 表示リミット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbStatement このオブジェクトが返却されます.
     */
    public MimdbStatement setLimit( int limit ) {
        this.limit = limit ;
        return this ;
    }
    
    /** コンパイルデータチェック. **/
    private final void checkCompile() throws Exception {
        if( tableName == null || tableName.length() <= 0 ) {
            throw new MimdbException( "テーブル名が設定されていません" ) ;
        }
        if( !allColumn && !countFlag && columns.size() <= 0 ) {
            throw new MimdbException( "表示条件が設定されていません" ) ;
        }
    }
    
    /** 表示カラム名の設定. **/
    private static final void setViewColumns( MimdbPreparedStatement out,MimdbTable table,ObjectList<String> columns )
        throws Exception {
        int p ;
        int len = columns.size() ;
        int[] columnArray = new int[ len ] ;
        String s ;
        for( int i = 0 ; i < len ; i ++ ) {
            s = columns.get( i ) ;
            if( s == null || s.length() <= 0 ) {
                throw new MimdbException( "カラム名がNULLで設定されています" ) ;
            }
            p = table.getColumnNameByNo( s ) ;
            if( p == -1 ) {
                throw new MimdbException( "テーブル[" + table.getName() + "]に、カラム名[" + s + "]は存在しません" ) ;
            }
            columnArray[ i ] = p ;
        }
        out.columns = new EqualsNoList( columnArray ) ;
        out.countFlag = false ;
    }
    
    /** ソートカラム名の設定. **/
    private static final void setSortColumns( MimdbPreparedStatement out,MimdbTable table,ObjectList<String> sortList,ObjectList<Boolean> descList )
        throws Exception {
        int p ;
        int len = sortList.size() ;
        int[] sort = new int[ len ] ;
        boolean[] desc = new boolean[ len ] ;
        String s ;
        for( int i = 0 ; i < len ; i ++ ) {
            s = sortList.get( i ) ;
            if( s == null || s.length() <= 0 ) {
                throw new MimdbException( "カラム名がNULLで設定されています" ) ;
            }
            p = table.getColumnNameByNo( s ) ;
            if( p == -1 ) {
                throw new MimdbException( "テーブル[" + table.getName() + "]に、カラム名[" + s + "]は存在しません" ) ;
            }
            sort[ i ] = p ;
            desc[ i ] = descList.get( i ) ;
        }
        out.sort = new SortElement() ;
        out.sort.sortNoList = sort ;
        out.sort.desc = desc ;
    }
    
    /** 判別情報の設定. **/
    private static final void setWhere( MimdbPreparedStatement out,MimdbTable table,ObjectList<MimdbSearchElement> list )
        throws Exception {
        MimdbSearchElement n ;
        int len = list.size() ;
        int par = 0 ;
        int andor = -1 ;// 未定義の場合は-1.
        int p ;
        
        WhereBlock block = new WhereBlock() ;
        WhereBlock top = block ;
        ObjectList<MimdbSearchElement> params = new ObjectList<MimdbSearchElement>() ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            // 条件設定の場合.
            switch( ( n = list.get( i ) ).getType() ) {
                case MimdbSearchType.PAR_START :    // (
                    par ++ ;
                    block = block.add( new WhereBlock() ) ;
                    break ;
                case MimdbSearchType.PAR_END :      // )
                    if( ( block = block.getTop() ) == null ) {
                        throw new MimdbException( "括弧の構成が不正です" ) ;
                    }
                    par -- ;
                    break ;
                case MimdbSearchType.AND :          // and
                    if( andor != -1 ) {
                        throw new MimdbException( "and条件は不正です" ) ;
                    }
                    andor = 0 ; // andの場合は0.
                    block.add( n ) ;
                    break ;
                case MimdbSearchType.OR :           // or
                    if( andor != -1 ) {
                        throw new MimdbException( "or条件は不正です" ) ;
                    }
                    andor = 1 ; // orの場合は1.
                    block.add( n ) ;
                    break ;
                case MimdbSearchType.OFF_LIMIT :    // Offset or Limit
                    // Preparendモードの場合は、パラメータ設定.
                    if( n.isPreparend() ) {
                        params.add( n ) ;
                    }
                    // Preparendで無い場合は、オフセット、リミット値を直接セット.
                    else if( "offset".equals( n.getColumn() ) ) {
                        out.defOffset = MimdbUtils.convertInt( n.getValue() ) ;
                    }
                    else {
                        out.defLimit = MimdbUtils.convertInt( n.getValue() ) ;
                    }
                    break ;
                default :                           // where判別条件.
                    andor = -1 ;
                    // カラム名をカラム項番に変換.
                    p = n.columnNameByNo( table ) ;
                    // 対象カラムがインデックス条件で無い場合.
                    if( !table.isIndex( p ) ) {
                        throw new MimdbException( "テーブル[" + table.getName() + "]のインデックスでないカラム[" +
                            n.getColumn() + "]は、where条件に定義できません" ) ;
                    }
                    // Preparendモードの場合は、パラメータ設定.
                    if( n.isPreparend() ) {
                        params.add( n ) ;
                    }
                    block.add( n ) ;
                    break ;
            }
        }
        if( par != 0 ) {
            throw new MimdbException( "括弧の整合性が取れていません" ) ;
        }
        else if( andor != -1 ) {
            throw new MimdbException( "And,Orの条件が不正です" ) ;
        }
        out.block = top ;               // Whereブロック.
        if( params.size() > 0 ) {
            len = params.size() ;
            MimdbSearchElement[] pms = new MimdbSearchElement[ len ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                pms[ i ] = params.get( i ) ;
            }
            out.preparendParams = pms ; // Whereパラメータセット.
            out.preparendParamsSize = len ;
        }
        else {
            out.preparendParams = null ;// Whereパラメータなし.
            out.preparendParamsSize = 0 ;
        }
    }
    
}

