package org.maachang.mimdb.core.impl;

import java.util.ArrayList;
import java.util.List;

import org.maachang.mimdb.MimdbException;
import org.maachang.mimdb.core.MimdbSearchElement;
import org.maachang.mimdb.core.MimdbSearchType;

/**
 * Timestampカラム.
 * 
 * @version 2013/10/10
 * @author masahito suzuki
 * @since MasterInMemDB 1.00
 */
public class MTimestampIndex extends MLongIndex {
    
    /**
     * コンストラクタ.
     * @param id DbIdを設定します.
     * @param n カラム名を設定します.
     * @param h ハッシュインデックスを利用する場合は[true].
     */
    public MTimestampIndex( long id,String n,boolean h ) {
        this( id,n,0,h ) ;
    }
    
    /**
     * コンストラクタ.
     * @param id DbIdを設定します.
     * @param n カラム名を設定します.
     * @param size 初期サイズを設定.
     * @param h ハッシュインデックスを利用する場合は[true].
     */
    public MTimestampIndex( long id,String n,int size,boolean h ) {
        if( n == null || ( n = n.trim().toLowerCase() ).length() <= 0 ) {
            throw new MimdbException( "カラム名が設定されていません" ) ;
        }
        if( size <= 0 ) {
            size = AbstractNumberIndex.DEF_SIZE ;
        }
        dbId = id ;
        workLength = size ;
        name = n ;
        useHashFlag = h ;
    }
    
    /**
     * 情報セット.
     * @param value 対象の情報を設定します.
     * @param lineNo 行番号を設定します.
     * @exception Exception 例外.
     */
    public void add( java.sql.Time value,int lineNo )
        throws Exception {
        if( value == null ) {
            super.add( null,lineNo ) ;
        }
        else {
            super.add( MimdbUtils.convertSqlTimestamp( value ).getTime(),lineNo ) ;
        }
    }
    
    /**
     * 情報セット.
     * @param value 対象の情報を設定します.
     * @param lineNo 行番号を設定します.
     * @exception Exception 例外.
     */
    public void add( Object value,int lineNo ) throws Exception {
        if( value == null ) {
            super.add( null,lineNo ) ;
        }
        else {
            super.add( MimdbUtils.convertSqlTimestamp( value ).getTime(),lineNo ) ;
        }
    }
    
    /**
     * カラムタイプを取得.
     * @return int カラムタイプが返却されます.
     */
    public int getType() {
        return COLUMN_TIMESTAMP ;
    }
    
    /**
     * DBカラムタイプを取得.
     * @return int DBカラムタイプ(java.sql.Types)が返却されます.
     */
    public int getDBType() {
        return java.sql.Types.TIMESTAMP ;
    }
    
    /** 検索内容の整合性処理. **/
    protected MimdbSearchElement _checkSearchInfo( MimdbSearchElement info ) throws Exception {
        // IN,between検索の場合.
        if( info.getType() == MimdbSearchType.TYPE_IN ||
            info.getType() == MimdbSearchType.TYPE_BETWEEN ) {
            // リストで無い場合は、リストに変換する.
            Object o = info.getValue() ;
            if( !( o instanceof List ) ) {
                List<Long> lst = new ArrayList<Long>( 1 ) ;
                lst.add( MimdbUtils.convertSqlTimestamp( o ).getTime() ) ;
                info.setValue( lst ) ;
            }
            else {
                List n = (List)o ;
                int len = n.size() ;
                List<Long> lst = new ArrayList<Long>( len ) ;
                for( int i = 0 ; i < len ; i ++ ) {
                    lst.add( MimdbUtils.convertSqlTimestamp( n.get( i ) ).getTime() ) ;
                }
                info.setValue( lst ) ;
            }
            return info ;
        }
        // in以外.
        info.setValue( MimdbUtils.convertSqlTimestamp( info.getValue() ).getTime() ) ;
        return info ;
    }
}
