package org.maachang.mimdb.core ;

import org.maachang.mimdb.MimdbException;

/**
 * 組み込み用ステートメント.
 * 
 * @version 2014/01/16
 * @author masahito suzuki
 * @since MasterInMemDB 1.02
 */
final class EmbeddedStatement implements MimdbStatement {
    
    /** コネクションオブジェクト. **/
    private EmbeddedConnection connection = null ;
    
    /** オフセット、リミット値. **/
    private int offset = -1 ;
    private int limit = -1 ;
    
    /** コンストラクタ. **/
    protected EmbeddedStatement( EmbeddedConnection conn ) {
        connection = conn ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public void close() {
        connection = null ;
    }
    
    /**
     * オブジェクトがクローズしているか取得.
     * @return boolean [true]の場合、クローズしています.
     */
    public boolean isClose() {
        return connection == null || connection.isClose() ;
    }
    
    /** チェック処理. **/
    protected void check() {
        if( isClose() ) {
            throw new MimdbException( "オブジェクトはクローズしています" ) ;
        }
    }
    
    /**
     * 表示オフセット値を設定.
     * この条件により、表示位置を確定できます.
     * @param off 表示オフセット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbStatement このオブジェクトが返却されます.
     */
    public MimdbStatement setOffset( int off ) {
        check() ;
        this.offset = off ;
        return this ;
    }
    
    
    /**
     * 表示リミット値を設定.
     * この条件により、表示位置を確定できます.
     * @param limit 表示リミット値を設定します.
     *            [-1]が設定された場合、表示幅は確定されません.
     * @return MimdbStatement このオブジェクトが返却されます.
     */
    public MimdbStatement setLimit( int limit ) {
        check() ;
        this.limit = limit ;
        return this ;
    }
    
    /** オフセット、リミット値をクリア. **/
    protected void clearOffLimit() {
        offset = -1 ;
        limit = -1 ;
    }
    
    /**
     * フェッチサイズを設定.
     * ※この値は、サーバーモードでの接続のみ有効となります.
     * @param size フェッチサイズを設定します.
     * @return MimdbStatement このオブジェクトが返却されます.
     */
    public MimdbStatement setFetchSize( int size ) {
        check() ;
        return this ;
    }
    
    /**
     * フェッチサイズを取得.
     * ※この値は、サーバーモードでの接続のみ有効となります.
     * @return int フェッチサイズが返却されます.
     */
    public int getFetchSize() {
        check() ;
        return -1 ;
    }
    
    /**
     * クエリー実行.
     * @param sql 対象のSQL文を設定します.
     * @return MimdbResult 実行結果が返却されます.
     * @exception Exception 例外.
     */
    public MimdbResult executeQuery( String sql )
        throws Exception {
        MimdbQuery query = MimdbQuery.create( sql ) ;
        query.setOffset( offset ) ;
        query.setLimit( limit ) ;
        offset = -1 ;
        limit = -1 ;
        MimdbResult ret = query.executeQuery() ;
        query.clear() ;
        return ret ;
    }
}
