package org.maachang.mimdb.core;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Remote用テーブル管理.
 * サーバー接続モードでのクライアントが認識している
 * テーブル情報群.
 * 
 * @version 2014/01/17
 * @author masahito suzuki
 * @since MasterInMemDB 1.02
 */
public class RemoteTableManager implements TableManager {
    /** 自オブジェクト. **/
    private static final RemoteTableManager SNGL = new RemoteTableManager() ;
    
    /** メモリーテーブル情報. **/
    private final Map<String,RemoteTable> tables = new ConcurrentHashMap<String,RemoteTable>() ;
    
    /**
     * コンストラクタ.
     */
    private RemoteTableManager() {
        
    }
    
    /**
     * オブジェクトを取得.
     * @return RemoteTableManager オブジェクトが返却されます.
     */
    public static final RemoteTableManager getInstance() {
        return SNGL ;
    }
    
    /**
     * オブジェクトを破棄.
     */
    public void destroy() {
        tables.clear() ;
    }
    
    /** テーブル名を取得 **/
    private static final String tableNameNoError( String name ) {
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        return name ;
    }
    
    /**
     * テーブルが登録されているかチェック.
     * @param name 対象のテーブルを設定します.
     * @return boolean [true]の場合、登録されています.
     */
    public boolean isTable( String name ) {
        if( ( name = tableNameNoError( name ) ) == null ) {
            return false ;
        }
        return tables.containsKey( name ) ;
    }
    
    /**
     * テーブル登録.
     * @param table 登録対象のテーブルを設定します.
     */
    public void put( RemoteTable table ) {
        if( table == null ) {
            throw new IllegalArgumentException( "テーブルは存在しません" ) ;
        }
        String name = table.name ;
        tables.put( name,table ) ;
    }
    
    /**
     * 登録テーブル情報の取得.
     * @param name テーブル名を設定します.
     * @return BaseTable テーブル情報が返されます.
     */
    public BaseTable get( String name ) {
        return tables.get( name ) ;
    }
    
    /**
     * 登録テーブル情報を削除.
     * @param name テーブル名を設定します.
     * @return boolean [true]の場合、削除できました.
     */
    public boolean remove( String name ) {
        tables.remove( name ) ;
        return true ;
    }
    
    /**
     * 登録テーブル数を取得.
     * @return int 登録されているテーブル数が返却されます.
     */
    public int size() {
        return tables.size() ;
    }
    
    /**
     * 登録テーブル一覧を取得.
     * @return String[] 登録されているテーブル名一覧が返却されます.
     */
    public String[] getTables() {
        int len = tables.size() ;
        if( len == 0 ) {
            return null ;
        }
        int cnt = 0 ;
        String[] ret = new String[ len ] ;
        Iterator<String> n = tables.keySet().iterator() ;
        if( n.hasNext() ) {
            ret[ cnt ++ ] = n.next() ;
        }
        return ret ;
    }
    
}

