package org.maachang.mimdb.core ;


/**
 * テーブル行データ.
 * 
 * @version 2013/10/14
 * @author masahito suzuki
 * @since MasterInMemDB 1.00
 */
public abstract class MimdbRow implements MimdbBase {
    /** 更新ID. **/
    protected long dbId ;
    
    /** テーブルデータ. **/
    protected MimdbTable baseTable ;
    
    /** 行番号. **/
    protected int lineNo ;
    
    /**
     * 更新IDを取得.
     * @return long 更新IDが返却されます.
     */
    public long getDbId() {
        return dbId ;
    }
    
    /**
     * テーブル名の取得.
     * @return String テーブル名が返却されます.
     */
    public String getTableName() {
        return baseTable.name ;
    }
    
    /**
     * テーブル情報の取得.
     * @return MimdbTable テーブル情報が返却されます.
     */
    public MimdbTable getTable() {
        return baseTable ;
    }
    
    /**
     * 対象データの行番号を取得.
     * @return int 行番号が返却されます.
     */
    public int getLineNo() {
        return lineNo ;
    }
    
    /**
     * カラム数の取得.
     * @return int カラム数が返却されます.
     */
    public int getColumnSize() {
        return baseTable.columnLength ;
    }
    
    /**
     * カラム名を取得.
     * @param no カラム項番を設定します.
     * @return String カラム名が返却されます.
     */
    public String getColumnName( int no ) {
        return baseTable.getColumnName( no ) ;
    }
    
    /**
     * 指定カラム名に対する項番を取得.
     * @param name カラム名を設定します.
     * @return int カラム項番が返却されます.
     *             [-1]の場合は対象カラム名は存在しません.
     */
    public int getColumnNameByNo( String name ) {
        return baseTable.getColumnNameByNo( name ) ;
    }
    
    /**
     * カラムタイプを取得.
     * @param no カラム項番を設定します.
     * @return int カラムタイプが返却されます.
     * java.lang.Boolean  : MimdbIndex.COLUMN_BOOL
     * java.lang.Integer  : MimdbIndex.COLUMN_INT
     * java.lang.Long     : MimdbIndex.COLUMN_LONG
     * java.lang.Double   : MimdbIndex.COLUMN_FLOAT
     * java.lang.String   : MimdbIndex.COLUMN_STRING
     * java.sql.Date      : MimdbIndex.COLUMN_DATE
     * java.sql.Time      : MimdbIndex.COLUMN_TIME
     * java.sql.Timestamp : MimdbIndex.COLUMN_TIMESTAMP
     * java.util.Date     : MimdbIndex.COLUMN_DATE
     */
    public int getColumnType( int no ) {
        return baseTable.getColumnType( no ) ;
    }
    
    /**
     * カラムタイプを取得.
     * @param name 対象のカラム名を設定します.
     * @return int カラムタイプが返却されます.
     *             -1が返却された場合は、対象カラムは存在しません.
     * java.lang.Boolean  : MimdbIndex.COLUMN_BOOL
     * java.lang.Integer  : MimdbIndex.COLUMN_INT
     * java.lang.Long     : MimdbIndex.COLUMN_LONG
     * java.lang.Double   : MimdbIndex.COLUMN_FLOAT
     * java.lang.String   : MimdbIndex.COLUMN_STRING
     * java.sql.Date      : MimdbIndex.COLUMN_DATE
     * java.sql.Time      : MimdbIndex.COLUMN_TIME
     * java.sql.Timestamp : MimdbIndex.COLUMN_TIMESTAMP
     * java.util.Date     : MimdbIndex.COLUMN_DATE
     */
    public int getColumnType( String name ) {
        return baseTable.getColumnType( name ) ;
    }
    
    /**
     * 指定項番のカラムがインデックスかチェック.
     * @param no 対象のカラム項番を設定します.
     * @return boolean [true]の場合は、インデックスです.
     */
    public boolean isIndex( int no ) {
        return baseTable.isIndex( no ) ;
    }
    
    /**
     * 指定項番のカラムがインデックスかチェック.
     * @param name 対象のカラム名を設定します.
     * @return boolean [true]の場合は、インデックスです.
     */
    public boolean isIndex( String name ) {
        return baseTable.isIndex( name ) ;
    }
    
    /**
     * 主キーカラム名を取得.
     * @return String 主キーカラム名が返却されます.
     *                [null]が返却された場合、主キーは存在しません.
     */
    public String getPrimaryIndexKey() {
        return baseTable.getPrimaryIndexKey() ;
    }
    
    /**
     * 主キーが有効かチェック.
     * @return boolean [true]の場合、主キーは有効です.
     */
    public boolean isPrimaryIndexKey() {
        return baseTable.isPrimaryIndexKey() ;
    }
    
    /**
     * ソート番号を取得.
     * @param no 対象のカラム項番を設定します.
     * @return int ソート番号が返却されます.
     */
    public int getSortNo( int no ) {
        MimdbIndex idx = baseTable.getIndex( no ) ;
        if( idx != null ) {
            return idx.getSortNoList()[ lineNo ] ;
        }
        return -1 ;
    }
    
    /**
     * ソート番号を取得.
     * @return String 主キーカラム名が返却されます.
     * @return int ソート番号が返却されます.
     */
    public int getSortNo( String name ) {
        MimdbIndex idx = baseTable.getIndex( name ) ;
        if( idx != null ) {
            return idx.getSortNoList()[ lineNo ] ;
        }
        return -1 ;
    }
    
    /**
     * カラム項番で情報を取得.
     * @param n 対象の項番を設定します.
     * @return Object 情報が返却されます.
     */
    public abstract Object getValue( int n ) ;
    
    /**
     * カラム名で情報を取得.
     * @param name 対象の名前を設定します.
     * @return Object 情報が返却されます.
     */
    public abstract Object getValue( String name ) ;
    
    /**
     * プライマリーキーオブジェクトを取得.
     * @return Object プライマリーキーオブジェクトが返却されます.
     */
    public abstract Object getPrimaryKey() ;
    
    /**
     * カラムオブジェクト一覧を取得.
     * @return Object[] カラムオブジェクト一覧が返却されます.
     */
    public abstract Object[] getValues() ;
    
    /**
     * 圧縮長を取得.
     * @return int 圧縮されたバイナリ長が返却されます.
     *             圧縮オブジェクトで無い場合は、ゼロが返却されます.
     */
    public abstract int compressLength() ;
}
