package org.seasar.dao.impl;

import java.lang.reflect.Field;
import java.sql.DatabaseMetaData;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.seasar.dao.BeanMetaData;
import org.seasar.dao.RelationPropertyType;
import org.seasar.extension.jdbc.ColumnNotFoundRuntimeException;
import org.seasar.extension.jdbc.PropertyType;
import org.seasar.extension.jdbc.ValueType;
import org.seasar.extension.jdbc.impl.PropertyTypeImpl;
import org.seasar.extension.jdbc.types.ValueTypes;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.PropertyDesc;
import org.seasar.framework.beans.PropertyNotFoundRuntimeException;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.util.CaseInsensitiveMap;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.DatabaseMetaDataUtil;
import org.seasar.framework.util.FieldUtil;

/**
 * @author higa
 *  
 */
public class BeanMetaDataImpl implements BeanMetaData {

	private Class beanClass_;

	private String tableName_;

	private CaseInsensitiveMap propertyTypes_ = new CaseInsensitiveMap();

	private Map propertyTypesByColumnName_ = new CaseInsensitiveMap();

	private List relationPropertyTypes_ = new ArrayList();
	
	private boolean hasDatabaseMetaData_ = false;
	
	private boolean persistent_ = true;
	
	private String[] primaryKeys_ = new String[0];
	
	private String autoInsertSql_;
	
	private String autoUpdateSql_;
	
	private String autoDeleteSql_;
	
	private String autoSelectList_;

	public BeanMetaDataImpl(Class beanClass) {
		beanClass_ = beanClass;
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(beanClass);
		setupTableName(beanDesc);
		setupPropertyType(beanDesc);
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getBeanClass()
	 */
	public Class getBeanClass() {
		return beanClass_;
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getTableName()
	 */
	public String getTableName() {
		return tableName_;
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getPropertyTypeSize()
	 */
	public int getPropertyTypeSize() {
		return propertyTypes_.size();
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getPropertyType(int)
	 */
	public PropertyType getPropertyType(int index) {
		return (PropertyType) propertyTypes_.get(index);
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getPropertyType(java.lang.String)
	 */
	public PropertyType getPropertyType(String propertyName)
			throws PropertyNotFoundRuntimeException {

		PropertyType propertyType = (PropertyType) propertyTypes_
				.get(propertyName);
		if (propertyType == null) {
			throw new PropertyNotFoundRuntimeException(beanClass_, propertyName);
		}
		return propertyType;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getVersionNoPropertyType()
	 */
	public PropertyType getVersionNoPropertyType()
			throws PropertyNotFoundRuntimeException {
		
		return getPropertyType(VERSION_NO_PROPERTY_NAME);
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getPropertyTypeByColumnName(java.lang.String)
	 */
	public PropertyType getPropertyTypeByColumnName(String columnName)
			throws ColumnNotFoundRuntimeException {

		PropertyType propertyType = (PropertyType) propertyTypesByColumnName_
				.get(columnName);
		if (propertyType == null) {
			throw new ColumnNotFoundRuntimeException(tableName_, columnName);
		}
		return propertyType;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#hasPropertyTypeByColumnName(java.lang.String)
	 */
	public boolean hasPropertyTypeByColumnName(String columnName) {
		return propertyTypesByColumnName_.get(columnName) != null;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#hasPropertyType(java.lang.String)
	 */
	public boolean hasPropertyType(String propertyName) {
		return propertyTypes_.get(propertyName) != null;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#hasVersionNoPropertyType()
	 */
	public boolean hasVersionNoPropertyType() {
		return hasPropertyType(VERSION_NO_PROPERTY_NAME);
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#convertFullColumnName(java.lang.String)
	 */
	public String convertFullColumnName(String alias) {
		if (hasPropertyTypeByColumnName(alias)) {
			return tableName_ + "." + alias;
		}
		int index = alias.lastIndexOf('_');
		if (index < 0) {
			throw new ColumnNotFoundRuntimeException(tableName_, alias);
		}
		String columnName = alias.substring(0, index);
		String relnoStr = alias.substring(index + 1);
		int relno = -1;
		try {
			relno = Integer.parseInt(relnoStr);
		} catch (Throwable t) {
			throw new ColumnNotFoundRuntimeException(tableName_, alias);
		}
		RelationPropertyType rpt = getRelationPropertyType(relno);
		if (!rpt.getBeanMetaData().hasPropertyTypeByColumnName(columnName)) {
			throw new ColumnNotFoundRuntimeException(tableName_, alias);
		}
		return rpt.getPropertyName() + "." + columnName;
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getRelationPropertyTypeSize()
	 */
	public int getRelationPropertyTypeSize() {
		return relationPropertyTypes_.size();
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getRelationPropertyType(int)
	 */
	public RelationPropertyType getRelationPropertyType(int index) {
		return (RelationPropertyType) relationPropertyTypes_.get(index);
	}

	/**
	 * @see org.seasar.dao.BeanMetaData#getRelationPropertyType(java.lang.String)
	 */
	public RelationPropertyType getRelationPropertyType(String propertyName)
			throws PropertyNotFoundRuntimeException {

		for (int i = 0; i < getRelationPropertyTypeSize(); i++) {
			RelationPropertyType rpt = (RelationPropertyType) relationPropertyTypes_
					.get(i);
			if (rpt != null
					&& rpt.getPropertyName().equalsIgnoreCase(propertyName)) {
				return rpt;
			}
		}
		throw new PropertyNotFoundRuntimeException(beanClass_, propertyName);
	}

	private void setupTableName(BeanDesc beanDesc) {
		if (beanDesc.hasField(TABLE_KEY)) {
			Field field = beanDesc.getField(TABLE_KEY);
			tableName_ = (String) FieldUtil.get(field, null);
		} else {
			tableName_ = ClassUtil.getShortClassName(beanClass_);
		}
	}

	private void setupPropertyType(BeanDesc beanDesc) {
		for (int i = 0; i < beanDesc.getPropertyDescSize(); ++i) {
			PropertyDesc pd = beanDesc.getPropertyDesc(i);
			String relnoKey = pd.getPropertyName() + RELNO_KEY_SUFFIX;
			if (beanDesc.hasField(relnoKey)) {
				RelationPropertyType rpt = createRelationPropertyType(beanDesc,
						pd, relnoKey);
				addRelationPropertyType(rpt);
			} else {
				PropertyType pt = createPropertyType(beanDesc, pd);
				addPropertyType(pt);
			}
		}
	}

	private RelationPropertyType createRelationPropertyType(BeanDesc beanDesc,
			PropertyDesc propertyDesc, String relnoKey) {

		Field field = beanDesc.getField(relnoKey);
		String[] myKeys = new String[0];
		String[] yourKeys = new String[0];
		int relno = FieldUtil.getInt(field, null);
		String relkeysKey = propertyDesc.getPropertyName() + RELKEYS_KEY_SUFFIX;
		if (beanDesc.hasField(relkeysKey)) {
			Field field2 = beanDesc.getField(relkeysKey);
			String relkeys = (String) FieldUtil.get(field2, null);
			StringTokenizer st = new StringTokenizer(relkeys, " \t\n\r\f,");
			List myKeyList = new ArrayList();
			List yourKeyList = new ArrayList();
			while (st.hasMoreTokens()) {
				String token = st.nextToken();
				int index = token.indexOf(':');
				if (index > 0) {
					myKeyList.add(token.substring(0, index));
					yourKeyList.add(token.substring(index + 1));
				} else {
					myKeyList.add(token);
					yourKeyList.add(token);
				}
			}
			myKeys = (String[]) myKeyList.toArray(new String[myKeyList
					.size()]);
			yourKeys = (String[]) yourKeyList
					.toArray(new String[yourKeyList.size()]);
		}
		RelationPropertyType rpt = new RelationPropertyTypeImpl(propertyDesc,
				relno, myKeys, yourKeys);
		return rpt;
	}

	private void addRelationPropertyType(RelationPropertyType rpt) {
		for (int i = relationPropertyTypes_.size(); i <= rpt.getRelationNo(); ++i) {
			relationPropertyTypes_.add(null);
		}
		relationPropertyTypes_.set(rpt.getRelationNo(), rpt);
	}

	private PropertyType createPropertyType(BeanDesc beanDesc,
			PropertyDesc propertyDesc) {

		String columnNameKey = propertyDesc.getPropertyName()
				+ COLUMN_KEY_SUFFIX;
		String columnName = propertyDesc.getPropertyName();
		if (beanDesc.hasField(columnNameKey)) {
			Field field = beanDesc.getField(columnNameKey);
			columnName = (String) FieldUtil.get(field, null);
		}
		ValueType valueType = ValueTypes.getValueType(propertyDesc
				.getPropertyType());
		PropertyType pt = new PropertyTypeImpl(propertyDesc, valueType,
				columnName);
		return pt;
	}

	private void addPropertyType(PropertyType propertyType) {
		propertyTypes_.put(propertyType.getPropertyName(), propertyType);
		propertyTypesByColumnName_.put(propertyType.getColumnName(),
				propertyType);
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#isPersistent()
	 */
	public boolean isPersistent() {
		return persistent_;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#setupDatabaseMetaData(java.sql.DatabaseMetaData)
	 */
	public synchronized boolean setupDatabaseMetaData(DatabaseMetaData dbMetaData) {
		if (hasDatabaseMetaData_) {
			return false;
		}
		Set primaryKeySet = DatabaseMetaDataUtil.getPrimaryKeySet(dbMetaData,
				tableName_);
		Set columnSet = DatabaseMetaDataUtil.getColumnSet(dbMetaData,
				tableName_);
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (primaryKeySet.contains(pt.getColumnName())) {
				pt.setPrimaryKey(true);
			} else {
				pt.setPrimaryKey(false);
			}
			if (columnSet.contains(pt.getColumnName())) {
				pt.setPersistent(true);
			} else {
				pt.setPersistent(false);
			}
		}
		persistent_ = columnSet.size() > 0;
		primaryKeys_ = (String[]) primaryKeySet.toArray(
				new String[primaryKeySet.size()]);
		setupAutoInsertSql();
		setupAutoUpdateSql();
		setupAutoDeleteSql();
		hasDatabaseMetaData_ = true;
		for (int i = 0; i < getRelationPropertyTypeSize(); ++i) {
			RelationPropertyType rpt = getRelationPropertyType(i);
			BeanMetaData bmd = rpt.getBeanMetaData();
			bmd.setupDatabaseMetaData(dbMetaData);
			rpt.setPersistent(bmd.isPersistent());
		}
		return true;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getPrimaryKeySize()
	 */
	public int getPrimaryKeySize() {
		return primaryKeys_.length;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getPrimaryKey(int)
	 */
	public String getPrimaryKey(int index) {
		return primaryKeys_[index];
	}
	
	private void setupAutoInsertSql() {
		StringBuffer buf = new StringBuffer(100);
		buf.append("INSERT INTO ");
		buf.append(getTableName());
		buf.append("(");
		int persistentColumnCount = 0;
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (pt.isPersistent()) {
				++persistentColumnCount;
				buf.append(pt.getColumnName());
				buf.append(", ");
			}
		}
		buf.setLength(buf.length() - 2);
		buf.append(") VALUES(");
		for (int i = 0; i < persistentColumnCount; ++i) {
			buf.append("?, ");
		}
		buf.setLength(buf.length() - 2);
		buf.append(")");
		autoInsertSql_ = buf.toString();
	}
	
	protected void setupAutoUpdateSql() {
		StringBuffer buf = new StringBuffer(100);
		buf.append("UPDATE ");
		buf.append(getTableName());
		buf.append(" SET ");
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (pt.isPersistent() && !pt.isPrimaryKey()) {
				if (pt.getPropertyName().equals(VERSION_NO_PROPERTY_NAME)) {
					buf.append(pt.getColumnName());
					buf.append(" = ");
					buf.append(pt.getColumnName());
					buf.append(" + 1, ");
				} else {
					buf.append(pt.getColumnName());
					buf.append(" = ?, ");
				}
			}
		}
		buf.setLength(buf.length() - 2);
		setupAutoUpdateWhere(buf);
		autoUpdateSql_ = buf.toString();
	}
	
	protected void setupAutoDeleteSql() {
		StringBuffer buf = new StringBuffer(100);
		buf.append("DELETE FROM ");
		buf.append(getTableName());
		setupAutoUpdateWhere(buf);
		autoDeleteSql_ = buf.toString();
	}
	
	protected void setupAutoUpdateWhere(StringBuffer buf) {
		buf.append(" WHERE ");
		for (int i = 0; i < getPrimaryKeySize(); ++i) {
			buf.append(getPrimaryKey(i));
			buf.append(" = ? AND ");
		}
		buf.setLength(buf.length() - 5);
		if (hasVersionNoPropertyType()) {
			PropertyType pt = getVersionNoPropertyType();
			buf.append(" AND ");
			buf.append(pt.getColumnName());
			buf.append(" = ?");
		}
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoInsertSql()
	 */
	public String getAutoInsertSql() {
		return autoInsertSql_;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoUpdateSql()
	 */
	public String getAutoUpdateSql() {
		return autoUpdateSql_;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoDeleteSql()
	 */
	public String getAutoDeleteSql() {
		return autoDeleteSql_;
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoSelectList()
	 */
	public String getAutoSelectList() {
		if (autoSelectList_ != null) {
			return autoSelectList_;
		}
		setupAutoSelectList();
		return autoSelectList_;
	}
	
	private synchronized void setupAutoSelectList() {
		if (autoSelectList_ != null) {
			return;
		}
		StringBuffer buf = new StringBuffer(100);
		buf.append("SELECT ");
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (pt.isPersistent()) {
				buf.append(tableName_);
				buf.append(".");
				buf.append(pt.getColumnName());
				buf.append(", ");
			}
		}
		for (int i = 0; i < getRelationPropertyTypeSize(); ++i) {
			RelationPropertyType rpt = getRelationPropertyType(i);
			BeanMetaData bmd = rpt.getBeanMetaData();
			for (int j = 0; j < bmd.getPropertyTypeSize(); ++j) {
				PropertyType pt = bmd.getPropertyType(j);
				String columnName = pt.getColumnName(); 
				if (rpt.isYourKey(columnName)) {
					continue;
				}
				buf.append(rpt.getPropertyName());
				buf.append(".");
				buf.append(columnName);
				buf.append(" AS ");
				buf.append(pt.getColumnName()).append("_").append(rpt.getRelationNo());
				buf.append(", ");
			}
		}
		buf.setLength(buf.length() - 2);
		autoSelectList_ = buf.toString();
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoInsertBindVariables(java.lang.Object)
	 */
	public Object[] getAutoInsertBindVariables(Object bean) {
		List variables = new ArrayList();
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (pt.isPersistent()) {
				variables.add(pt.getPropertyDesc().getValue(bean));
			}
		}
		return variables.toArray();
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoUpdateBindVariables(java.lang.Object)
	 */
	public Object[] getAutoUpdateBindVariables(Object bean) {
		List bindVariables = new ArrayList();
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (pt.isPersistent() && !pt.isPrimaryKey()) {
				if (!pt.getPropertyName().equals(VERSION_NO_PROPERTY_NAME)) {
					bindVariables.add(pt.getPropertyDesc().getValue(bean));
				}
			}
		}
		addAutoUpdateWhereBindVariables(bindVariables, bean);
		return bindVariables.toArray();
	}
	
	/**
	 * @see org.seasar.dao.BeanMetaData#getAutoDeleteBindVariables(java.lang.Object)
	 */
	public Object[] getAutoDeleteBindVariables(Object bean) {
		List bindVariables = new ArrayList();
		addAutoUpdateWhereBindVariables(bindVariables, bean);
		return bindVariables.toArray();
	}
	
	private void addAutoUpdateWhereBindVariables(List bindVariables, Object bean) {
		for (int i = 0; i < getPrimaryKeySize(); ++i) {
			PropertyType pt = getPropertyTypeByColumnName(getPrimaryKey(i));
			bindVariables.add(pt.getPropertyDesc().getValue(bean));
		}
		if (hasVersionNoPropertyType()) {
			PropertyType pt = getVersionNoPropertyType();
			bindVariables.add(pt.getPropertyDesc().getValue(bean));
		}
	}
}