package org.seasar.nazuna.amf;

import java.io.DataInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.seasar.log.Logger;
import org.seasar.util.EMap;
import org.seasar.util.PropertyDesc;
import org.seasar.util.Reflector;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

public final class AMFReader {

	private static Logger _logger = Logger.getLogger(AMFReader.class);
	private DataInputStream _inputStream;
	private AMFMessage _message = new AMFMessage();
	private ArrayList _sharedObjects;

	public AMFReader(DataInputStream inputStream) {
		_inputStream = inputStream;
	}

	public final AMFMessage read() throws IOException {
		skipHeaders();
		readBodies();
		return _message;
	}

	public final void skipHeaders() throws IOException {
		_inputStream.readUnsignedShort();
		int headerCount = _inputStream.readUnsignedShort();
		for (int i = 0; i < headerCount; ++i) {
			_inputStream.readUTF();
			readBoolean();
			_inputStream.readInt();
			readData();
		}
	}

	public final void readBodies() throws IOException {
		int bodyCount = _inputStream.readUnsignedShort();
		for (int i = 0; i < bodyCount; ++i) {
			_sharedObjects = new ArrayList();
			String target = _inputStream.readUTF();
			String response = _inputStream.readUTF();
			int length = _inputStream.readInt();
			Object data = readData();
			_message.addBody(target, response, data);
		}
	}
	
	public final Object readData() throws IOException {
		byte dataType = _inputStream.readByte();
		return readData(dataType);
	}
	
	public final Object readData(byte dataType) throws IOException {
		switch (dataType) {
			case AMFDataType.NUMBER :
				double d = _inputStream.readDouble();
				_logger.debug("readNumber:" + d);
				return new Double(d);
			case AMFDataType.BOOLEAN :
				Boolean b = readBoolean();
				_logger.debug("readBoolean:" + b);
				return b;
			case AMFDataType.STRING :
				String s = _inputStream.readUTF();
				_logger.debug("readString:" + s);
				return s;
			case AMFDataType.OBJECT :
				return readMap();
			case AMFDataType.NULL :
				_logger.debug("readNull:");
				return null;
			case AMFDataType.FLASHED_SHARED_OBJECT :
				return readFlashedSharedObject();
			case AMFDataType.ARRAY_SHARED_OBJECT :
				_logger.debug("readArraySharedObject:");
				return null;
			case AMFDataType.ARRAY :
				return readArray();
			case AMFDataType.DATE :
				Date date = readDate();
				_logger.debug("readDate:" + date);
				return date;
			case AMFDataType.AS_OBJECT :
				_logger.debug("readASObject:");
				return null;
			case AMFDataType.XML :
				_logger.debug("readXML:");
				return readXML();
			case AMFDataType.CUSTOM_CLASS :
				return readCustomClass();
			default :
				return null;
		}
	}

	public final Object readCustomClass() throws IOException {
		String type = _inputStream.readUTF();
		_logger.debug("readCustomClass:" + type);
		Class clazz = Reflector.getClass(type);
		Object target = Reflector.newInstance(clazz);
		addSharedObject(target);
		EMap propertyDescMap = Reflector.getPropertyDescMap(clazz);
		String propertyName = _inputStream.readUTF();
		for (byte dataType = _inputStream.readByte(); dataType != 9;
				dataType = _inputStream.readByte()) {

			Object value = readData(dataType);
			_logger.debug("propetyName=" + propertyName + ",value=" + value);
			PropertyDesc propertyDesc = (PropertyDesc) propertyDescMap.get(propertyName);
			value = AMFUtil.adjustValue(value, propertyDesc.getPropertyType());
			if (propertyDesc.getWriteMethod() != null) {
				propertyDesc.setValue(target, value);
			}
			propertyName = _inputStream.readUTF();
		}
		return target;
	}

	public final Map readMap() throws IOException {
		_logger.debug("readMap:");
		Map map = new HashMap();
		addSharedObject(map);
		String key = _inputStream.readUTF();
		for (byte dataType = _inputStream.readByte(); dataType != 9;
				dataType = _inputStream.readByte()) {

			Object value = readData(dataType);
			_logger.debug("key=" + key + ",value=" + value);
			map.put(key, value);
			key = _inputStream.readUTF();
		}
		return map;
	}

	public final ArrayList readArray() throws IOException {
		ArrayList array = new ArrayList();
		addSharedObject(array);
		int length = _inputStream.readInt();
		_logger.debug("readArray:size=" + length);
		for (int i = 0; i < length; i++) {
			Object item = readData();
			_logger.debug("item(" + i + ")=" + item);
			array.add(item);
		}
		return array;
	}

	public final Boolean readBoolean() throws IOException {
		byte value = _inputStream.readByte();
		if (value == 1) {
			return Boolean.TRUE;
		} else {
			return Boolean.FALSE;
		}
	}

	public final Date readDate() throws IOException {
		double ms = _inputStream.readDouble();
		int offset = _inputStream.readUnsignedShort() * AMFConstants.MILLS_PER_HOUR;
		int defaultOffset = TimeZone.getDefault().getRawOffset();
		ms += (double) defaultOffset - offset;
		Date date = new Date((long) ms);
		return date;
	}

	public final Document readXML() throws IOException {
		_inputStream.skip(4);
		try {
            return DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(_inputStream);
        } catch (SAXException ex) {
            throw convertIOException(ex);
        } catch (ParserConfigurationException ex) {
            throw convertIOException(ex);
		}
	}

	public final Object readFlashedSharedObject() throws IOException {
		int index = _inputStream.readUnsignedShort();
		Object target = _sharedObjects.get(index);
		_logger.debug("readFlashedSharedObject:index=" + index + ",value=" + target);
		return target;
	}
	
	private void addSharedObject(Object o) {
		_logger.debug("addSharedObject:index=" + _sharedObjects.size() + ",value=" + o);
		_sharedObjects.add(o);
	}
	
	private IOException convertIOException(Exception ex) {
		IOException ioe = new IOException(ex.getMessage());
        ioe.initCause(ex);
        return ioe;
	}
}