package org.seasar.system;

import java.lang.reflect.Proxy;
import java.util.Map;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.seasar.util.EMap;
import org.seasar.util.SeasarRuntimeException;

public final class JMXUtil {

    private static Map _nameMap = new EMap();
    private static Map _mbeanInfoMap = new EMap();

    private JMXUtil() {
    }

    public static ObjectName getObjectName(final String name) {
        ObjectName oname = (ObjectName) _nameMap.get(name);
        if (oname != null) {
            return oname;
        }
        synchronized (_nameMap) {
        	oname = (ObjectName) _nameMap.get(name);
            if (oname == null) {
                try {
                    oname = new ObjectName(name);
                } catch (MalformedObjectNameException ex) {
                    throw new SeasarRuntimeException("ESSR0017", new Object[]{ex}, ex);
                }
                _nameMap.put(name, oname);
            }
        }
        return oname;
    }

	public static String getMBeanName(String name) {
    	return "MBean:name=" + name;
    }
    
    public static String getJNDIName(String name) {
    	return "mbean/" + name;
    }
    
    public static ObjectName getMBeanObjectName(String name) {
    	return getObjectName(getMBeanName(name));
    }
    
    public static Class getMBeanInterface(Class mbeanClass) {
    	Class concrete = mbeanClass;
    	Class mbeanInterface = null;
      	while (null != concrete) {
        	mbeanInterface = getMBeanInterface(concrete, concrete.getInterfaces());
         	if (null != mbeanInterface) {
            	return mbeanInterface;
         	}
         	concrete = concrete.getSuperclass();
      	}
      	return null;
   	}

   	private static Class getMBeanInterface(Class concrete, Class[] interfaces) {
    	if (Proxy.isProxyClass(concrete)) {
    		return getMBeanInterfaceForProxy(concrete, interfaces);
    	} else {
    		return getMBeanInterfaceForStandartMBean(concrete, interfaces);
    	}
   	}
   	
   	private static Class getMBeanInterfaceForStandartMBean(Class concrete, Class[] interfaces) {
    	String name = concrete.getName() + "MBean";
      	for (int i = 0; i < interfaces.length; ++i) {
        	if (interfaces[i].getName().equals(name)) {
            	return interfaces[i];
         	}
      	}
      	return null;
   	}
   	
   	private static Class getMBeanInterfaceForProxy(Class concrete, Class[] interfaces) {
      	for (int i = 0; i < interfaces.length; ++i) {
        	if (interfaces[i].getName().endsWith("MBean")) {
            	return interfaces[i];
         	}
      	}
      	return null;
   	}
}