package org.seasar.util;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Array;
import java.util.AbstractCollection;
import java.util.AbstractList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;


public class EArrayList extends AbstractCollection
		implements List, Cloneable, Externalizable {

	static final long serialVersionUID = 1L;
    private transient Object _elements[];
    private transient int _size = 0;

    public EArrayList() {
        this(10);
    }

    public EArrayList(final int initialCapacity) {
        if (initialCapacity < 0) {
            throw new IllegalArgumentException("Illegal Capacity: " +
                    initialCapacity);
        }
        _elements = new Object[initialCapacity];
    }

    public EArrayList(final Collection c) {
        _size = c.size();
        _elements = new Object[(_size * 110) / 100];
        c.toArray(_elements);
    }

    public EArrayList(final Object[] array) {
        addAll(array);
    }

    public final void trimToSize() {
        Object oldElements[] = _elements;
        _elements = new Object[_size];
        System.arraycopy(oldElements, 0, _elements, 0, _size);
    }

    public final int getCapacity() {
        return _elements.length;
    }

    public final int size() {
        return _size;
    }

    public final boolean isEmpty() {
        return _size == 0;
    }

    public final boolean contains(final Object element) {
        return indexOf(element) >= 0;
    }

    public final boolean containsAll(Collection c) {
        for (Iterator i = c.iterator(); i.hasNext(); ) {
            if (!contains(i.next())) {
                return false;
            }
        }
        return true;
    }

    public final int indexOf(final Object element) {
        if (element == null) {
            for (int i = 0; i < _size; i++) {
                if (_elements[i] == null) {
                    return i;
                }
            }
        } else {
            for (int i = 0; i < _size; i++) {
                if (element.equals(_elements[i])) {
                    return i;
                }
            }
        }
        return -1;
    }

    public final int lastIndexOf(final Object element) {
        if (element == null) {
            for (int i = _size - 1; i >= 0; i--) {
                if (_elements[i] == null) {
                    return i;
                }
            }
        } else {
            for (int i = _size - 1; i >= 0; i--) {
                if (element.equals(_elements[i])) {
                    return i;
                }
            }
        }
        return -1;
    }

    public final Object clone() {
        EArrayList copy = new EArrayList((_size * 110) / 100);
        System.arraycopy(_elements, 0, copy._elements, 0, _size);
        copy._size = _size;
        return copy;
    }

    public final Object[] toArray() {
        Object[] array = new Object[_size];
        System.arraycopy(_elements, 0, array, 0, _size);
        return array;
    }

    public final Object[] toArray(final Object proto[]) {
        Object[] array = proto;
        if (proto.length < _size) {
            array = (Object[]) Array.newInstance(
                    proto.getClass().getComponentType(), _size);
        }
        System.arraycopy(_elements, 0, array, 0, _size);
        if (array.length > _size) {
            array[_size] = null;
        }
        return array;
    }

    public final Object get(final int index) {
        checkRange(index);
        return _elements[index];
    }

    public final Object set(final int index, final Object element) {
        final int n = index - _size;
        if (n >= 0) {
            for (int i = 0; i <= n; i++) {
                add(null);
            }
        }
        Object oldElement = _elements[index];
        _elements[index] = element;
        return oldElement;
    }

    public final boolean add(final Object element) {
        ensureCapacity(_size + 1);
        _elements[_size++] = element;
        return true;
    }

    public final void add(final int index, final Object element) {
        checkHighRange(index);
        ensureCapacity(_size + 1);
        System.arraycopy(_elements, index, _elements, index + 1, _size - index);
        _elements[index] = element;
        ++_size;
    }

    public final boolean addAll(final Collection c) {
        int numNew = c.size();
        if (numNew == 0) {
            return false;
        }
        ensureCapacity(_size + numNew);
        System.arraycopy(c.toArray(), 0, _elements, _size, numNew);
        _size += numNew;
        return true;
    }

    public final boolean addAll(final int index, final Collection c) {
        checkHighRange(index);
        int numNew = c.size();
        if (numNew == 0) {
            return false;
        }
        ensureCapacity(_size + numNew);
        int numMoved = _size - index;
        if (numMoved > 0) {
            System.arraycopy(_elements, index, _elements, index + numNew, numMoved);
        }
        System.arraycopy(c.toArray(), 0, _elements, index, c.size());
        _size += numNew;
        return true;
    }

    public final void addAll(final Object[] array) {
        if (array == null) {
            throw new IllegalArgumentException("array");
        }
        _size = array.length;
        _elements = new Object[(_size * 110) / 100];
        System.arraycopy(array, 0, _elements, 0, _size);
    }

    public final Object remove(final int index) {
        checkRange(index);
        Object oldElement = _elements[index];
        int numMoved = _size - index - 1;
        if (numMoved > 0) {
            System.arraycopy(_elements, index + 1, _elements, index, numMoved);
        }
        _elements[--_size] = null;
        return oldElement;
    }

    public final boolean removeAll(Collection c) {
        boolean modified = false;
        for (int i = _size - 1; i >= 0; i--) {
            if (c.contains(_elements[i])) {
                remove(i);
                modified = true;
            }
        }
        return modified;
    }

    public final boolean retainAll(final Collection c) {
        boolean modified = false;
        for (int i = _size - 1; i >= 0; i--) {
            if (!c.contains(_elements[i])) {
                remove(i);
                modified = true;
            }
        }
        return modified;
    }

    public final void clear() {
        for (int i = 0; i < _size; i++) {
            _elements[i] = null;
        }
        _size = 0;
    }

    public final Iterator iterator() {
        return new Itr(this);
    }

    public final ListIterator listIterator() {
        return listIterator(0);
    }

    public final ListIterator listIterator(final int index) {
        checkHighRange(index);
        return new ListItr(this, index);
    }

    public final List subList(int fromIndex, int toIndex) {
        checkRange(fromIndex, toIndex);
        return new SubList(this, fromIndex, toIndex);
    }

    public final String toString() {
        StringBuffer buf = new StringBuffer(1024);
        buf.append("[");
        for (int i = 0; i < _size; i++) {
            buf.append(String.valueOf(_elements[i]));
            if (i < _size - 1) {
                buf.append(", ");
            }
        }
        buf.append("]");
        return buf.toString();
    }

    public final boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof List) || o == null) {
            return false;
        }

        List l = (List) o;
        if (_size != l.size()) {
            return false;
        }
        ListIterator i1 = listIterator();
        ListIterator i2 = l.listIterator();
        while (i1.hasNext() && i2.hasNext()) {
            Object o1 = i1.next();
            Object o2 = i2.next();
            if (!(o1 == null ? o2 == null : o1.equals(o2))) {
                return false;
            }
        }
        return true;
    }

    public final int hashCode() {
        int hashCode = 1;
        for (Iterator i = iterator(); i.hasNext(); ) {
            Object obj = i.next();
            hashCode = 31 * hashCode + (obj == null ? 0 : obj.hashCode());
        }
        return hashCode;
    }

    public final void writeExternal(final ObjectOutput out) throws IOException {
        out.writeInt(_elements.length);
        out.writeInt(_size);
        for (int i = 0; i < _size; i++) {
            out.writeObject(_elements[i]);
        }
    }

    public final void readExternal(final ObjectInput in)
             throws IOException, ClassNotFoundException {

        int num = in.readInt();
        _size = in.readInt();
        _elements = new Object[num];
        for (int i = 0; i < _size; i++) {
            _elements[i] = in.readObject();
        }
    }

    private final void ensureCapacity(final int minCapacity) {
    	Object oldElements[] = _elements;
        if (minCapacity > oldElements.length) {
            int newCapacity = (minCapacity * 3) / 2 + 1;
            _elements = new Object[newCapacity];
            System.arraycopy(oldElements, 0, _elements, 0, _size);
        }
    }

    private final void checkRange(final int index) {
        if (index >= _size) {
            throw new ArrayIndexOutOfBoundsException("index:" + index);
        }
    }

    private final void checkHighRange(final int index) {
        if (index > _size) {
            throw new ArrayIndexOutOfBoundsException("index:" + index);
        }
    }

    private void checkRange(final int fromIndex, final int toIndex) {
        if (fromIndex > toIndex) {
            throw new ArrayIndexOutOfBoundsException("fromIndex:" + fromIndex +
                    ",toIndex:" + toIndex);
        }
        checkHighRange(toIndex);
    }

    private class Itr implements Iterator {

        protected final List _list;
        protected int _current = 0;
        protected int _last = -1;

        Itr(final List list) {
            _list = list;
        }

        public final boolean hasNext() {
            return _current != _list.size();
        }

        public final Object next() {
            try {
                Object n = _list.get(_current);
                _last = _current++;
                return n;
            } catch (IndexOutOfBoundsException e) {
                throw new NoSuchElementException();
            }
        }

        public final void remove() {
            if (_last == -1) {
                throw new IllegalStateException();
            }
            _list.remove(_last);
            if (_last < _current) {
                _current--;
            }
            _last = -1;
        }
    }

    private final class ListItr extends Itr implements ListIterator {

        ListItr(final List list, final int index) {
            super(list);
            _current = index;
        }

        public final boolean hasPrevious() {
            return _current != 0;
        }

        public final Object previous() {
            try {
                Object p = _list.get(--_current);
                _last = _current;
                return p;
            } catch (IndexOutOfBoundsException e) {
                throw new NoSuchElementException();
            }
        }

        public final int nextIndex() {
            return _current;
        }

        public final int previousIndex() {
            return _current - 1;
        }

        public final void set(final Object element) {
            if (_last == -1) {
                throw new IllegalStateException();
            }
            _list.set(_last, element);
        }

        public final void add(final Object element) {
            _list.add(_current++, element);
            _last = -1;
        }
    }

    private final class SubList extends AbstractList {
    	
	    private List _list;
	    private int _offset;
	    private int _size;

	    SubList(List list, int fromIndex, int toIndex) {
	        if (fromIndex < 0) {
	            throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
	        }
	        if (toIndex > list.size()) {
	            throw new IndexOutOfBoundsException("toIndex = " + toIndex);
	        }
	        if (fromIndex > toIndex) {
	            throw new IllegalArgumentException("fromIndex(" + fromIndex +
	                                               ") > toIndex(" + toIndex + ")");
	        }
	        _list = list;
	        _offset = fromIndex;
	        _size = toIndex - fromIndex;
	    }

	    public final Object set(int index, Object element) {
	        return _list.set(index+_offset, element);
	    }

	    public final Object get(int index) {
	        return _list.get(index + _offset);
	    }

	    public final int size() {
	        return _size;
	    }

	    public final void add(int index, Object element) {
	        _list.add(index + _offset, element);
	        ++_size;
	    }

	    public final Object remove(int index) {
	        --_size;
	        return _list.remove(index+_offset);
	    }

	    public final boolean addAll(Collection c) {
	        return addAll(_size, c);
	    }
	
	    public final boolean addAll(int index, Collection c) {
	        int cSize = c.size();
	        if (cSize == 0) {
	            return false;
	        }
	        _list.addAll(_offset + index, c);
	        _size += cSize;
	        return true;
	    }

	    public final Iterator iterator() {
	        return listIterator();
	    }
	
	    public final ListIterator listIterator(final int index) {
	        return new ListIterator() {
	        	
	            private ListIterator _iter = _list.listIterator(index + _offset);
	
	            public boolean hasNext() {
	                return nextIndex() < _size;
	            }
	
	            public Object next() {
	                if (hasNext()) {
	                    return _iter.next();
	                } else {
	                    throw new NoSuchElementException();
	                }
	            }
	
	            public boolean hasPrevious() {
	                return previousIndex() >= 0;
	            }
	
	            public Object previous() {
	                if (hasPrevious()) {
	                    return _iter.previous();
	                } else {
	                    throw new NoSuchElementException();
	                }
	            }
	
	            public int nextIndex() {
	                return _iter.nextIndex() - _offset;
	            }
	
	            public int previousIndex() {
	                return _iter.previousIndex() - _offset;
	            }
	
	            public void remove() {
	                _iter.remove();
	                _size--;
	            }
	
	            public void set(Object o) {
	                _iter.set(o);
	            }
	
	            public void add(Object o) {
	                _iter.add(o);
	                _size++;
	            }
	        };
	    }
	
	    public final List subList(int fromIndex, int toIndex) {
	        return new SubList(this, fromIndex, toIndex);
	    }
	}
}
