package test.org.seasar.nazuna;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.nazuna.NazunaTokenizer;

public class NazunaTokenizerTest extends TestCase {

    public NazunaTokenizerTest(String name) {
        super(name);
    }

    public void testEOF() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("");
    	assertEquals("1", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    	assertEquals("2", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testWhitespace() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("\t       \n");
    	assertEquals("1", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testHyphen() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("       - ");
    	assertEquals("1", '-', tokenizer.nextToken());
    	assertEquals("2", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testInteger() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("       123");
    	assertEquals("1", NazunaTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("2", new Integer(123), tokenizer.ival);
    	assertEquals("3", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    	
    	NazunaTokenizer tokenizer2 = new NazunaTokenizer("    -1234567890");
    	assertEquals("4", NazunaTokenizer.TT_INTEGER, tokenizer2.nextToken());
    	assertEquals("5", new Integer(-1234567890), tokenizer2.ival);
    	
    	NazunaTokenizer tokenizer3 = new NazunaTokenizer(String.valueOf(Integer.MAX_VALUE));
    	assertEquals("6", NazunaTokenizer.TT_INTEGER, tokenizer3.nextToken());
    	assertEquals("7", new Integer(Integer.MAX_VALUE), tokenizer3.ival);
    	
    	NazunaTokenizer tokenizer4 = new NazunaTokenizer(String.valueOf(Integer.MIN_VALUE));
    	assertEquals("8", NazunaTokenizer.TT_INTEGER, tokenizer4.nextToken());
    	assertEquals("9", new Integer(Integer.MIN_VALUE), tokenizer4.ival);
    }
    
    public void testLong() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("       12345678901");
    	assertEquals("1", NazunaTokenizer.TT_LONG, tokenizer.nextToken());
    	assertEquals("2", new Long("12345678901"), tokenizer.lval);
    	assertEquals("3", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testDouble() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("       123.45 ");
    	assertEquals("1", NazunaTokenizer.TT_DOUBLE, tokenizer.nextToken());
    	assertEquals("2", new Double("123.45"), tokenizer.dval);
    	assertEquals("3", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testOrdinary() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("(1 + 2)");
    	assertEquals("1", '(', tokenizer.nextToken());
    	assertEquals("2", NazunaTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("3", new Integer(1), tokenizer.ival);
    	assertEquals("4", '+', tokenizer.nextToken());
    	assertEquals("5", NazunaTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("6", new Integer(2), tokenizer.ival);
    	assertEquals("7", ')', tokenizer.nextToken());
    	assertEquals("8", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testWord() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("123aaa45");
    	assertEquals("1", NazunaTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("2", new Integer(123), tokenizer.ival);
    	assertEquals("3", NazunaTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("4", "aaa45", tokenizer.sval);
    	assertEquals("5", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testQuote() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("'\n123''aaa'aa");
    	assertEquals("1", '\'', tokenizer.nextToken());
    	assertEquals("2", "\n123'aaa", tokenizer.sval);
    	assertEquals("3", NazunaTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("4", "aa", tokenizer.sval);
    	assertEquals("5", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testQuote2() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("out.println('a\b')");
    	tokenizer.nextToken();
    	tokenizer.nextToken();
    	assertEquals("1", '\'', tokenizer.nextToken());
    	assertEquals("2", "a\b", tokenizer.sval);
    	tokenizer.nextToken();
    	assertEquals("3", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testAtMark() throws Exception {
    	NazunaTokenizer tokenizer = new NazunaTokenizer("@aaa");
    	assertEquals("1", NazunaTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("2", "@aaa", tokenizer.sval);
    	assertEquals("3", NazunaTokenizer.TT_EOF, tokenizer.nextToken());
    }

    protected void setUp() throws Exception {
    }

    protected void tearDown() throws Exception {
    }

    public static Test suite ( ) {
        return new TestSuite(NazunaTokenizerTest.class);
    }

    public static void main (String[] args) {
        junit.textui.TestRunner.main(new String[]{NazunaTokenizerTest.class.getName()});
    }
}