#!/usr/bin/perl
#
# Rest Interface of LISM
#
# This code was developped by SECIOSS (http://www.secioss.co.jp/).
#
#              Copyright (C) 2011 SECIOSS CORPORATION
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation.
#

use strict;
use LISM;
use Net::LDAP::Util qw(ldap_error_text);
use DBI;
use CGI qw(:standard);
use HTTP::Request::Common qw(GET POST);
use HTTP::Cookies;
use LWP::UserAgent;
use JSON;
use XML::Simple;
use File::Basename;
use Config::General;
use MIME::Base64;
use Encode;
use Data::Dumper;

our $CONF = 'lism_restapi.conf';
our $error;

sub config
{
    my $config = Config::General->new($CONF);
    my %conf = $config->getall;

    if (!defined($conf{'basedn'})) {
        $conf{'basedn'} = 'o=lism,o=cgi';
    }

    return %conf;
}

sub factory
{
    my (%conf) = @_;

    my $lism = new LISM;
    $lism->config('basedn', $conf{'basedn'});
    if (defined($conf{'admin'})) {
        $lism->config('admindn', "cn=$conf{'admin'},$conf{'basedn'}");
    }
    foreach my $key (qw(syncdir conf adminpw)) {
        $lism->config($key, $conf{$key});
    }

    if ($lism->init()) {
        return undef;
    }

    return $lism;
}

sub getUser
{
    my (%conf) = @_;

    my ($token) = ($ENV{QUERY_STRING} =~ /oauth_token=([^&]+)/);
    if (!$token) {
        $error = "Invalid request";
        return undef;
    }

    my $db = DBI->connect($conf{oauth_dsn}, $conf{oauth_dbuser}, $conf{oauth_dbpasswd});
    if (!$db) {
        $error = "Can't get access token";
        return undef;
    }

    my $sql = "select client_id, expires, scope, username, session from tokens where oauth_token = '$token'";
    my $sth = $db->prepare($sql);
    if (!$sth->execute) {
        $error = "Can't get access token";
        return undef;
    }

    # get the record from the result
    my @data = $sth->fetchrow_array;
    $sth->finish;

    if ($data[1] && time() > $data[1]) {
        $error = "The access token has expired";
        return undef;
    }

    return ($data[3], $data[4]);
}

sub search
{
    my ($lism, $base, $scope, $sizeLim, $timeLim, $filter, $attrs) = @_;
    my @entries;

    $scope = defined($scope) ? $scope : 2;
    $filter = $filter ? $filter : '(objectClass=*)';
    $filter =~ s/&amp;/&/g;

    my @objects;
    my ($rc, @entries) = $lism->search($base, $scope, 0, $sizeLim, $timeLim, $filter, 0, split(/,/, $attrs));
    for (my $i = 0; $i < @entries; $i++) {
        $objects[$i] = {};
        foreach my $line (split(/\n/, $entries[$i])) {
            my ($attr, $value) = ($line =~ /([^:]+): *(.*)$/);
            if ($attr eq 'dn') {
                $objects[$i]{$attr} = $value;
            } else {
                if (!defined($objects[$i]{$attr})) {
                    $objects[$i]{$attr} = [];
                }
                push(@{$objects[$i]{$attr}}, $value);
            }
        }
    }

    return {code => $rc, message => ldap_error_text($rc), entries => \@objects};
}

sub add
{
    my ($lism, $dn, $json_req) = @_;
    my $entryStr;

    $entryStr = "dn: $dn\n";
    my $req = decode_json($json_req);
    foreach my $attr (keys %{$req}) {
        my @values;
        if (ref($req->{$attr}) eq 'ARRAY') {
            @values = @{$req->{$attr}};
        } else {
            $values[0] = $req->{$attr};
        }
        for (my $i = 0; $i < @values; $i++) {
            $entryStr = "$entryStr$attr: $values[$i]\n";
        }
    }

    my $rc = $lism->add($entryStr);

    return {code => $rc, message => ldap_error_text($rc)};
}

sub modify
{
    my ($lism, $dn, $json_req) = @_;
    my @changes = ();

    my $req = decode_json($json_req);
    foreach my $action (keys %{$req}) {
        if (ref($req->{$action}) eq 'ARRAY') {
            foreach my $info (@{$req->{$action}}) {
                foreach my $attr (keys %{$info}) {
                    my @values;
                    if (ref($info->{$attr}) eq 'ARRAY') {
                        foreach my $value (@{$info->{$attr}}) {
                            push(@values, $value);
                        }
                    } else {
                        $values[0] = $info->{$attr};
                    }
                    push(@changes, uc($action), $attr, @values);
                }
            }
        } else {
            foreach my $attr (keys %{$req->{$action}}) {
                my @values;
                if (ref($req->{$action}->{$attr}) eq 'ARRAY') {
                    foreach my $value (@{$req->{$action}->{$attr}}) {
                        push(@values, $value);
                    }
                } else {
                        $values[0] = $req->{$action}->{$attr};
                }
                push(@changes, uc($action), $attr, @values);
            }
        }
    }

    my $rc = $lism->modify($dn, @changes);

    return {code => $rc, message => ldap_error_text($rc)};
}

sub delete
{
    my ($lism, $dn) = @_;

    my $rc = $lism->delete($dn);

    return {code => $rc, message => ldap_error_text($rc)};
}

my ($file, $location, $suffix) = fileparse($ENV{SCRIPT_NAME}, '\.cgi', '\.pl');

my $q = CGI->new;
$q->charset('utf-8');
print $q->header();

my ($action) = ($ENV{QUERY_STRING} =~ /action=([^&]+)/);
my ($id) = ($ENV{QUERY_STRING} =~ /id=([^&]+)/);

my %conf = config();
my ($user, $session) = getUser(%conf);
if (!$user) {
    print encode_json({code => 49, message => $error});
}

my $lism = factory(%conf);
if (!defined($lism)) {
    print encode_json({-1, "Can't load LISM"});
}
$lism->{bind}{dn} = "uid=$user";

my $objectdn = defined($conf{$file.'dn'}) ? $conf{$file.'dn'}.',' : '';
$objectdn =~ s/\%u/$user/g;
my $dn = $objectdn.$conf{'basedn'};
if ($id) {
    if ($file eq 'user') {
        if ($id ne $user) {
            print encode_json({-1, "Invalid request(id)"});
        }
    } else {
        $dn = (defined($conf{$file.'attr'}) ? $conf{$file.'attr'} : 'cn').'='.$id.",$dn";
    }
}

my $res;
if ($action eq 'search') {
    my $scope = $q->param('scope');
    my $sizelimit = $q->param('sizelimit');
    my $timelimit = $q->param('timelimit');
    my $filter = $q->param('filter');
    my $attrs = $q->param('attrs');
    $res = &search($lism, $dn, $scope, $sizelimit, $timelimit, $filter, $attrs);
} elsif ($action eq 'add') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &add($lism, $dn, $q->param('POSTDATA'));
} elsif ($action eq 'modify') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &modify($lism, $dn, $q->param('POSTDATA'));
} elsif ($action eq 'delete') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &delete($lism, $dn);
} elsif ($action eq 'post' || $action eq 'get') {
    my $cookie_jar = HTTP::Cookies->new(
        file => '/var/tmp/cookie.'.$user,
        autosave => 1
    );
    my $ua = LWP::UserAgent->new;
    if ($session) {
        my ($name, $value) = split(/=/, $session);
        my ($scheme, $host, $port) = ($conf{url} =~ /^(https?):\/\/([^:\/]+):?([0-9]*)/);
        if (!$port) {
            $port = $scheme eq 'http' ? '80' : '443';
        }
        $cookie_jar->set_cookie(1, $name, $value, '/', $host, $port);
        $ua->cookie_jar($cookie_jar);
    }
    my $req;
    if ($action eq 'post') {
        my %data;
        foreach my $param (split(/&/, $ENV{QUERY_STRING})) {
            my ($key, $value) = split(/=/, $param);
            $data{$key} = $value;
        }
        $req = POST $conf{url}, [%data];
    } else {
        $req = GET $conf{url}.'?'.$ENV{QUERY_STRING};
    }
    my $response = $ua->request($req);
    if ($response->is_success) {
        my $format = defined($conf{format}) ? $conf{format} : 'json';
        if ($format eq 'xml') {
            $res = eval {XMLin($response->content)};
        } else {
            $res = decode_json($response->content);
        }
    } else {
        $res = {code => -1, message => "Can't connect server"};
    }
    $cookie_jar->clear;
} else {
    $res = {code => -1, message => "Invalid action"};
}

print encode_json($res);

1;
