<?php
/**
 *  DB.php
 *
 *  PHP versions 4 and 5
 *
 *  @package    Auth
 *
 *  @author     Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 *  @copyright  2016 SECIOSS, INC.
 *
 *  @version    CVS: $Id$
 */
require_once 'Secioss/AutoLogin/Container.php';

/**
 *  AutoLogin_Container_DB
 *
 *  @package    Auth
 *
 *  @author     Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 *  @copyright  2016 SECIOSS, INC.
 *
 *  @version    CVS: $Id$
 */
class AutoLogin_Container_DB extends AutoLogin_Container
{
    // {{{ AutoLogin_Container_DB

    /**
     *  AutoLogin_Container_DBクラスのコンストラクタ
     *
     *  @access public
     *
     *  @param  mixed   $options        DBの設定
     *
     *  @return mixed   0:正常終了 PEAR_Error:エラー
     */
    public function __construct($options)
    {
        parent::__construct($options);
        if (!$this->options['logincol']) {
            $this->options['logincol'] = $this->options['usercol'];
        }

        $arr_dsn = explode(' ', $this->options['dsn']);
        foreach ($arr_dsn as $this->options['dsn']) {
            $rc = $this->_connect();
            if (!$rc) {
                break;
            }
        }
        register_shutdown_function([&$this, '_disconnect']);

        return $rc;
    }

    // }}}

    // {{{ _connect()

    /**
     * DBサーバーに接続する
     *
     * @access private
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function _connect()
    {
        $db_conn = 0;
        $db_type = isset($this->options['db_type']) && $this->options['db_type'] ? $this->options['db_type'] : 'mysql';
        $db_user = $this->options['db_user'];
        $db_pass = $this->options['db_pass'];
        $db_name = $this->options['db_name'];
        $db_hosts = explode(' ', $this->options['db_host']);
        $db_timeout = isset($this->options['db_timeout']) ? $this->options['db_timeout'] : 10;
        foreach ($db_hosts as $db_host) {
            try {
                $db_conn++;
                $port = '';
                if (strpos($db_host, ':')) {
                    $hostinfo = explode(':', $db_host);
                    $db_host = $hostinfo[0];
                    $port = $hostinfo[1];
                }
                $dsn = "$db_type:host=$db_host;".($port ? "port=$port;" : '')."dbname=$db_name;charset=utf8";
                $this->conn = new PDO($dsn, $db_user, $db_pass, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION, PDO::ATTR_PERSISTENT => true, PDO::ATTR_TIMEOUT => $db_timeout]);
                break;
            } catch (PDOException $e) {
                if ($db_conn == count($db_hosts)) {
                    $this->conn = null;
                    return PEAR::raiseError($e->getMessage(), AUTO_LOGIN_ERROR);
                }
            }
        }

        return 0;
    }

    // }}}

    // {{{ _disconnect()

    /**
     * DBサーバーへの接続を切断する
     *
     * @access private
     */
    public function _disconnect()
    {
        if ($this->conn) {
            $this->conn = null;
        }
    }

    // {{{ fetchData()

    /**
     * DBサーバーからユーザー情報を取得する
     *
     * @param  string username
     * @param mixed $username
     *
     * @return mixed
     */
    public function fetchData($username)
    {
        if (!$username || strlen($username) > 255) {
            return PEAR::raiseError('Invalid user id', AUTO_LOGIN_INVALID_VALUE);
        }
        $tenant = '';
        if (preg_match('/^(.+)@([^@]+)$/', $username, $matches)) {
            $username = $matches[1];
            $tenant = $matches[2];
            $tenant = preg_replace('/([*()])/', '\\$1', $tenant);
        }
        $this->tenant = $tenant;

        if (!$this->conn) {
            return PEAR::raiseError("Can't contact DB server", AUTO_LOGIN_ERROR);
        }

        $username = $this->conn->quote($username);
        $sql = 'select * from '.$this->options['table'].' where '.$this->options['usercol']."='$username'";
        if ($this->tenant) {
            $sql .= " and tenant = '".$this->tenant."'";
        }

        try {
            $sth = $this->conn->prepare($sql);
            $sth->execute();
            $num = $sth->rowCount();
            if ($num == 0) {
                return PEAR::raiseError("User doesn't exist", AUTO_LOGIN_NO_USER);
            } elseif ($num != 1) {
                return PEAR::raiseError("User isn't unique", AUTO_LOGIN_ERROR);
            }

            $row = $sth->fetch(PDO::FETCH_ASSOC);
            $this->id = $row[$this->options['idcol']];
            $this->prop = [];
            foreach ($row as $key => $value) {
                if ($this->options['col_to_array'] && in_array($key, explode(' ', $this->options['col_to_array']))) {
                    $value = json_decode($value, true);
                }
                $this->prop[$key] = $value;
            }
            if (isset($this->prop[$this->options['logincol']])) {
                $this->loginid = $this->prop[$this->options['logincol']];
            }
        } catch (PDOException $e) {
            return PEAR::raiseError($e->getMessage()."($sql)", AUTO_LOGIN_ERROR);
        }
    }

    // }}}

    // {{{ auth()

    /**
     *  ユーザーの認証を行う
     *
     *  @access public
     *
     *  @param  string   パスワード
     * @param mixed $password
     *
     *  @return string   0:成功 1:失敗
     */
    public function auth($password)
    {
        if (!$password || !isset($this->prop[$this->options['pwdcol']])) {
            return 1;
        }

        $userpasswd = $this->prop[$this->options['pwdcol']];

        return $this->hashPasswd($password) == $userpasswd ? 0 : 1;
    }

    // }}}

    // {{{ getPassword()

    /**
     *  暗号化されているパスワードを復号化して取得する
     *
     *  @access public
     *
     *  @return string   パスワード
     */
    public function getPassword()
    {
        if (isset($this->prop[$this->options['encryptpwdcol']])) {
            return $this->decrypt($this->prop[$this->options['encryptpwdcol']]);
        } else {
            return '';
        }
    }

    // }}}

    public function setProp($prop)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        $setval = '';
        foreach ($prop as $key => $value) {
            if ($this->options['col_to_array'] && in_array($key, explode(' ', $this->options['col_to_array']))) {
                $value = json_encode($value);
            } else {
                $value = is_array($value) ? $value[0] : $value;
            }
            $value = $this->conn->quote($value);
            $setval .= ($setval ? ',' : '')." $key = '$value'";
        }

        $sql = 'update '.$this->options['table']." set $setval where ".$this->options['idcol']." = '".$this->id."'";
        if ($this->tenant) {
            $sql .= " and tenant = '".$this->tenant."'";
        }

        try {
            $this->conn->query($sql);
        } catch (PDOException $e) {
            return PEAR::raiseError($e->getMessage()."($sql)", AUTO_LOGIN_ERROR);
        }

        return true;
    }

    // {{{ setPassword()

    /**
     *  パスワードを暗号化してDBに格納する
     *
     *  @access public
     *
     *  @param  string   パスワード
     * @param null|mixed $password
     * @param null|mixed $app
     *
     *  @return mixed    true:正常終了 PEAR_Error:エラー
     */
    public function setPassword($password = null, $app = null)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if (!$password || strlen($password) > 255) {
            return PEAR::raiseError('Invalid password', AUTO_LOGIN_INVALID_VALUE);
        }

        $setval = $this->options['pwdcol']." = '".$this->hashPasswd($password)."'";
        if ($this->options['encryptpwdcol']) {
            $setval = $setval.', '.$this->options['encryptpwdcol']." = '".$encrypt."'";
        }
        if ($this->options['pwdtimecol']) {
            $setval = "$setval, ".$this->options['pwdtimecol']." = '".date('YmdHis')."'";
        }
        if ($this->options['encryptpwdcol']) {
            $encrypt = $this->encrypt($password);
            if (PEAR::isError($encrypt)) {
                return $encrypt;
            }
            $setval = $setval.', '.$this->options['encryptpwdcol']." = '".$encrypt."'";
        }
        $sql = 'update '.$this->options['table']." set $setval where ".$this->options['idcol']." = '".$this->id."'";

        try {
            $this->conn->query($sql);
        } catch (PDOException $e) {
            return PEAR::raiseError($e->getMessage()."($sql)", AUTO_LOGIN_ERROR);
        }

        return true;
    }

    // }}}

    // {{{ _setDefaults()

    /**
     * optionsにデフォルト値を設定する
     *
     * @access private
     */
    public function _setDefaults()
    {
        parent::_setDefaults();

        $this->options['dsn'] = '';
        $this->options['table'] = '';
        $this->options['idcol'] = '';
        $this->options['usercol'] = '';
        $this->options['pwdcol'] = '';
        $this->options['logincol'] = '';
        $this->options['encryptpwdcol'] = '';
        $this->options['pwdtimecol'] = '';
        $this->options['secretcol'] = 'initsecret';
        $this->options['pincol'] = '';
        $this->options['col_to_array'] = 'seciossdeviceidxfido';
    }

    // }}}

    // {{{ setSecret()

    /**
     *  シークレットを暗号化してDBに格納する
     *
     *  @access public
     *
     *  @param  string   シークレット
     * @param mixed      $secret
     * @param null|mixed $pin
     * @param null|mixed $deviceid
     * @param null|mixed $device
     * @param null|mixed $otplen
     * @param null|mixed $timewindow
     * @param null|mixed $os
     * @param null|mixed $ip
     *
     *  @return mixed    true:正常終了 PEAR_Error:エラー
     */
    public function setSecret($secret, $pin = null, $deviceid = null, $device = null, $otplen = null, $timewindow = null, $os = null, $ip = null)
    {
        if (!$secret || strlen($secret) > 255) {
            return PEAR::raiseError('Invalid secret', AUTO_LOGIN_INVALID_VALUE);
        }

        $encrypt = $this->encrypt($secret);
        if (PEAR::isError($encrypt)) {
            return $encrypt;
        }

        $prop = [$this->options['secretcol'] => $encrypt];
        if ($this->options['pincol'] && $pin) {
            $encrypt = $this->encrypt($secret);
            if (PEAR::isError($encrypt)) {
                return $encrypt;
            }
            $prop[$this->options['pincol']] = $encrypt;
        }
        if ($otplen && $otplen > 0) {
            $prop['otp_length'] = $otplen;
        }
        if ($timewindow && $timewindow > 0) {
            $prop['otp_timewindow'] = $timewindow;
        }

        $file = null;
        if (isset($_GET['app'])) {
            $file = $_GET['app'];
        } elseif (preg_match('/\/(secret|qrsecret|websecret)\.php$/', $_SERVER['SCRIPT_NAME'], $matches)) {
            $file = $matches[1];
        }
        $type = null;
        switch ($file) {
            case 'secret':
                //$type = 'hard';
                $type = 1;
                break;
            case 'qrsecret':
                if (isset($_GET['st'])) {
                    //$type = 'soft';
                    $type = 3;
                } else {
                    //$type = 'slinkpass';
                    $type = 2;
                }
                break;
            case 'websecret':
                //$type = 'web';
                $type = 4;
                break;
        }
        $prop['otp_type'] = $type;

        return $this->setProp($prop);
    }

    // }}}

    public function add($username, $prop)
    {
        $cols = [];
        $vals = [];
        foreach ($prop as $key => $value) {
            $value = is_array($value) ? $value[0] : $value;
            $value = $this->conn->quote($value);
            $cols[] = $key;
            $vals[] = $value;
        }
        $sql = 'insert into '.$this->options['table'].'('.$this->options['usercol'].', '.join(', ', $cols).") values('$username', ".join(', ', $vals).')';

        try {
            $this->conn->query($sql);
        } catch (PDOException $e) {
            return PEAR::raiseError($e->getMessage()."($sql)", AUTO_LOGIN_ERROR);
        }

        return true;
    }
}
