<?php
/**
 * AutoLoginSimple.php
 *
 * PHP 5.4
 *
 * @package Auth
 *
 * @author Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 * @copyright 2020 SECIOSS, INC.
 */
namespace Secioss;

use PEAR;

/**
 * Secioss\AutoLoginSimple
 */
class AutoLoginSimple
{
    /**
     * AutoLoginSimple クラスのコンストラクタ
     *
     * @access public
     *
     * @param string $driver  ストレージドライバの型
     * @param mixed  $options ストレージドライバの設定
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function __construct($driver, $options)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);

        $storage = $this->_factory($driver);
        if (PEAR::isError($storage)) {
            return $storage;
        }
        $this->storage = $storage;

        return 0;
    }

    /**
     * ユーザー情報を取得する
     *
     * @access public
     *
     * @param string $username ユーザー名（ユーザーID）
     *
     * @return bool true:正常終了 PEAR_Error:エラー
     */
    public function setUser($username)
    {
        $this->username = $username;

        $rc = $this->storage->fetchData($username);
        if (PEAR::isError($rc)) {
            return $rc;
        }

        return true;
    }

    /**
     * テナント情報を取得する
     *
     * @access public
     *
     * @param string $tenant テナントID
     *
     * @return mixed テナント情報
     */
    public function getTenants($tenant = null)
    {
        return $this->storage->getTenants($tenant);
    }

    /**
     * シークレットを設定する
     *
     * @access public
     *
     * @param string     $secret     シークレット
     * @param null|mixed $pin
     * @param null|mixed $deviceid
     * @param null|mixed $device
     * @param null|mixed $otplen
     * @param null|mixed $timewindow
     * @param null|mixed $os
     * @param null|mixed $ip
     * @param null|mixed $euser
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function updateSecret($secret, $pin = null, $deviceid = null, $device = null, $otplen = null, $timewindow = null, $os = null, $ip = null, $euser = null)
    {
        $rc = 0;

        if ($euser) {
            $this->storage->euser = $euser;
        }
        if (!$this->app) {
            $rc = $this->storage->validateSecret($secret, $pin);
        }
        if ($rc) {
            switch ($rc) {
                case AUTO_LOGIN_INVALID_VALUE:
                    $message = 'Invalid secret or PIN';
                    break;
            }
            return PEAR::raiseError($message, $rc);
        }
        if ($otplen && $otplen != 6 && $otplen != 8) {
            return PEAR::raiseError('Invalid otp length', AUTO_LOGIN_INVALID_VALUE);
        }
        if ($timewindow && $timewindow != 30 && $timewindow != 60) {
            return PEAR::raiseError('Invalid time window', AUTO_LOGIN_INVALID_VALUE);
        }

        return $this->storage->setSecret($secret, $pin, $deviceid, $device, $otplen, $timewindow, $os, $ip);
    }

    /**
     * プロパティへのアクセサ(R)
     *
     * $login->get()
     * priviledge/index.php
     *
     * $login->storage->get()
     * others\php
     *
     * @access public
     *
     * @param string $key プロパティ名
     *
     * @return mixed プロパティ
     */
    public function get($key)
    {
        if (isset($this->storage->prop[$key])) {
            return $this->storage->prop[$key];
        }
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * websecret.php
     *
     * @access public
     *
     * @param string $key   プロパティ名
     * @param string $value 値
     *
     * @return mixed プロパティ
     */
    public function set($key, $value)
    {
        return $this->storage->setProp([$key => $value]);
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * $login->storage->getProp(
     * lib/account.php
     * lib/setautologin.php
     * lib/resetotp.php
     * user/secret.php
     *
     * @access public
     *
     * @param mixed $keys プロパティ名
     *
     * @return mixed プロパティ
     */
    public function getProp($keys = null)
    {
        $prop = [];
        if ($keys) {
            foreach ($keys as $key) {
                if (isset($this->storage->prop[$key])) {
                    $prop[$key] = $this->storage->prop[$key];
                }
            }
        } else {
            $prop = $this->storage->prop;
        }
        return $prop;
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * $login->storage->getProp(
     * account.php
     * loginid.php
     * appcookieid.php
     * appradiusid.php
     * u2f.php
     * motpmail.php
     * checkotp.php
     * idmap.php
     * threat_detection.php
     *
     * @access public
     *
     * @param array $prop プロパティ名 => 値
     *
     * @return mixed プロパティ
     */
    public function setProp($prop)
    {
        return $this->storage->setProp($prop);
    }

    /**
     * シークレットの妥当性を確認する
     *
     * @access public
     *
     * @param string $secret シークレット
     *
     * @return mixed 0:正常終了
     */
    public function validateSecret($secret)
    {
        return 0;
    }

    /**
     * optionsにデフォルト値を設定する
     *
     * @access protected
     */
    protected function _setDefaults()
    {
    }

    /**
     * optionsに値を設定する
     *
     * @access protected
     *
     * @param  array
     * @param mixed $array
     */
    protected function _parseOptions($array)
    {
        if (is_array($array)) {
            foreach ($array as $key => $value) {
                $this->options[$key] = $value;
            }
        }
    }

    /**
     * ストレージドライバのオブジェクトを返す
     *
     * @access private
     * @static
     *
     * @param string $driver ストレージクラスの型
     *
     * @return object Object   Storageオブジェクト
     */
    private function _factory($driver)
    {
        $class = 'Secioss\Simple\\'.$driver;
        if (class_exists($class)) {
            return new $class($this->options);
        } else {
            return PEAR::raiseError('Driver not exist', AUTO_LOGIN_ERROR);
        }
    }
}
