<?php
/**
 * SSO.php
 *
 * PHP 5.4
 *
 * @package Auth
 *
 * @author Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 * @copyright 2020 SECIOSS, INC.
 */
namespace Secioss\Enterprise;

use Config;
use PEAR;

/**
 * Secioss\Enterprise\SSO
 */
class SSO extends \Secioss\Simple\SSO implements \Secioss\Enterprise\Storage
{
    /**
     * SSOクラスのコンストラクタ
     *
     * @access public
     *
     * @param mixed $options SSOの設定
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function __construct($options)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);

        $rc = parent::__construct($options);

        return $rc;
    }

    /**
     * optionsにデフォルト値を設定する
     *
     * @access private
     */
    public function _setDefaults()
    {
        parent::_setDefaults();

        $this->options['loginattr'] = '';
        $this->options['encryptpwdattr'] = 'seciossencryptedpassword';
        $this->options['soappath'] = '';
    }

    public function getLoginId()
    {
    }

    /**
     * 暗号化されているパスワードを復号化して取得する
     *
     * @access public
     *
     * @return string パスワード
     */
    public function getPassword()
    {
        if (isset($this->prop[$this->options['encryptpwdattr']])) {
            return $this->decrypt($this->prop[$this->options['encryptpwdattr']]);
        } else {
            return '';
        }
    }

    /**
     * パスワードを暗号化してLDAPに格納する
     *
     * @access public
     *
     * @param string パスワード
     * @param null|mixed $password
     * @param null|mixed $app
     * @param mixed      $init
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function setPassword($password = null, $app = null, $init = false)
    {
        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $path = file_exists('/Perl64') ? '/Perl64/bin' : '/Perl/bin';
            $loginattr = $this->options['loginattr'] ? $this->options['loginattr'] : 'uid';
            $output = system("$path/perl /opt/secioss/sbin/pwsync ".$this->loginid." $password $loginattr 2>&1", $rc);
            if ($rc) {
                return PEAR::raiseError('pwsync command failed: '.join(' ', $output));
            }

            return true;
        }

        $file = '/opt/secioss/var/www/conf/'.$this->options['soappath'].'lismapi.conf';
        if (!file_exists($file)) {
            return PEAR::raiseError("Can't find config file", AUTO_LOGIN_ERROR);
        }

        $conf = new Config();
        $root = &$conf->parseConfig($file, 'apache');
        if (PEAR::isError($root)) {
            return PEAR::raiseError("Can't read config file", AUTO_LOGIN_ERROR);
        }

        $params = $root->toArray();
        if (!isset($params['root']['admin']) || !isset($params['root']['adminpw'])) {
            return PEAR::raiseError('Invalid configuration', AUTO_LOGIN_ERROR);
        }
        $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on' ? 'https' : 'http').'://localhost:'.$_SERVER['SERVER_PORT'];
        try {
            $soap = new SoapClient(null, ['uri' => "$url/SoapLism", 'location' => "$url/api/".$this->options['soappath'].'lism.cgi', 'trace' => 1]);
            $result = $soap->__soapCall('bind', ['cn='.$params['root']['admin'].',o=lism', $params['root']['adminpw']]);
        } catch (SoapFault $exception) {
            $error = $exception->getMessage().' '.preg_replace("/[\r\n]/", ' ', $soap->__getLastResponse());
            return PEAR::raiseError($error);
        }
        if (!$result) {
            return PEAR::raiseError("Can't connect LISM", 80);
        } elseif ($result[0]) {
            return PEAR::raiseError(ldap_err2str($result[0]), $result[0]);
        }

        $sessid = $result[2];
        $loginattr = $this->options['loginattr'] ? $this->options['loginattr'] : 'uid';
        $result = $soap->__soapCall('search', [$sessid, 'o=lism', 2, 0, 0, 0, "($loginattr=".$this->loginid.')', 0, [$loginattr]]);
        if (!$result) {
            return PEAR::raiseError("Can't connect LISM", 80);
        } elseif ($result[0]) {
            return PEAR::raiseError(ldap_err2str($result[0]), $result[0]);
        } elseif (!preg_match('/^dn: (.+)/', $result[2][0], $matches)) {
            return PEAR::raiseError('No user', 32);
        }
        $dn = $matches[1];

        $result = $soap->__soapCall('bind', [$dn, $password]);
        if ($result[0] === 0) {
            return true;
        }

        $result = $soap->__soapCall('modify', ['sessid' => $sessid, 'dn' => $dn, 'modifyRequest' => ['REPLACE' => ['userpassword' => [$password]]]]);
        if (!$result) {
            return PEAR::raiseError("Can't connect LISM", 80);
        } elseif ($result[0]) {
            return PEAR::raiseError(ldap_err2str($result[0]), $result[0]);
        }

        return true;
    }

    public function deletePassword($app)
    {
    }

    public function getPwdChangedTime()
    {
    }

    public function getPwdHistory()
    {
    }

    public function getPwdLockout()
    {
    }

    public function isPwdMustChange()
    {
    }

    public function getProfiles()
    {
    }

    public function getProfileConf($tenant = null)
    {
    }

    public function getPwdPolicy($tenant = null)
    {
    }

    public function setDevice($device, $deviceid, $status, $os, $ip, $note, $appver)
    {
    }

    public function deleteDevice($device, $deviceid)
    {
    }

    public function getDeviceCurrentNum($ltype, $dtype, $newid, $tenant)
    {
    }
}
