<?php
/**
 *  Util.php
 *
 *  PHP version 5.4+
 *
 *  @package    Util
 *
 *  @author     SECIOSS <info@secioss.co.jp>
 *  @copyright  2020 SECIOSS, INC.
 *
 *  @version    $Id$
 */
namespace Secioss;

class Util
{
    private static $SECIOSS_CHARACTER_CODE = [
        '!' => '\x21',
        '"' => '\x22',
        '#' => '\x23',
        '$' => '\x24',
        '%' => '\x25',
        '&' => '\x26',
        '\'' => '\x27',
        '(' => '\x28',
        ')' => '\x29',
        '*' => '\x2a',
        '+' => '\x2b',
        ',' => '\x2c',
        '-' => '\x2d',
        '.' => '\x2e',
        '/' => '\x2f',
        ':' => '\x3a',
        ';' => '\x3b',
        '<' => '\x3c',
        '=' => '\x3d',
        '>' => '\x3e',
        '?' => '\x3f',
        '@' => '\x40',
        '[' => '\x5b',
        // '\\' => '\x5c', must be replaced before others
        ']' => '\x5d',
        '^' => '\x5e',
        '_' => '\x5f',
        '`' => '\x60',
        '{' => '\x7b',
        '|' => '\x7c',
        '}' => '\x7d',
        '~' => '\x7e',
    ];

    /**
     * 文字列に含まれる特殊記号を文字コードを含んだ独自の文字へ変換する。
     *
     * @access public static
     *
     * @param string $str 指定の文字列
     *
     * @return string 変換後の文字列
     */
    public static function strtohex($str)
    {
        $res = '';
        if (isset($str)) {
            $search = array_keys(self::$SECIOSS_CHARACTER_CODE);
            $replace = array_values(self::$SECIOSS_CHARACTER_CODE);
            $tmp = str_replace('\\', '\x5c', $str);
            $res = str_replace($search, $replace, $tmp);
        }
        return $res;
    }

    /**
     * strtohex によって変換された文字列を復号化する。
     *
     * @access public static
     *
     * @param string $str 特殊変換された文字
     *
     * @return string 復号化後の文字列
     */
    public static function hextostr($str)
    {
        $res = '';
        if (isset($str)) {
            $search = array_values(self::$SECIOSS_CHARACTER_CODE);
            $replace = array_keys(self::$SECIOSS_CHARACTER_CODE);
            $tmp = str_replace('\x5c', '\\', $str);
            $res = str_replace($search, $replace, $tmp);
        }
        return $res;
    }

    /**
     * ランダム文字列
     *
     * @access public static
     *
     * @param int $len 文字列の長さ
     *
     * @return string ランダム文字列
     */
    public static function random($len)
    {
        $token = array_merge(range('0', '9'), range('a', 'z'), range('A', 'Z'));

        $str = '';
        for ($i = 0; $i < $len; $i++) {
            $str = $str.$token[rand(0, count($token) - 1)];
        }

        return $str;
    }

    /**
     * スカラー値を要素数1の配列として返す
     *
     * @access public static
     *
     * @param mixed $v 配列として扱う値
     *
     * @return array 配列に変換された値
     */
    public static function to_array($v)
    {
        if (is_array($v)) {
            return $v;
        } elseif ($v) {
            return [$v];
        } else {
            return [];
        }
    }

    /**
     * パスワードの暗号化
     * パスワードを暗号化する。対応する暗号方式はCRYPT、MD5、SHA、SSHA。
     *
     * @access public static
     *
     * @param string      $passwd パスワード
     * @param null|string $pwhash
     * @param null|string $salt
     *
     * @return string 暗号化されたパスワード
     */
    public static function hashPasswd($passwd, $pwhash = null, $salt = null)
    {
        if (!$pwhash) {
            return $passwd;
        }

        $opts = preg_split('/:/D', $pwhash);
        $htype = $opts[0];
        $otype = isset($opts[1]) ? $opts[1] : '';
        $num = isset($opts[2]) ? $opts[2] : 1;

        for ($i = 0; $i < $num; $i++) {
            switch ($htype) {
                case 'CRYPT':
                    $token = array_merge(range('0', '9'), range('a', 'z'), range('A', 'Z'));
                    if (!$salt) {
                        $salt = $token[rand(0, count($token) - 1)].$token[rand(0, count($token) - 1)];
                    }
                    $passwd = crypt($passwd, $salt);
                    break;
                case 'MD5':
                    $passwd = md5($passwd);
                    $passwd = $otype == 'hex' ? $passwd : base64_encode(pack('H*', $passwd));
                    break;
                case 'SHA':
                    $passwd = sha1($passwd);
                    $passwd = $otype == 'hex' ? $passwd : base64_encode(pack('H*', $passwd));
                    break;
                case 'SSHA':
                    if (!$salt) {
                        $salt = openssl_random_pseudo_bytes(4);
                    }
                    $passwd = base64_encode(sha1($passwd.$salt, true).$salt);
                    break;
                case 'AD':
                    $passwd = '"'.$passwd.'"';
                    $adpasswd = '';
                    for ($j = 0; $j < strlen($passwd); $j++) {
                        $adpasswd .= "{$passwd[$j]}\000";
                    }
                    return $adpasswd;
                    break;
                case 'CLEARTEXT':
                    return $passwd;
                    break;
                default:
                    return $passwd;
            }
        }

        return $passwd;
    }

    public static function cmpPasswd($passwd, $hashedpwd, $pwhash = null)
    {
        $tab = '';
        if ($pwhash == 'CRYPT' || $pwhash == 'MD5' || $pwhash == 'SHA' || $pwhash == 'SSHA') {
            $tab = '{'.$pwhash.'}';
        }

        if ($pwhash == 'SSHA') {
            $salt = substr(base64_decode(substr($hashedpwd, 6)), 20);
            return $hashedpwd == $tab.(self::hashPasswd($passwd, $pwhash, $salt));
        } elseif ($pwhash == 'CRYPT') {
            $salt = substr($hashedpwd, strlen($tab), 2);
            return $hashedpwd == $tab.(self::hashPasswd($passwd, $pwhash, $salt));
        } else {
            return $hashedpwd == $tab.(self::hashPasswd($passwd, $pwhash));
        }
    }

    public static function ssl_encrypt($string, $keyfile)
    {
        $publickey = openssl_get_publickey('file://'.$keyfile);
        if (openssl_public_encrypt($string, $encrypted, $publickey, OPENSSL_PKCS1_OAEP_PADDING)) {
            return base64_encode($encrypted);
        } else {
            PEAR::raiseError('Encryption failed', AUTO_LOGIN_ERROR);
        }
    }

    public static function ssl_decrypt($string, $keyfiles)
    {
        $string = base64_decode($string);
        foreach ($keyfiles as $keyfile) {
            $privatekey = openssl_get_privatekey('file://'.$keyfile);
            if (openssl_private_decrypt($string, $decrypted, $privatekey, OPENSSL_PKCS1_OAEP_PADDING)) {
                return $decrypted;
            }
        }
        return PEAR::raiseError('Decryption failed', AUTO_LOGIN_ERROR);
    }

    /**
     * 文字列の暗号化を行う
     *
     * @access public static
     *
     * @param string     $string  文字列
     * @param string     $keyfile 公開鍵のファイルパス
     * @param null|mixed $sslkey
     * @param null|mixed $aeskey
     *
     * @return string 暗号化された文字列
     */
    public static function encrypt($string, $sslkey = null, $aeskey = null)
    {
        if ($string === null || $string === '') {
            return $string;
        }

        if ($sslkey) {
            return self::ssl_encrypt($string, $sslkey);
        } elseif ($aeskey) {
            return Crypt::encrypt($string, $aeskey);
        } else {
            return PEAR::raiseError('Key is invalid', AUTO_LOGIN_ERROR);
        }
    }

    /**
     * 文字列の復号化を行う
     *
     * @access public static
     *
     * @param mixed      $string
     * @param null|mixed $sslkey
     * @param null|mixed $aeskey
     *
     * @return string 復号化された文字列
     */
    public static function decrypt($string, $sslkey = null, $aeskey = null)
    {
        if ($string === null || $string === '') {
            return $string;
        }

        if (is_array($sslkey) && count($sslkey)) {
            return self::ssl_decrypt($string, $sslkey);
        } elseif ($aeskey) {
            return Crypt::decrypt($string, $aeskey);
        } else {
            return PEAR::raiseError('Key is invalid', AUTO_LOGIN_ERROR);
        }
    }

    /**
     * ポストデータの値を変数から取得する
     *
     * @access public static
     *
     * @param string $value     ＆｛関数（＄｛変数｝）｝
     * @param string $variables ユーザー属性値
     *
     * @return string 暗号化された文字列
     */
    public static function parsePostval($value, $variables = [])
    {
        // Parse query strings and cookies
        if (preg_match('/%{([^}]+)}/', $value, $matches)) {
            if (!isset($_GET[$matches[1]]) && !isset($_COOKIE[$matches[1]])) {
                return '';
            }
            $value = preg_replace('/%{'.$matches[1].'}/', ($_GET[$matches[1]] ? $_GET[$matches[1]] : $_COOKIE[$matches[1]]), $value);
        }

        // Parse variables ['loginid' => 'user01']
        // eg. &{function(${loginid})} -> &{function(user01)}
        foreach ($variables as $attr => $data) {
            if (preg_match('/\${'.$attr.'}/', $value)) {
                $value = preg_replace('/\${'.$attr.'}/', $data, $value);
            }
        }

        // Parse and invoke functions
        // eg. &{sha1(user01)} -> sha1(user01)
        // OR  &{sha256(user01)} -> hash(sha256, user01)
        if (preg_match('/&{([^}]+)}/', $value, $matches)) {
            // &{  function(user01) }
            for ($i = 1; $i < count($matches); $i++) {
                if (preg_match('/([^\(]+)\(([^\)]*)\)/', $matches[$i], $elts)) {
                    // function  (  user01  )
                    if (function_exists($elts[1])) {
                        // md5 sha1
                        $output = $elts[1]($elts[2]);
                    } elseif (in_array($elts[1], hash_algos())) {
                        // sha256 etc.
                        $output = hash($elts[1], $elts[2]);
                    } else {
                        $output = $elts[1].'('.$elts[2].')';
                    }
                    $value = preg_replace('/&{'.addcslashes($matches[$i], '()').'}/', $output, $value);
                // &{ function\(user01\) }
                } else {
                    return '';
                }
            }
        }

        return $value;
    }
}
