/* Copyright(C) 2006-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "senna_in.h"
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "sym.h"
#include "ql.h"

static sen_obj *_native_method_records(sen_ctx *c, sen_obj *args, sen_ql_co *co);
static sen_obj *_native_method_object(sen_ctx *c, sen_obj *args, sen_ql_co *co);
static sen_obj *_native_method_void(sen_ctx *c, sen_obj *args, sen_ql_co *co);

inline static void
rec_obj_bind(sen_obj *obj, sen_records *rec, sen_id cls)
{
  obj->type = sen_ql_records;
  obj->flags = SEN_OBJ_NATIVE|SEN_OBJ_ALLOCATED;
  obj->class = cls;
  obj->u.r.records = rec;
  obj->u.o.func = _native_method_records;
}

inline static void
obj_obj_bind(sen_obj *obj, sen_id cls, sen_id self)
{
  obj->type = sen_ql_object;
  obj->class = cls;
  obj->u.o.self = self;
  obj->flags = SEN_OBJ_NATIVE;
  obj->u.o.func = _native_method_object;
}

inline static sen_obj *
slot_value_obj(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  sen_id *ip;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  ip = NILP(car) ? sen_ra_at(slot->u.o.ra, id) : sen_ra_get(slot->u.o.ra, id);
  if (!ip) { return NIL; }
  if (!NILP(car)) {
    switch (car->type) {
    case sen_ql_object :
      if (car->class != slot->u.o.class) { return NIL; }
      *ip = car->u.o.self;
      break;
    case sen_ql_bulk :
      {
        sen_db_store *cls;
        if (!(cls = sen_db_store_by_id(slot->db, slot->u.o.class)) ||
            !(*ip = sen_sym_get(cls->u.c.keys, car->u.b.value))) {
          return NIL;
        }
      }
      break;
    default :
      return NIL;
      break;
    }
    // todo : trigger
  }
  if (!*ip) { return NIL; }
  if (!res) { res = sen_obj_new(c); }
  obj_obj_bind(res, slot->u.o.class, *ip);
  return res;
}

inline static sen_obj *
slot_value_ra(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  void *vp;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  vp = NILP(car) ? sen_ra_at(slot->u.f.ra, id) : sen_ra_get(slot->u.f.ra, id);
  if (!vp) { return NIL; }
  if (!NILP(car)) {
    switch (car->type) {
    case sen_ql_bulk :
      if (sizeof(int32_t) == slot->u.f.ra->header->element_size) {
        int32_t i = sen_atoi(car->u.b.value,
                             (char *)car->u.b.value + car->u.b.size, NULL);
        memcpy(vp, &i, sizeof(int32_t));
      } else {
        if (car->u.b.size != slot->u.f.ra->header->element_size) { return NIL; }
        memcpy(vp, car->u.b.value, car->u.b.size);
      }
      break;
    case sen_ql_int :
      if (sizeof(int32_t) != slot->u.f.ra->header->element_size) { return NIL; }
      memcpy(vp, &car->u.i.i, sizeof(int32_t));
      break;
    default :
      return NIL;
    }
  // todo : trigger
  }
  if (!res) { res = sen_obj_new(c); }
  if (slot->u.f.ra->header->element_size == sizeof(int32_t)) {
    res->type = sen_ql_int;
    memcpy(&res->u.i.i, vp, sizeof(int32_t));
  } else {
    res->type = sen_ql_bulk;
    res->u.b.size = slot->u.f.ra->header->element_size;
    res->u.b.value = vp;
  }
  return res;
}

inline static sen_obj *
slot_value_ja(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  void *vp;
  uint32_t value_len;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  vp = (void *)sen_ja_ref(slot->u.v.ja, id, &value_len);
  // todo : unref
  if (NILP(car)) {
    if (!vp) { return NIL; }
    if (!res) { res = sen_obj_new(c); }
    res->type = sen_ql_bulk;
    res->u.b.size = value_len;
    res->u.b.value = vp;
    return res;
  } else {
    sen_db_relation *t;
    // todo : support append and so on..
    if (!BULKP(car)) { return NIL; }
    if (value_len == car->u.b.size && !memcmp(vp, car->u.b.value, value_len)) {
      return car;
    }
    for (t = slot->relations; t; t = t->next) {
      if (t->type == sen_db_before_update_trigger) {
        sen_db_store *index = t->target;
        if (sen_index_upd(index->u.i.index, _sen_sym_key(index->u.i.index->keys, id),
                          vp, value_len, car->u.b.value, car->u.b.size)) {
          SEN_LOG(sen_log_error, "sen_index_upd failed. id=%d key=(%s) id'=%d", id, _sen_sym_key(index->u.i.index->keys, id), sen_sym_at(index->u.i.index->keys, _sen_sym_key(index->u.i.index->keys, id)));
        }
      }
    }
    return sen_ja_put(slot->u.v.ja, id, car->u.b.value, car->u.b.size, 0) ? NIL : car;
  }
}

inline static sen_obj *
slot_value(sen_ctx *c, sen_db_store *slot, sen_id obj, sen_obj *args, sen_obj *res)
{
  switch (slot->type) {
  case sen_db_obj_slot :
    return slot_value_obj(c, slot, obj, args, res);
    break;
  case sen_db_ra_slot :
    return slot_value_ra(c, slot, obj, args, res);
    break;
  case sen_db_ja_slot :
    return slot_value_ja(c, slot, obj, args, res);
    break;
  case sen_db_idx_slot :
    {
      sen_records *rec;
      const char *key = _sen_sym_key(slot->u.i.index->lexicon, obj);
      if (!key) { return NIL; }
      if (!(rec = sen_index_sel(slot->u.i.index, key, strlen(key)))) {
        return NIL;
      }
      if (!res) { res = sen_obj_new(c); }
      rec_obj_bind(res, rec, slot->u.i.class);
      return res;
    }
    break;
  default :
    return NIL;
    break;
  }
}

inline static char *
str_value(sen_obj *o)
{
  if (o->flags & SEN_OBJ_SYMBOL) {
    char *r = SEN_SET_STRKEY_BY_VAL(o);
    return *r == ':' ? r + 1 : r;
  } else if (o->type == sen_ql_bulk) {
    return o->u.b.value;
  }
  return NULL;
}

static sen_obj *
_native_method_object(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_db_store *slot;
  sen_obj *obj, *car, *res = c->code;
  POP(car, args);
  if (!(obj = c->code) || !(msg = str_value(car))) { return res; }
  if (*msg == ':') {
    if (msg[1] == 'k' || msg[1] == 'K') { /* :key */
      const char *key;
      sen_db_store *cls = sen_db_store_by_id(c->db, obj->class);
      if (!cls || !(key = _sen_sym_key(cls->u.c.keys, obj->u.o.self))) {
        return F;
      }
      res = sen_ql_mk_string(c, key, strlen(key));
    }
  } else {
    if (!(slot = sen_db_class_slot(c->db, obj->class, msg))) { return F; }
    res = slot_value(c, slot, obj->u.o.self, args, NULL);
  }
  return res;
}

sen_obj *
sen_ql_class_at(sen_ctx *c, sen_db_store *cls, const void *key, int flags, sen_obj *res)
{
  sen_id id = flags ? sen_sym_get(cls->u.c.keys, key) : sen_sym_at(cls->u.c.keys, key);
  if (id) {
    if (!res) { res = sen_obj_new(c); }
    obj_obj_bind(res, cls->id, id);
    return res;
  } else {
    return NIL;
  }
}

static sen_obj *
_native_method_void(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  if (!c->code) { return F; }
  return c->code;
}

static int
compar_ra(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  void *va, *vb;
  sen_id *pa, *pb;
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ra_at(raa, *pa);
  vb = sen_ra_at(rab, *pb);
  if (va) {
    if (vb) {
      if (raa->header->element_size == sizeof(int)) {
        // todo : support uint
        return *((int *)va) - *((int *)vb);
      } else {
        return memcmp(va, vb, raa->header->element_size);
      }
    } else {
      return 1;
    }
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_ja(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  int r;
  const void *va, *vb;
  uint32_t la, lb;
  sen_id *pa, *pb;
  sen_ja *jaa = (sen_ja *)ra->userdata, *jab = (sen_ja *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ja_ref(jaa, *pa, &la);
  vb = sen_ja_ref(jab, *pb, &lb);
  if (va) {
    if (vb) {
      if (la > lb) {
        if ((r = memcmp(va, vb, lb))) {
          return r;
        } else {
          return 1;
        }
      } else {
        if ((r = memcmp(va, vb, la))) {
          return r;
        } else {
          return la == lb ? 0 : -1;
        }
      }
    } else {
      return 1;
    }
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_key(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb;
  sen_sym *ka = ra->userdata, *kb = rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = _sen_sym_key(ka, *pa);
  vb = _sen_sym_key(kb, *pb);
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

// from index.c
typedef struct {
  int score;
  sen_id subid;
} subrec;

typedef struct {
  int score;
  int n_subrecs;
  subrec subrecs[1];
} recinfo;

/* todo : must be unified with scm.c:cons() */
inline static sen_obj *
cons(sen_ctx *c, sen_obj *a, sen_obj *b)
{
  sen_obj *o = sen_obj_new(c);
  o->type = sen_ql_list;
  o->flags = SEN_OBJ_REFERER;
  o->u.l.car = a;
  o->u.l.cdr = b;
  return o;
}

inline static sen_obj *
class_slot(sen_ctx *c, sen_id base, char *msg)
{
  sen_db_store *slot;
  char buf[SEN_SYM_MAX_KEY_SIZE];
  if (sen_db_class_slotpath(c->db, base, msg, buf)) { return F; }
  if (!(slot = sen_db_store_open(c->db, buf))) { return F; }
  return sen_ql_mk_symbol(c, buf);
}

static sen_rc
slotexp_prepare(sen_ctx *c, sen_id base, sen_obj *expr, sen_records *records)
{
  char *str;
  sen_obj *e, *slot, *r;
  if (!LISTP(expr)) { return sen_invalid_argument; }
  e = CAR(expr);
  if (LISTP(e)) {
    r = CAAR(e);
    if (SLOTP(r) || CLASSP(r)) { return sen_success; }
    for (r = NIL; LISTP(CAR(e)); e = CAR(e)) {
      if (LISTP(CDR(e))) { r = cons(c, CDR(e), r); }
    }
    if (!(str = str_value(CAR(e)))) { return sen_invalid_argument; }
    if (*str == '\0') {
      if (!records) { return sen_invalid_argument; }
      base = records->subrec_id;
      slot = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, base));
      if (!CLASSP(slot)) { return sen_invalid_argument; }
      r = cons(c, cons(c, slot, CDR(e)), r);
    } else {
      if ((slot = class_slot(c, base, str)) == F) { return sen_invalid_argument; }
      r = cons(c, cons(c, slot, CDR(e)), r);
      base = slot->class;
    }
    for (e = CDR(r); LISTP(e); e = CDR(e)) {
      if (!(str = str_value(CAAR(e))) ||
          (slot = class_slot(c, base, str)) == F) { return sen_invalid_argument; }
      CAR(e)->u.l.car = slot;
      base = slot->class;
    }
    expr->u.l.car = r;
  } else {
    if (!(str = str_value(e))) {
      return (e == T || e == NIL || e == F) ? sen_success : sen_invalid_argument;
    }
    if (*str == ':') {
      switch (str[1]) {
      case 'K' : /* :key */
      case 'k' :
        expr->u.l.car = NIL;
        break;
      case 'S' : /* :score */
      case 's' :
        if (!records) { return sen_invalid_argument; }
        expr->u.l.car = T;
        break;
      case 'N' : /* :nsubrecs */
      case 'n' :
        if (!records) { return sen_invalid_argument; }
        expr->u.l.car = F;
        break;
      default :
        return sen_invalid_argument;
        break;
      }
    } else {
      if ((slot = class_slot(c, base, str)) == F) { return sen_invalid_argument; }
      expr->u.l.car = cons(c, cons(c, slot, NIL), NIL);
    }
  }
  return sen_success;
}

static sen_obj *
slotexp_exec(sen_ctx *c, sen_obj *expr, sen_obj *value, recinfo *ri)
{
  sen_obj *t;
  sen_db_store *slot;
  if (expr == T) {
    value->type = sen_ql_int;
    value->u.i.i = ri->score;
    return value;
  }
  if (expr == F) {
    value->type = sen_ql_int;
    value->u.i.i = ri->n_subrecs;
    return value;
  }
  if (LISTP(expr)) {
    sen_obj *car;
    POP(t, expr);
    car = CAR(t);
    if (CLASSP(car)) {
      int i = 0;
      if (INTP(CADR(t))) { i = CADR(t)->u.i.i; }
      obj_obj_bind(value, car->u.o.self, ri->subrecs[i].subid);
    } else {
      slot = sen_db_store_by_id(c->db, car->u.o.self);
      value = slot_value(c, slot, value->u.o.self, CDR(t), value);
    }
  }
  while (value != NIL && LISTP(expr)) {
    POP(t, expr);
    if (!LISTP(t)) { break; }
    slot = sen_db_store_by_id(c->db, CAR(t)->u.o.self);
    value = slot_value(c, slot, value->u.o.self, CDR(t), value);
  }
  return value;
}

static int
compar_expr(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  sen_obj oa, ob, *va, *vb;
  sen_id *pa, *pb;
  recinfo *ria, *rib;
  sen_ctx *c = (sen_ctx *) arg;
  sen_obj *exa = (sen_obj *)ra->userdata, *exb = (sen_obj *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, (void **)&ria);
  oa.u.o.self = *pa;
  sen_set_element_info(rb->records, b, (void **)&pb, (void **)&rib);
  ob.u.o.self = *pb;

  va = slotexp_exec(c, exa, &oa, ria);
  vb = slotexp_exec(c, exb, &ob, rib);

  if (va == NIL) { return (vb == NIL) ? 0 : -1; }
  if (vb == NIL) { return 1; }

  if (va->type != vb->type) {
    SEN_LOG(sen_log_error, "obj type unmatch in compar_expr");
    return 0;
  }

  switch (va->type) {
  case sen_ql_object :
    {
      sen_db_store *ca, *cb;
      if (!(ca = sen_db_store_by_id(c->db, va->class)) ||
           (cb = sen_db_store_by_id(c->db, vb->class))) {
         SEN_LOG(sen_log_error, "clas open failed in compar_expr");
         return 0;
      }
      return strcmp(_sen_sym_key(ca->u.c.keys, va->u.o.self),
                    _sen_sym_key(cb->u.c.keys, vb->u.o.self));
    }
    break;
  case sen_ql_bulk :
    {
      int r;
      uint32_t la = va->u.b.size, lb = vb->u.b.size;
      if (la > lb) {
        if ((r = memcmp(va->u.b.value, vb->u.b.value, lb))) {
          return r;
        } else {
          return 1;
        }
      } else {
        if ((r = memcmp(va->u.b.value, vb->u.b.value, la))) {
          return r;
        } else {
          return la == lb ? 0 : -1;
        }
      }
    }
    break;
  case sen_ql_int :
    return va->u.i.i - vb->u.i.i;
    break;
  default :
    SEN_LOG(sen_log_error, "invalid value in compar_expr");
    break;
  }
  return 0;
}

static int
compar_obj(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb, *oa, *ob;
  sen_sym *key = (sen_sym *)arg;
  // todo : target class may not be identical
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = (oa = sen_ra_at(raa, *pa)) ? _sen_sym_key(key, *oa) : NULL;
  vb = (ob = sen_ra_at(rab, *pb)) ? _sen_sym_key(key, *ob) : NULL;
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

static int
group_obj(sen_records *ra, const sen_recordh *a, void *gkey, void *arg)
{
  sen_id *pa, *oa;
  sen_ra *raa = (sen_ra *)ra->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  if (!(oa = sen_ra_at(raa, *pa))) { return 1; }
  memcpy(gkey, oa, sizeof(sen_id));
  return 0;
}

static sen_obj *
_native_method_records(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  POP(car, args);
  if (!(msg = str_value(car))) { return res; }
  switch (*msg) {
  case '\0' : /* get instance by key */
    {
      char *name;
      sen_db_store *cls;
      POP(car, args);
      if (!(name = str_value(car))) { return F; }
      if (c->code->class) {
        cls = sen_db_store_by_id(c->db, c->code->class);
        res = sen_ql_class_at(c, cls, name, 0, NULL);
        if (!NILP(res) &&
            !sen_set_at(c->code->u.r.records->records, &res->u.o.self, NULL)) {
          res = NIL;
        }
      } else {
        res = sen_ql_at(c, name);
        if (!(res->flags & SEN_OBJ_NATIVE) ||
            !sen_set_at(c->code->u.r.records->records, &res->u.o.self, NULL)) {
          res = NIL;
        }
      }
    }
    break;
  case ':' :
    switch (msg[1]) {
    case 'd' : /* :difference */
    case 'D' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        if (LISTP(args)) {
          POP(car, args);
          if (RECORDSP(car)) {
            sen_records_difference(r, car->u.r.records);
          }
        }
      }
      break;
    case 'g' : /* :group */
    case 'G' :
      {
        char *str;
        int limit = 0;
        sen_db_store *slot;
        sen_group_optarg arg;
        sen_obj *rec = c->code;
        POP(car, args);
        if (!(str = str_value(car))) { break; }
        if (!(slot = sen_db_class_slot(c->db, rec->class, str))) { break; }
        if (slot->type != sen_db_obj_slot) { break; } // todo : support others
        rec->u.r.records->userdata = slot->u.o.ra;
        arg.mode = sen_sort_descending;
        arg.func = group_obj;
        arg.func_arg = NULL;
        arg.key_size = sizeof(sen_id);
        POP(car, args);
        if (!sen_obj2int(car)) { limit = car->u.i.i; }
        POP(car, args);
        if ((str = str_value(car)) && (*str == 'a')) {
          arg.mode = sen_sort_ascending;
        }
        if (!sen_records_group(rec->u.r.records, limit, &arg)) {
          rec->u.r.records->subrec_id = rec->class;
          rec->class = slot->u.o.class;
          // rec->u.r.records->keys = sen_db_store_by_id(c->store, slot->u.o.class)->u.c.keys;
          res = rec;
        }
      }
      break;
    case 'i' : /* :intersect */
    case 'I' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        while (LISTP(args)) {
          POP(car, args);
          if (!RECORDSP(car)) { continue; }
          sen_records_intersect(r, car->u.r.records);
          car->type = sen_ql_void;
          car->u.o.func = _native_method_void;
          car->flags &= ~SEN_OBJ_ALLOCATED;
        }
      }
      break;
    case 'n' : /* :nrecs */
    case 'N' :
      res = sen_obj_new(c);
      res->type = sen_ql_int;
      res->u.i.i = sen_records_nhits(c->code->u.r.records);
      break;
    case 's' :
    case 'S' :
      {
        switch (msg[2]) {
        case 'o' : /* :sort */
        case 'O' :
          {
            int limit = 10;
            const char *str;
            sen_sort_optarg arg;
            sen_obj *rec = c->code;
            arg.compar = NULL;
            arg.compar_arg = (void *)(intptr_t)rec->u.r.records->record_size;
            arg.mode = sen_sort_descending;
            if ((str = str_value(CAR(args)))) {
              if (*str == ':') {
                switch (str[1]) {
                case 's' : /* :score */
                  break;
                case 'k' : /* :key */
                  if (rec->class) {
                    sen_db_store *cls = sen_db_store_by_id(c->db, rec->class);
                    if (cls) {
                      rec->u.r.records->userdata = cls->u.c.keys;
                      arg.compar = compar_key;
                    }
                  } else {
                    rec->u.r.records->userdata = c->db->keys;
                    arg.compar = compar_key;
                  }
                  break;
                }
              } else {
                sen_db_store *slot = sen_db_class_slot(c->db, rec->class, str);
                if (slot) {
                  switch (slot->type) {
                  case sen_db_ra_slot :
                    rec->u.r.records->userdata = slot->u.f.ra;
                    arg.compar = compar_ra;
                    break;
                  case sen_db_ja_slot :
                    rec->u.r.records->userdata = slot->u.v.ja;
                    arg.compar = compar_ja;
                    break;
                  case sen_db_obj_slot :
                    {
                      sen_db_store *cls = sen_db_store_by_id(c->db, slot->u.o.class);
                      if (cls) {
                        rec->u.r.records->userdata = slot->u.o.ra;
                        arg.compar = compar_obj;
                        arg.compar_arg = cls->u.c.keys;
                      }
                    }
                    break;
                  default :
                    break;
                  }
                }
              }
            } else {
              if (slotexp_prepare(c, rec->class, args, rec->u.r.records)) {
                res = F;
                break;
              }
              rec->u.r.records->userdata = CAR(args);
              arg.compar = compar_expr;
              arg.compar_arg = c;
            }
            POP(car, args);
            POP(car, args);
            if (!sen_obj2int(car)) { limit = car->u.i.i; }
            POP(car, args);
            if ((str = str_value(car)) && *str == 'a') {
              arg.mode = sen_sort_ascending;
            }
            if (!sen_records_sort(rec->u.r.records, limit, &arg)) { res = rec; }
          }
          break;
        case 'u' : /* :subtract */
        case 'U' :
          {
            sen_records *r = c->code->u.r.records;
            res = c->code;
            while (LISTP(args)) {
              POP(car, args);
              if (!RECORDSP(car)) { continue; }
              sen_records_subtract(r, car->u.r.records);
              car->type = sen_ql_void;
              car->u.o.func = _native_method_void;
              car->flags &= ~SEN_OBJ_ALLOCATED;
            }
          }
          break;
        default :
          {
            /* ambiguous message. todo : return error */
            res = F;
          }
        }
      }
      break;
    case 'u' : /* :union */
    case 'U' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        while (LISTP(args)) {
          POP(car, args);
          if (!RECORDSP(car)) { continue; }
          sen_records_union(r, car->u.r.records);
          car->type = sen_ql_void;
          car->u.o.func = _native_method_void;
          car->flags &= ~SEN_OBJ_ALLOCATED;
        }
      }
      break;
    case '+' : /* :+ (iterator next) */
      {
        sen_id *rid;
        sen_records *r = c->code->u.r.records;
        if (c->code->class) {
          POP(res, args);
          if (res->type == sen_ql_object && res->class == c->code->class) {
            if (sen_records_next(r, NULL, 0, NULL)) {
              sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
              res->u.o.self = *rid;
            } else {
              res->type = sen_ql_void;
              res->u.o.func = _native_method_void;
            }
          }
        } else {
          if (sen_records_next(r, NULL, 0, NULL)) {
            sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
            res = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, *rid));
          } else {
            res = NIL;
          }
        }
      }
      break;
    case '\0' : /* : (iterator begin) */
      {
        sen_id *rid;
        sen_records *r = c->code->u.r.records;
        sen_records_rewind(r);
        if (sen_records_next(r, NULL, 0, NULL)) {
          sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
          if (c->code->class) {
            res = sen_obj_new(c);
            obj_obj_bind(res, c->code->class, *rid);
          } else {
            res = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, *rid));
          }
        } else {
          res = NIL;
        }
      }
      break;
    }
    break;
  default : /* invalid message */
    res = F;
    break;
  }
  return res;
}

inline static sen_obj *
rec_obj_new(sen_ctx *c, sen_db_store *cls, sen_rec_unit record_unit,
            sen_rec_unit subrec_unit, unsigned int max_n_subrecs)
{
  sen_records *r;
  sen_obj *res;
  if (!(r = sen_records_open(record_unit, subrec_unit, max_n_subrecs))) { return NULL; }
  r->keys = cls->u.c.keys;
  res = sen_obj_new(c);
  rec_obj_bind(res, r, cls->id);
  return res;
}

inline static sen_obj *
query_obj_new(sen_ctx *c, const char *str, unsigned int str_len,
              sen_sel_operator default_op, int max_exprs, sen_encoding encoding)
{
  sen_query *q;
  sen_obj *res;
  if (!(q = sen_query_open(str, str_len, default_op, max_exprs, encoding))) {
    return NULL;
  }
  res = sen_obj_new(c);
  res->type = sen_ql_query;
  res->flags = SEN_OBJ_ALLOCATED;
  res->u.q.query = q;
  return res;
}

typedef struct {
  sen_db_store *slot;
  sen_obj *expr;
  sen_obj *value;
  sen_obj *res;
  sen_obj buf;
} match_spec;

inline static sen_obj*
match_prepare(sen_ctx *c, match_spec *spec, sen_id base, sen_obj *expr)
{
  sen_obj *car, *res;
  char *op, *name;
  if (!LISTP(expr)) { return NIL; }
  POP(car, expr);
  if (!(op = str_value(car))) { return NIL; }
  if (*op != 'e') { return NIL; } // todo : support other operators
  if ((name = str_value(CAR(expr)))) {
    if (!(spec->slot = sen_db_class_slot(c->db, base, name))) { return NIL; }
    spec->expr = NULL;
  } else {
    sen_obj *e;
    if (slotexp_prepare(c, base, expr, NULL)) { return NIL; }
    spec->expr = CAR(expr);
    for (e = spec->expr; CDR(e) != NIL; e = CDR(e));
    spec->slot = sen_db_store_by_id(c->db, CAAR(e)->u.o.self);
  }
  POP(car, expr);
  POP(car, expr);
  switch (spec->slot->type) {
  case sen_db_obj_slot :
    switch (car->type) {
    case sen_ql_object :
      if (car->class != spec->slot->u.o.class) { return NIL; }
      spec->value = car;
      break;
    case sen_ql_bulk :
      {
        sen_id si;
        sen_db_store *sc;
        if (!(sc = sen_db_store_by_id(spec->slot->db, spec->slot->u.o.class)) ||
            !(si = sen_sym_at(sc->u.c.keys, car->u.b.value))) {
          return NIL;
        }
        spec->value = &spec->buf;
        obj_obj_bind(spec->value, spec->slot->u.o.class, si);
      }
      break;
    default :
      return NIL;
      break;
    }
    break;
  case sen_db_ra_slot :
    if (spec->slot->u.f.ra->header->element_size == sizeof(int32_t)) {
      if (sen_obj2int(car)) { return NIL; }
      spec->value = car;
      break;
    }
    /* fall through */
  case sen_db_ja_slot :
    if (!BULKP(car)) { return NIL; }
    spec->value = car;
    break;
  case sen_db_idx_slot :
    return NIL;
    break;
  default :
    return NIL;
    break;
  }
  POP(res, expr);
  if (RECORDSP(res)) {
    /* todo : support operator */
  } else {
    sen_db_store *cls;
    cls = sen_db_store_by_id(c->db, base);
    if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
      return NIL;
    }
  }
  spec->res = res;
  return res;
}

/* todo : integrate scm.c:eqv() */
inline static int
eqv(sen_obj *a, sen_obj *b)
{
  if (a->type != b->type) { return 0; }
  switch (a->type) {
  case sen_ql_object :
    return (a->class == b->class && a->u.o.self == b->u.o.self);
    break;
  case sen_ql_bulk :
    return (a->u.b.size == b->u.b.size &&
            !memcmp(a->u.b.value, b->u.b.value, a->u.b.size));
    break;
  case sen_ql_int :
    return (a->u.i.i == b->u.i.i);
    break;
  default :
    /* todo : support other types */
    return 0;
    break;
  }
}

inline static int
match_exec(sen_ctx *c, match_spec *spec, sen_id id)
{
  sen_obj buf, *value;
  if (spec->expr) {
    buf.u.o.self = id;
    value = slotexp_exec(c, spec->expr, &buf, NULL);
  } else {
    value = slot_value(c, spec->slot, id, NIL, &buf);
  }
  if (NILP(value) || !eqv(spec->value, value)) {
    return 0;
  }
  sen_set_get(spec->res->u.r.records->records, &id, NULL);
  return 1;
}


struct _ins_stat {
  sen_obj *slots;
  int nslots;
  int nrecs;
};

// todo : refine
#define MAXSLOTS 0x100

static sen_obj *
_native_method_class(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_id base;
  int load = 0;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  base = c->code->u.o.self;
  SEN_QL_CO_BEGIN(co);
  POP(car, args);
  if (!(msg = str_value(car))) { return res; }
  switch (*msg) {
  case '\0' : /* get instance by key */
    {
      char *name;
      sen_db_store *cls;
      POP(car, args);
      if (!(name = str_value(car))) { return F; }
      cls = sen_db_store_by_id(c->db, base);
      res = sen_ql_class_at(c, cls, name, 0, NULL);
    }
    break;
  case ':' :
    switch (msg[1]) {
    case 'a' : /* :add (define slot) */
    case 'A' :
      {
        char *name;
        sen_id tcls;
        sen_db_store *slot;
        sen_db_store_spec spec;
        sen_obj *cdr, *targets = NULL;

        POP(car, args);
        if (!(name = str_value(car))) { return F; }
        if (sen_db_class_slot(c->db, base, name)) { return T; /* already exists */ }
        POP(car, args);
        tcls = car->u.o.self;
        spec.u.s.class = tcls;
        spec.u.s.size = 0;
        spec.u.s.collection_type = 0;
        switch (car->type) {
        case sen_db_raw_class :
          {
            sen_db_store *cls = sen_db_store_by_id(c->db, tcls);
            if (!cls) { return F; }
            spec.type = (cls->u.bc.element_size > 8) ? sen_db_ja_slot : sen_db_ra_slot;
            spec.u.s.size = cls->u.bc.element_size;
          }
          break;
        case sen_db_class :
          spec.type = sen_db_obj_slot;
          break;
        case sen_db_obj_slot :
        case sen_db_ra_slot :
        case sen_db_ja_slot :
          spec.type = sen_db_idx_slot;
          break;
        default :
          return F;
        }
        for (cdr = args; LISTP(cdr); cdr = CDR(cdr)) {
          char *opt = str_value(CAR(cdr));
          if (opt) {
            switch (*opt) {
            case 'a' :
              spec.u.s.collection_type = 1; // todo : array
              break;
            case 's' :
              spec.u.s.collection_type = 2; // todo : set
              break;
            case 'i' :
              spec.type = sen_db_idx_slot;
              spec.u.s.collection_type = 3;
            default :
              if (!sen_obj2int(CAR(cdr))) { spec.u.s.size = CAR(cdr)->u.i.i; }
              break;
            }
          }
          if (LISTP(CAR(cdr))) { targets = CAR(cdr); }
        }
        {
          char buf[SEN_SYM_MAX_KEY_SIZE];
          if (sen_db_class_slotpath(c->db, base, name, buf)) { return F; }
          if (!(slot = sen_db_store_create(c->db, buf, &spec))) { return F; }
          if (spec.type == sen_db_idx_slot) {
            sen_db_store *ts;
            sen_db_store_rel_spec rs;
            rs.type = sen_db_index_target;
            while (LISTP(targets)) {
              char *tsname = str_value(CAR(targets));
              if (tsname) {
                if ((ts = sen_db_class_slot(c->db, tcls, tsname))) {
                  rs.target = ts->id;
                  sen_db_store_add_relation(slot, &rs);
                }
              }
              targets = CDR(targets);
            }
          }
          res = sen_ql_mk_symbol(c, buf);
          sen_ql_bind_symbol(slot, res);
        }
      }
      break;
    case 'c' : /* :common-prefix-search */
    case 'C' :
      {
        char *name;
        sen_id id;
        sen_db_store *cls = sen_db_store_by_id(c->db, base);
        POP(car, args);
        if (!(name = str_value(car))) { return F; }
        if ((id = sen_sym_common_prefix_search(cls->u.c.keys, name))) {
          if ((res = sen_obj_new(c))) {
            obj_obj_bind(res, base, id);
          }
        } else {
          res = NIL;
        }
      }
      break;
    case 'd' : /* :delete */
    case 'D' :
      // todo : delete
      break;
    case 'l' : /* :load */
    case 'L' :
      load = 1;
      break;
    case 'n' :
    case 'N' :
      {
        sen_db_store *cls;
        switch (msg[2]) {
        case 'e' : /* :new */
        case 'E' :
          {
            char *name;
            POP(car, args);
            if (!(name = str_value(car))) { return F; }
            cls = sen_db_store_by_id(c->db, base);
            res = sen_ql_class_at(c, cls, name, 1, NULL);
            if (!NILP(res)) {
              sen_obj cons, dummy;
              sen_db_store *slot;
              cons.type = sen_ql_list;
              cons.flags = SEN_OBJ_REFERER;
              cons.u.l.cdr = NIL;
              while (LISTP(args)) {
                POP(car, args);
                if (!(msg = str_value(car))) { break; }
                POP(car, args);
                if (NILP(car)) { break; }
                if (!(slot = sen_db_class_slot(c->db, base, msg))) { break; }
                cons.u.l.car = car;
                slot_value(c, slot, res->u.o.self, &cons, &dummy);
              }
            }
          }
          break;
        case 'r' : /* :nrecs */
        case 'R' :
          {
            cls = sen_db_store_by_id(c->db, base);
            res = sen_obj_new(c);
            res->type = sen_ql_int;
            res->u.i.i = sen_sym_size(cls->u.c.keys);
          }
          break;
        default :
          {
            /* ambiguous message. todo : return error */
            res = F;
          }
        }
      }
      break;
    case 'p' : /* :prefix-search */
    case 'P' :
      {
        char *name;
        sen_db_store *cls = sen_db_store_by_id(c->db, base);
        POP(car, args);
        if (!(name = str_value(car))) { return F; }
        if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
          return NIL;
        }
        sen_sym_prefix_search_with_set(cls->u.c.keys, name, res->u.r.records->records);
      }
      break;
    case 'r' : /* :remove-slot */
    case 'R' :
      // todo : remove
      break;
    case 's' :
    case 'S' :
      switch (msg[2]) {
      case 'c' : /* :scan-select */
      case 'C' :
        {
          match_spec spec;
          sen_db_store *cls = sen_db_store_by_id(c->db, base);
          sen_id id = SEN_SYM_NIL, maxid = sen_sym_curr_id(cls->u.c.keys);
          res = match_prepare(c, &spec, base, CAR(args));
          if (NILP(res)) { break; }
          while (++id <= maxid) { match_exec(c, &spec, id); }
        }
        break;
      case 'u' : /* :suffix-search */
      case 'U' :
        {
          char *name;
          sen_db_store *cls = sen_db_store_by_id(c->db, base);
          POP(car, args);
          if (!(name = str_value(car))) { return F; }
          if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
            return NIL;
          }
          sen_sym_suffix_search_with_set(cls->u.c.keys, name, res->u.r.records->records);
        }
        break;
      case 'l' : /* :slots */
      case 'L' :
        {
          char *name;
          char buf[SEN_SYM_MAX_KEY_SIZE];
          POP(car, args);
          if (!(name = str_value(car))) { name = ""; }
          if (sen_db_class_slotpath(c->db, base, name, buf)) { return NIL; }
          {
            sen_records *r;
            if (!(r = sen_records_open(sen_rec_document, sen_rec_none, 0))) {
              return NIL;
            }
            r->keys = c->db->keys;
            res = sen_obj_new(c);
            rec_obj_bind(res, r, 0);
          }
          sen_sym_prefix_search_with_set(c->db->keys, buf, res->u.r.records->records);
        }
        break;
      }
      break;
    case '+' : /* :+ (iterator next) */
      {
        sen_db_store *cls;
        cls = sen_db_store_by_id(c->db, base);
        POP(res, args);
        if (res->type == sen_ql_object && res->class == cls->id) {
          res->u.o.self = sen_sym_next(cls->u.c.keys, res->u.o.self);
          if (res->u.o.self == SEN_SYM_NIL) {
            res->type = sen_ql_void;
            res->u.o.func = _native_method_void;
          }
        }
      }
      break;
    case '\0' : /* : (iterator begin) */
      {
        sen_id id;
        sen_db_store *cls;
        cls = sen_db_store_by_id(c->db, base);
        id = sen_sym_next(cls->u.c.keys, SEN_SYM_NIL);
        if (id == SEN_SYM_NIL) {
          res = NIL;
        } else {
          res = sen_obj_new(c);
          obj_obj_bind(res, cls->id, id);
        }
      }
      break;
    }
    break;
  default : /* :slotname */
    res = class_slot(c, base, msg);
    break;
  }
  if (load) {
    int i;
    sen_obj *s;
    struct _ins_stat *stat;
    for (s = args, i = 0; LISTP(s); s = CDR(s), i++) {
      car = CAR(s);
      if (!(msg = str_value(car))) { return F; }
      if ((s->u.l.car = class_slot(c, base, msg)) == F) { return F; }
    }
    if (!(s = sen_obj_alloc(c, sizeof(struct _ins_stat)))) { /* todo */ }
    stat = (struct _ins_stat *)s->u.b.value; // todo : not GC safe
    stat->slots = args;
    stat->nslots = i + 1;
    stat->nrecs = 0;
    do {
      SEN_QL_CO_WAIT(co, stat);
      if (BULKP(args) && args->u.b.size) {
        char *tokbuf[MAXSLOTS];
        sen_db_store *cls, *slot;
        sen_obj val, obj, cons, dummy;
        cons.type = sen_ql_list;
        cons.flags = SEN_OBJ_REFERER;
        cons.u.l.car = &val;
        cons.u.l.cdr = NIL;
        val.type = sen_ql_bulk;
        if (sen_str_tok(args->u.b.value, args->u.b.size, '\t', tokbuf, MAXSLOTS, NULL) == stat->nslots) {
          sen_obj *o;
          cls = sen_db_store_by_id(c->db, base);
          *tokbuf[0] = '\0';
          o = sen_ql_class_at(c, cls, args->u.b.value, 1, &obj);
          if (NILP(o)) { continue; }
          for (s = stat->slots, i = 1; i < stat->nslots; s = CDR(s), i++) {
            val.u.b.value = tokbuf[i - 1] + 1;
            val.u.b.size = tokbuf[i] - val.u.b.value;
            if (!(slot = sen_db_store_by_id(c->db, CAR(s)->u.o.self))) { /* todo */ }
            slot_value(c, slot, obj.u.o.self, &cons, &dummy); // todo : refine cons
          }
          stat->nrecs++;
        }
      } else {
        co->mode |= SEN_QL_TAIL;
      }
    } while (!(co->mode & (SEN_QL_HEAD|SEN_QL_TAIL)));
    res = sen_obj_new(c);
    res->type = sen_ql_int;
    res->u.i.i = stat->nrecs;
  }
  SEN_QL_CO_END(co);
  return res;
}

static sen_obj *
_native_method_slot(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_id base;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  base = c->code->u.o.self;
  POP(car, args);
  if (!(msg = str_value(car))) { return res; }
  switch (*msg) {
  case '\0' :
    {
      char *name;
      sen_db_store *cls, *slot;
      POP(car, args);
      if (!(name = str_value(car))) { return F; }
      if (!(slot = sen_db_store_by_id(c->db, base))) { return F; }
      if (!(cls = sen_db_slot_class_by_id(c->db, base))) { return F; }
      res = sen_ql_class_at(c, cls, name, 0, NULL);
      if (!NILP(res)) {
        res = slot_value(c, slot, res->u.o.self, args, res);
      }
    }
    break;
  case ':' :
    if (msg[1] == 'i' || msg[1] == 'I') { /* :index-search */
      sen_obj *q;
      sen_db_store *slot;
      sen_sel_operator op = sen_sel_or;
      if (!IDX_SLOTP(c->code)) { break; }
      POP(q, args);
      if (!QUERYP(q)) {
        if (!BULKP(q)) { return F; }
        q = query_obj_new(c, q->u.b.value, q->u.b.size, sen_sel_and, 32, c->encoding);
      }
      /* TODO: specify record unit */
      /* (idxslot query ((slot1 weight1) (slot2 weight2) ...) records operator+ */
      if (!(slot = sen_db_store_by_id(c->db, c->code->u.o.self))) { return F; }
      POP(car, args);
      /* TODO: handle weights */
      POP(res, args);
      if (RECORDSP(res)) {
        char *ops;
        POP(car, args);
        if ((ops = str_value(car))) {
          switch (*ops) {
          case '+': op = sen_sel_or; break;
          case '-': op = sen_sel_but; break;
          case '*': op = sen_sel_and; break;
          case '>': op = sen_sel_adjust; break;
          }
        }
      } else {
        sen_db_store *cls;
        if (!(cls = sen_db_store_by_id(c->db, slot->u.i.class)) ||
            !(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
          return F;
        }
      }
      sen_query_exec(slot->u.i.index, q->u.q.query, res->u.r.records, op);
    }
    break;
  }
  return res;
}

void
sen_ql_bind_symbol(sen_db_store *dbs, sen_obj *symbol)
{
  symbol->type = dbs->type;
  symbol->flags |= SEN_OBJ_NATIVE;
  symbol->u.o.self = dbs->id;
  switch (symbol->type) {
  case sen_db_class :
    symbol->u.o.func = _native_method_class;
    symbol->class = 0;
    break;
  case sen_db_obj_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.o.class;
    break;
  case sen_db_ra_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.f.class;
    break;
  case sen_db_ja_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.v.class;
    break;
  case sen_db_idx_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.i.class;
    break;
  default :
    symbol->u.o.func = _native_method_void;
    symbol->class = 0;
    break;
  }
}

static sen_obj *
_native_method_sen_query(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  sen_obj *o = NULL, *s = CAR(args);
  /* TODO: placeholder */
  if (BULKP(s)) {
    /* TODO: operator, exprs, encoding */
    o = query_obj_new(c, s->u.b.value, s->u.b.size, sen_sel_and, 32, c->encoding);
  }
  return o;
}

static sen_obj *
_native_method_sen_snip(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  /* FIXME: use uint instead of int */
  /* args: (str@bulk width@int max_results@int cond1@list cond2@list ...) */
  /* cond: (keyword@bulk [opentag@bulk closetag@bulk]) */
  sen_obj *ret = NULL, *str = CAR(args), *cur = CDR(args);
  if (BULKP(str)) {
    sen_snip *s;
    unsigned int width = 10, max_results;
    if (!LISTP(cur) || sen_obj2int(CAR(cur))) {
      return NIL;
    }
    width = CAR(cur)->u.i.i;
    cur = CDR(cur);
    if (!LISTP(cur) || sen_obj2int(CAR(cur))) {
      return NIL;
    }
    max_results = CAR(cur)->u.i.i;
    cur = CDR(cur);
    /* FIXME: mapping */
    if (!(s = sen_snip_open(c->encoding, SEN_SNIP_NORMALIZE, width, max_results,
                            NULL, 0, NULL, 0, (sen_snip_mapping *)-1))) {
      return NIL;
    }
    for (; LISTP(cur); cur = CDR(cur)) {
      if (LISTP(CAR(cur))) {
        sen_obj *cl = CAR(cur), *kw = CAR(cl);
        if (BULKP(kw)) {
          char *ot = NULL, *ct = NULL;
          uint32_t ot_l = 0, ct_l = 0;
          if (BULKP(CADR(cl))) {
            ot = CADR(cl)->u.b.value;
            ot_l = CADR(cl)->u.b.size;
            if (BULKP(CADDR(cl))) {
              ct = CADDR(cl)->u.b.value;
              ct_l = CADDR(cl)->u.b.size;
            }
          }
          if (!(sen_snip_add_cond(s, kw->u.b.value, kw->u.b.size,
                                  ot, ot_l, ct, ct_l))) {
            /* TODO: error handling */
          }
        }
      }
    }
    {
      unsigned int max_len, nresults;
      if (!(sen_snip_exec(s, str->u.b.value, str->u.b.size, &nresults, &max_len))) {
        if ((ret = sen_obj_alloc(c, sizeof(char) * max_len * nresults))) {
          unsigned int i, tlen = 0;
          for (i = 0; i < nresults; i++) {
            unsigned int len;
            if (!(sen_snip_get_result(s, i, ret->u.b.value + tlen, &len))) {
              /* TODO: concat with specified string */
              tlen += len;
            }
          }
          ret->u.b.size = tlen;
        }
      }
    }
    sen_snip_close(s);
  }
  return ret;
}

static sen_obj *
_native_method_db(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_db_store *cls;
  sen_obj *car, *res = c->code;
  POP(car, args);
  if (!(msg = str_value(car))) { return res; }
  if (*msg == ':') {
    switch (msg[1]) {
    case 'd' : /* :drop */
    case 'D' :
      {
        // todo : drop
      }
      break;
    case 'n' : /* :new (define class) */
    case 'N' :
      {
        char *name;
        sen_obj *cdr;
        sen_db_store_spec spec;
        spec.type = sen_db_class;
        spec.u.c.size = 0;
        spec.u.c.flags = SEN_INDEX_NORMALIZE;
        spec.u.c.encoding = c->encoding;
        POP(car, args);
        if (!(name = str_value(car))) { return F; }
        if (sen_db_store_open(c->db, name)) { return T; /* already exists */ }
        for (cdr = args; LISTP(cdr); cdr = CDR(cdr)) {
          char *opt = str_value(CAR(cdr));
          if (opt) {
            switch (*opt) {
            case 'd' :
            case 'D' :
              spec.u.c.flags |= SEN_INDEX_DELIMITED;
              break;
            case 'e' :
            case 'E' :
              spec.u.c.encoding = sen_enc_euc_jp;
              break;
            case 'k' :
            case 'K' :
              spec.u.c.encoding = sen_enc_koi8r;
              break;
            case 'l' :
            case 'L' :
              spec.u.c.encoding = sen_enc_latin1;
              break;
            case 'n' :
            case 'N' :
              spec.u.c.flags |= SEN_INDEX_NGRAM;
              break;
            case 'r' :
            case 'R' :
              spec.type = sen_db_raw_class;
              break;
            case 's' :
            case 'S' :
              switch (opt[1]) {
              case 'j' :
              case 'J' :
                spec.u.c.encoding = sen_enc_sjis;
                break;
              case 'i' :
              case 'I' :
                spec.u.c.flags |= SEN_SYM_WITH_SIS;
                break;
              }
              break;
            case 'u' :
            case 'U' :
              spec.u.c.encoding = sen_enc_utf8;
              break;
            default :
              if (!sen_obj2int(CAR(cdr))) { spec.u.c.size = CAR(cdr)->u.i.i; }
              break;
              // todo : without key mode
            }
          }
        }
        if (!(cls = sen_db_store_create(c->db, name, &spec))) { return F; }
        res = sen_ql_mk_symbol(c, name);
        sen_ql_bind_symbol(cls, res);
      }
      break;
    case 'p' : /* :prefix-search */
    case 'P' :
      {
        char *name;
        POP(car, args);
        if (!(name = str_value(car))) { return F; }
        {
          sen_records *r;
          if (!(r = sen_records_open(sen_rec_document, sen_rec_none, 0))) {
            return NIL;
          }
          r->keys = c->db->keys;
          res = sen_obj_new(c);
          rec_obj_bind(res, r, 0);
        }
        sen_sym_prefix_search_with_set(c->db->keys, name, res->u.r.records->records);
        {
          sen_id *rid;
          sen_set_eh *eh;
          sen_set_cursor *sc = sen_set_cursor_open(res->u.r.records->records);
          while ((eh = sen_set_cursor_next(sc, (void **) &rid, NULL))) {
            if (strchr(_sen_sym_key(c->db->keys, *rid), '.')) {
              sen_set_del(res->u.r.records->records, eh);
            }
          }
          sen_set_cursor_close(sc);
        }
      }
      break;
    }
  }
  return res;
}

const char *
_sen_obj_key(sen_db *db, sen_obj *obj)
{
  sen_db_store *cls;
  switch (obj->type) {
  case sen_ql_object :
    if (!(cls = sen_db_store_by_id(db, obj->class))) { return NULL; }
    return _sen_sym_key(cls->u.c.keys, obj->u.o.self);
  case sen_db_raw_class :
  case sen_db_class :
  case sen_db_obj_slot :
  case sen_db_ra_slot :
  case sen_db_ja_slot :
  case sen_db_idx_slot :
    return _sen_sym_key(db->keys, obj->u.o.self);
  default :
    return NULL;
  }
}

/* todo : remove? */
inline static sen_ql_method_func *
name2method(sen_ctx *c, const char *name)
{
  sen_obj *obj = sen_ql_at(c, name);
  if (!NATIVE_METHODP(obj)) { return NULL; }
  return obj->u.o.func;
}

static sen_obj *
_native_method_sen_output(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  sen_rbuf buf;
  sen_obj result, obj, *car, *fargs;
  if (sen_rbuf_init(&buf, 1)) { return F; }
  memset(&result, 0, sizeof(sen_obj));
  POP(car, args);
  switch (car->type) {
  case sen_ql_records :
    {
      sen_id *rp, base;
      recinfo *ri;
      sen_obj *slots, *s, *t, *v;
      const sen_recordh *rh;
      int i, offset = 0, limit = 10;
      sen_records *r = car->u.r.records;
      base = car->class;
      sen_rbuf_itoa(&buf, sen_records_nhits(r));
      c->output(c, &buf, SEN_QL_MORE);
      POP(slots, args);
      if (!LISTP(slots)) { goto exit; }
      for (s = slots; LISTP(s); s = CDR(s)) {
        if (slotexp_prepare(c, base, s, r)) { goto exit; }
      }
      POP(car, args);
      if (!sen_obj2int(car)) { offset = car->u.i.i; }
      POP(car, args);
      if (!sen_obj2int(car)) { limit = car->u.i.i; }
      sen_records_rewind(r);
      for (i = 0; i < offset; i++) {
        if (!sen_records_next(r, NULL, 0, NULL)) { goto exit; }
      }
      for (i = 0; i < limit; i++) {
        if (!sen_records_next(r, NULL, 0, NULL) ||
            !(rh = sen_records_curr_rec(r)) ||
            sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
          goto exit;
        }
        SEN_RBUF_REWIND(&buf);
        for (s = slots;;) {
          POP(t, s);
          obj_obj_bind(&obj, base, *rp);
          v = slotexp_exec(c, t, &obj, ri);
          sen_obj_inspect(c, v, &buf, SEN_OBJ_INSPECT_SYM);
          if (!LISTP(s)) { break; }
          SEN_RBUF_PUTC(&buf, '\t');
        }
        c->output(c, &buf, SEN_QL_MORE);
      }
    }
    break;
  case sen_ql_object :
    {
      sen_obj *slots;
      POP(slots, args);
      if (!LISTP(slots)) { goto exit; }
      // check && compile slots
      for (;;) {
        char *value;
        sen_obj *t;
        sen_ql_method_func *func = NULL;
        POP(t, slots);
        if (LISTP(t) && LISTP(CDR(t))) {
          char *fname = str_value(CADR(t));
          if (fname) { func = name2method(c, fname); }
          fargs = CDDR(t);
          t = CAR(t);
        }
        if ((value = str_value(t))) {
          if (*value == ':') {
            switch (value[1]) {
            case 'k' :
              {
                const char *key = _sen_obj_key(c->db, car);
                if (key) { SEN_RBUF_PUTS(&buf, key); }
              }
              break;
            default :
              break;
            }
          } else {
            sen_obj *res = &result;
            sen_db_store *slot = sen_db_class_slot(c->db, car->class, value);
            if (!slot) { break; }
            res = slot_value(c, slot, car->u.o.self, NIL, res);
            if (NILP(res)) { break; }
            if (func) {
              sen_obj param;
              param.type = sen_ql_list;
              param.u.l.car = res;
              param.u.l.cdr = fargs;
              res = func(c, &param, NULL);
              if (NILP(res)) { break; }
            }
            if (res->flags & SEN_OBJ_SYMBOL) {
              char *r = SEN_SET_STRKEY_BY_VAL(res);
              SEN_RBUF_PUTS(&buf, (*r == ':') ? r + 1 : r);
            } else {
              switch(res->type) {
              case sen_ql_object :
                {
                  const char *key = _sen_obj_key(c->db, res);
                  if (key) { SEN_RBUF_PUTS(&buf, key); }
                }
                break;
              case sen_ql_bulk :
                sen_rbuf_write(&buf, res->u.b.value, res->u.b.size);
                break;
              case sen_ql_int :
                sen_rbuf_itoa(&buf, res->u.i.i);
                break;
              default :
                break;
              }
            }
          }
        }
        if (!LISTP(slots)) { break; }
        SEN_RBUF_PUTC(&buf, '\t');
      }
      c->output(c, &buf, SEN_QL_MORE);
    }
    break;
  case sen_ql_list :
    {
      sen_obj *list;
      for (list = car;;) {
        sen_obj *res;
        sen_ql_method_func *func = NULL;
        POP(res, list);
        if (LISTP(res) && LISTP(CDR(res))) {
          char *fname = str_value(CADR(res));
          if (fname) { func = name2method(c, fname); }
          func = name2method(c, fname);
          if (func) {
            sen_obj param;
            param.type = sen_ql_list;
            param.u.l.car = CAR(res);
            param.u.l.cdr = CDDR(res);
            if (!(res = func(c, &param, NULL))) {
              break;
            }
          }
        }
        if (res->flags & SEN_OBJ_SYMBOL) {
          char *r = SEN_SET_STRKEY_BY_VAL(res);
          SEN_RBUF_PUTS(&buf, (*r == ':') ? r + 1 : r);
        } else {
          switch(res->type) {
          case sen_ql_object :
            {
              const char *key = _sen_obj_key(c->db, res);
              if (key) { SEN_RBUF_PUTS(&buf, key); }
            }
            break;
          case sen_ql_bulk :
            sen_rbuf_write(&buf, res->u.b.value, res->u.b.size);
            break;
          case sen_ql_int :
            sen_rbuf_itoa(&buf, res->u.i.i);
            break;
          default :
            break;
          }
        }
        if (!LISTP(list)) { break; }
        SEN_RBUF_PUTC(&buf, '\t');
      }
      c->output(c, &buf, SEN_QL_MORE);
    }
    break;
  }
exit :
  sen_rbuf_fin(&buf);
  return T;
}

#define flags(p)         ((p)->flags)
#define issymbol(p)     (flags(p) & SEN_OBJ_SYMBOL)
#define ismacro(p)      (flags(p) & SEN_OBJ_MACRO)

static void js_output(sen_ctx *c, sen_obj *obj, sen_rbuf *buf);

static void
js_output_with_format(sen_ctx *c, sen_obj *args, sen_rbuf *buf)
{
  sen_obj result, *car, obj, *fargs;
  memset(&result, 0, sizeof(sen_obj));
  POP(car, args);
  switch (car->type) {
  case sen_ql_records :
    {
      sen_id *rp;
      recinfo *ri;
      sen_obj *slots;
      const sen_recordh *rh;
      int i, hashp = 0, offset = 0, limit = 10;
      sen_records *r = car->u.r.records;
      obj.type = sen_ql_object;
      obj.class = car->class;
      POP(slots, args);
      if (!LISTP(slots)) {
        js_output(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
      }
      // todo : check && compile slots
      POP(car, args);
      if (!sen_obj2int(car)) { offset = car->u.i.i; }
      POP(car, args);
      if (!sen_obj2int(car)) { limit = car->u.i.i; }
      sen_records_rewind(r);
      for (i = 0; i < offset; i++) {
        if (!sen_records_next(r, NULL, 0, NULL)) { break; }
      }
      SEN_RBUF_PUTC(buf, '[');
      for (i = 0; i < limit; i++) {
        int oddp;
        sen_obj *s = slots;
        if (!sen_records_next(r, NULL, 0, NULL) ||
            !(rh = sen_records_curr_rec(r)) ||
            sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
          break;
        }
        obj.u.o.self = *rp;
        if (i) { SEN_RBUF_PUTS(buf, ", "); }
        SEN_RBUF_PUTC(buf, hashp ? '{' : '[');
        for (oddp = 0;; oddp = 1 - oddp) {
          char *value;
          sen_obj *t;
          sen_ql_method_func *func = NULL;
          POP(t, s);
          if (LISTP(t)) {
            char *fname = str_value(CADR(t));
            if (fname) { func = name2method(c, fname); }
            fargs = CDDR(t);
            t = CAR(t);
          }
          if ((value = str_value(t))) {
            if (hashp && !oddp) {
              sen_rbuf_str_esc(buf, value, -1, c->encoding);
            } else {
              if (*value == ':') {
                switch (value[1]) {
                case 'k' :
                  {
                    const char *key = _sen_obj_key(c->db, &obj);
                    if (key) {
                      sen_rbuf_str_esc(buf, key, -1, c->encoding);
                    } else {
                      SEN_RBUF_PUTS(buf, "<LOSTKEY>");
                    }
                  }
                  break;
                case 's' :
                  sen_rbuf_itoa(buf, ri->score);
                  break;
                default :
                  break;
                }
              } else {
                sen_obj *res = &result;
                sen_db_store *slot = sen_db_class_slot(c->db, obj.class, value);
                if (!slot) { break; }
                res = slot_value(c, slot, obj.u.o.self, NIL, res);
                if (NILP(res)) { break; }
                if (func) {
                  sen_obj param;
                  param.type = sen_ql_list;
                  param.u.l.car = res;
                  param.u.l.cdr = fargs;
                  res = func(c, &param, NULL);
                  if (NILP(res)) {
                    break;
                  }
                }
                if (res->flags & SEN_OBJ_SYMBOL) {
                  char *r = SEN_SET_STRKEY_BY_VAL(res);
                  sen_rbuf_str_esc(buf, (*r == ':') ? r + 1 : r, -1, c->encoding);
                } else {
                  switch(res->type) {
                  case sen_ql_object :
                    {
                      const char *key = _sen_obj_key(c->db, res);
                      if (key) {
                        sen_rbuf_str_esc(buf, key, -1, c->encoding);
                      } else {
                        SEN_RBUF_PUTS(buf, "<LOSTKEY>");
                      }
                    }
                    break;
                  case sen_ql_bulk :
                    sen_rbuf_str_esc(buf, res->u.b.value, res->u.b.size, c->encoding);
                    break;
                  case sen_ql_int :
                    sen_rbuf_itoa(buf, res->u.i.i);
                    break;
                  default :
                    break;
                  }
                }
              }
            }
          }
          if (!LISTP(s)) { break; }
          SEN_RBUF_PUTS(buf, (hashp && !oddp) ? ": " : ", ");
        }
        SEN_RBUF_PUTC(buf, hashp ? '}' : ']');
      }
      SEN_RBUF_PUTC(buf, ']');
    }
    break;
  case sen_ql_object :
    {
      int oddp, hashp = 0;
      sen_obj *slots;
      POP(slots, args);
      if (!LISTP(slots)) {
        js_output(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
      }
      SEN_RBUF_PUTC(buf, hashp ? '{' : '[');
      for (oddp = 0;; oddp = 1 - oddp) {
        char *value;
        sen_obj *t;
        sen_ql_method_func *func = NULL;
        POP(t, slots);
        if (LISTP(t) && LISTP(CDR(t))) {
          char *fname = str_value(CADR(t));
          if (fname) { func = name2method(c, fname); }
          fargs = CDDR(t);
          t = CAR(t);
        }
        if ((value = str_value(t))) {
          if (hashp && !oddp) {
            sen_rbuf_str_esc(buf, value, -1, c->encoding);
          } else {
            if (*value == ':') {
              switch (value[1]) {
              case 'k' :
                {
                  const char *key = _sen_obj_key(c->db, car);
                  if (key) {
                    sen_rbuf_str_esc(buf, key, -1, c->encoding);
                  } else {
                    SEN_RBUF_PUTS(buf, "<LOSTKEY>");
                  }
                }
                break;
              default :
                break;
              }
            } else {
              sen_obj *res = &result;
              sen_db_store *slot = sen_db_class_slot(c->db, car->class, value);
              if (!slot) { break; }
              res = slot_value(c, slot, car->u.o.self, NIL, res);
              if (NILP(res)) { break; }
              if (func) {
                sen_obj param;
                param.type = sen_ql_list;
                param.u.l.car = res;
                param.u.l.cdr = fargs;
                res = func(c, &param, NULL);
                if (NILP(res)) { break; }
              }
              if (res->flags & SEN_OBJ_SYMBOL) {
                char *r = SEN_SET_STRKEY_BY_VAL(res);
                sen_rbuf_str_esc(buf, (*r == ':') ? r + 1 : r, -1, c->encoding);
              } else {
                switch(res->type) {
                case sen_ql_object :
                  {
                    const char *key = _sen_obj_key(c->db, res);
                    if (key) {
                      sen_rbuf_str_esc(buf, key, -1, c->encoding);
                    } else {
                      SEN_RBUF_PUTS(buf, "<LOSTKEY>");
                    }
                  }
                  break;
                case sen_ql_bulk :
                  sen_rbuf_str_esc(buf, res->u.b.value, res->u.b.size, c->encoding);
                  break;
                case sen_ql_int :
                  sen_rbuf_itoa(buf, res->u.i.i);
                  break;
                default :
                  break;
                }
              }
            }
          }
        }
        if (!LISTP(slots)) { break; }
        SEN_RBUF_PUTS(buf, (hashp && !oddp) ? ": " : ", ");
      }
      SEN_RBUF_PUTC(buf, hashp ? '}' : ']');
    }
    break;
  default :
    js_output(c, car, buf);
    break;
  }
}

static void
js_output(sen_ctx *c, sen_obj *obj, sen_rbuf *buf)
{
  if (!obj || obj == NIL) {
    SEN_RBUF_PUTS(buf, "null");
  } else if (obj == T) {
    SEN_RBUF_PUTS(buf, "true");
  } else if (obj == F) {
    SEN_RBUF_PUTS(buf, "false");
  } else {
    switch (obj->type) {
    case sen_ql_void :
      if (issymbol(obj)) {
        const char *r = SEN_SET_STRKEY_BY_VAL(obj);
        sen_rbuf_str_esc(buf, (*r == ':') ? r + 1 : r, -1, c->encoding);
      } else {
        SEN_RBUF_PUTS(buf, "null");
      }
      break;
    case sen_db_raw_class :
    case sen_db_class :
    case sen_db_obj_slot :
    case sen_db_ra_slot :
    case sen_db_ja_slot :
    case sen_db_idx_slot :
    case sen_ql_object :
      {
        const char *key = _sen_obj_key(c->db, obj);
        if (key) {
          sen_rbuf_str_esc(buf, key, -1, c->encoding);
        } else {
          SEN_RBUF_PUTS(buf, "#<CLASS>");
        }
      }
      break;
    case sen_ql_records :
      {
        int i;
        sen_id *rp;
        recinfo *ri;
        sen_obj o;
        const sen_recordh *rh;
        sen_records *r = obj->u.r.records;
        sen_records_rewind(r);
        o.type = sen_ql_object;
        o.class = obj->class;
        SEN_RBUF_PUTC(buf, '[');
        for (i = 0;; i++) {
          if (!sen_records_next(r, NULL, 0, NULL) ||
              !(rh = sen_records_curr_rec(r)) ||
              sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
            break;
          }
          if (i) { SEN_RBUF_PUTS(buf, ", "); }
          o.u.o.self = *rp;
          {
            const char *key = _sen_obj_key(c->db, &o);
            if (key) {
              sen_rbuf_str_esc(buf, key, -1, c->encoding);
            } else {
              SEN_RBUF_PUTS(buf, "<LOSTKEY>");
            }
          }
        }
        SEN_RBUF_PUTC(buf, ']');
      }
      break;
    case sen_ql_query :
      SEN_RBUF_PUTS(buf, "#<QUERY>");
      break;
    case sen_ql_bulk :
      sen_rbuf_str_esc(buf, obj->u.b.value, obj->u.b.size, c->encoding);
      break;
    case sen_ql_list :
      if (obj->u.l.car == sen_ql_mk_symbol(c, ":")) {
        js_output_with_format(c, obj->u.l.cdr, buf);
      } else if (obj->u.l.car == sen_ql_mk_symbol(c, "@")) {
        int oddp;
        SEN_RBUF_PUTC(buf, '{');
        for (obj = obj->u.l.cdr, oddp = 0;; oddp = 1 - oddp) {
          js_output(c, obj->u.l.car, buf);
          if ((obj = obj->u.l.cdr) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTS(buf, oddp ? ", " : ": ");
            } else {
              SEN_RBUF_PUTS(buf, " . ");
              js_output(c, obj, buf);
              SEN_RBUF_PUTC(buf, '}');
              break;
            }
          } else {
            SEN_RBUF_PUTC(buf, '}');
            break;
          }
        }
      } else {
        SEN_RBUF_PUTC(buf, '[');
        for (;;) {
          js_output(c, obj->u.l.car, buf);
          if ((obj = obj->u.l.cdr) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTS(buf, ", ");
            } else {
              SEN_RBUF_PUTS(buf, " . ");
              js_output(c, obj, buf);
              SEN_RBUF_PUTC(buf, ']');
              break;
            }
          } else {
            SEN_RBUF_PUTC(buf, ']');
            break;
          }
        }
      }
      break;
    case sen_ql_native_method :
      SEN_RBUF_PUTS(buf, "#<NATIVE_METHOD>");
      break;
    case sen_ql_proc :
      SEN_RBUF_PUTS(buf, "#<PROCEDURE>");
      break;
    case sen_ql_syntax :
      SEN_RBUF_PUTS(buf, "#<SYNTAX>");
      break;
    case sen_ql_closure :
      if (ismacro(obj)) {
        SEN_RBUF_PUTS(buf, "#<MACRO>");
      } else {
        SEN_RBUF_PUTS(buf, "#<CLOSURE>");
      }
      break;
    case sen_ql_continuation :
      SEN_RBUF_PUTS(buf, "#<CONTINUATION>");
      break;
    case sen_ql_int :
      sen_rbuf_itoa(buf, obj->u.i.i);
      break;
    default :
      if (issymbol(obj)) {
        const char *r = SEN_SET_STRKEY_BY_VAL(obj);
        sen_rbuf_str_esc(buf, (*r == ':') ? r + 1 : r, -1, c->encoding);
      } else {
        SEN_RBUF_PUTS(buf, "#<?(");
        sen_rbuf_itoa(buf, obj->type);
        SEN_RBUF_PUTS(buf, ")?>");
      }
      break;
    }
  }
}

static sen_obj *
_native_method_js_output(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  sen_rbuf buf;
  sen_obj result, *car;
  if (sen_rbuf_init(&buf, 1)) { return F; }
  memset(&result, 0, sizeof(sen_obj));
  POP(car, args);
  js_output(c, car, &buf);
  c->output(c, &buf, SEN_QL_MORE);
  sen_rbuf_fin(&buf);
  return T;
}

void
sen_ql_def_db_methods(sen_ctx *c)
{
  sen_ql_def_native_method(c, "<db>", _native_method_db);
  sen_ql_def_native_method(c, "sen-query", _native_method_sen_query);
  sen_ql_def_native_method(c, "sen-snip", _native_method_sen_snip);
  sen_ql_def_native_method(c, "sen-output", _native_method_sen_output);
  sen_ql_def_native_method(c, "js-output", _native_method_js_output);
}
