/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _SEN_QL_H
#define _SEN_QL_H

#include <setjmp.h>
#include "store.h"

#ifdef __cplusplus
extern "C" {
#endif

/**** query language ****/

#define sen_ql_void          0x10
#define sen_ql_object        0x11
#define sen_ql_records       0x12
#define sen_ql_bulk          0x13
#define sen_ql_int           0x14
#define sen_ql_float         0x15
#define sen_ql_native_method 0x16
#define sen_ql_query         0x17
#define sen_ql_op            0x18
#define sen_ql_syntax        0x19
#define sen_ql_proc          0x1a
#define sen_ql_promise       0x1b
#define sen_ql_closure       0x1c
#define sen_ql_continuation  0x1d
#define sen_ql_list          0x40

typedef struct _sen_obj sen_obj;
typedef struct _sen_ctx sen_ctx;
typedef struct _sen_ql_co sen_ql_co;
typedef sen_obj *
sen_ql_method_func(sen_ctx *, sen_obj *, sen_ql_co *);

struct _sen_obj {
  uint8_t type;
  uint8_t flags;
  uint16_t nrefs;
  sen_id class;
  union {
    struct {
      sen_id self;
      sen_ql_method_func *func;
    } o;
    struct {
      sen_records *records; // todo : unify with o;
    } r;
    struct {
      char *value;
      uint32_t size;
    } b;
    struct {
      sen_obj *car;
      sen_obj *cdr;
    } l;
    struct {
      sen_obj *func;
    } f;
    struct {
      int32_t i;
    } i;
    struct {
      double d;
    } d;
    struct {
      sen_query *query;
    } q;
    struct {
      int8_t op;
      int8_t mode;
      int16_t weight;
      int32_t option;
    } op;
  } u;
};

#define SEN_QL_HEAD 1
#define SEN_QL_TAIL 2

#define SEN_QL_MORE 1

#define SEN_QL_CO_BEGIN(c) if (c) { switch((c)->last) { case 0:
#define SEN_QL_CO_WAIT(c,d) \
  (c)->last = __LINE__; (c)->data = (d); return NULL;\
  case __LINE__: (d) = (c)->data;
#define SEN_QL_CO_END(c) (c)->last = 0; }}

#define SEN_QL_PLACEHOLDER_CHAR '?'

#define SEN_QL_SET_MODE(c,m) ((c)->feed_mode = (m))
#define SEN_QL_GET_MODE(c) ((c)->feed_mode)
#define SEN_QL_GET_STAT(c) ((c)->stat)

struct _sen_ql_co {
  uint16_t mode;
  uint16_t last;
  void *data;
};

enum {
  sen_ql_atonce = 0,
  sen_ql_step
};

#define SEN_QL_TOPLEVEL  0x00
#define SEN_QL_EVAL      0x40
#define SEN_QL_NATIVE    0x41
#define SEN_QL_WAIT_EXPR 0xc0
#define SEN_QL_WAIT_ARG  0xc1
#define SEN_QL_WAIT_DATA 0xc2

#define SEN_QL_WAITINGP(c) ((c)->stat & 0x80)

struct _sen_ctx {

  jmp_buf error_jmp;

  uint32_t seqno;
  uint32_t lseqno;
  uint32_t nbinds;
  uint32_t nunbinds;
  uint8_t feed_mode;
  uint8_t stat;

  uint8_t batchmode;
  uint8_t gc_verbose;

  sen_encoding encoding;

  int tok;
  char *cur;
  char *str_end;
  sen_obj **pht;

  sen_obj arg;
  sen_db *db;
  sen_set *objects;
  sen_set *symbols;
  sen_ql_method_func *doing;
  sen_obj *phs;            /* list of placeholders */
  sen_ql_co co;

  sen_obj *args;      /* register for arguments of function */
  sen_obj *envir;      /* stack register for current environment */
  sen_obj *code;      /* register for current code */
  sen_obj *dump;      /* stack register for next evaluation */
  sen_obj *value;
  sen_obj *global_env;
  uint8_t op;

  char *inbuf;
  sen_rbuf *outbuf;

  void (*output)(sen_ctx *c, sen_rbuf *buf, int flags);

  union {
    void *ptr;
    int fd;
    uint32_t u32;
    uint64_t u64;
  } data;
};

extern sen_obj *sen_ql_nil;  /* special cell representing empty cell */
extern sen_obj *sen_ql_t;    /* special cell representing #t */
extern sen_obj *sen_ql_f;    /* special cell representing #f */

#define NIL sen_ql_nil
#define T sen_ql_t
#define F sen_ql_f

// FIXME : should be private
#define SEN_OBJ_ALLOCATED 1
#define SEN_OBJ_MARKED 2
#define SEN_OBJ_SYMBOL 4
#define SEN_OBJ_MACRO  8
#define SEN_OBJ_PROMISE 16
#define SEN_OBJ_REFERER 32
#define SEN_OBJ_NATIVE 64


sen_ctx *sen_ctx_open(sen_db *s, sen_rbuf *outbuf);
sen_rc sen_ctx_close(sen_ctx *c);

void sen_ql_init_globals(sen_ctx *c);
sen_obj *sen_ql_load(sen_ctx *c, const char *filename);

sen_obj *sen_ql_at(sen_ctx *c, const char *key);
sen_obj *sen_obj_new(sen_ctx *c);
sen_obj *sen_obj_alloc(sen_ctx *c, uint32_t size);
sen_obj *sen_obj_clear(sen_obj *o);
void sen_ql_def_db_methods(sen_ctx *c);

#define SEN_OBJ_INSPECT_ESC 1
#define SEN_OBJ_INSPECT_SYM 2

void sen_obj_inspect(sen_ctx *c, sen_obj *obj, sen_rbuf *buf, int flags);
void sen_ql_def_native_method(sen_ctx *c, const char *name,
                                  sen_ql_method_func *func);
const char *_sen_obj_key(sen_db *db, sen_obj *obj);
sen_obj *sen_ql_feed(sen_ctx *c, char *str, uint32_t str_size, int mode);
sen_obj *sen_ql_sexp_parser(sen_ctx *c, const char *str, uint32_t str_size,
                               sen_obj **phs);
sen_rc sen_obj2int(sen_obj *o);

sen_obj *sen_ql_mk_symbol(sen_ctx *c, const char *name);
void sen_ql_bind_symbol(sen_db_store *dbs, sen_obj *symbol);
sen_obj *sen_ql_mk_string(sen_ctx *c, const char *str, unsigned int len);

void sen_ql_init_const(void);

#define SEN_OBJ2VALUE(o,v,s) ((v) = (o)->u.b.value, (s) = (o)->u.b.size)
#define SEN_VALUE2OBJ(o,v,s) ((o)->u.b.value = (v), (o)->u.b.size = (s))

#define NILP(c) ((c) == NIL || !(c) || (c)->type == sen_ql_void)
#define OBJECTP(c) ((c) && (c)->type == sen_ql_object)
#define RECORDSP(c) ((c) && (c)->type == sen_ql_records)
#define BULKP(c) ((c) && (c)->type == sen_ql_bulk)
#define LISTP(c) ((c) && ((c)->type & sen_ql_list))
#define INTP(c) ((c) && (c)->type == sen_ql_int)
#define CLASSP(c) ((c) && (c)->type == sen_db_class)
#define RAW_CLASSP(c) ((c) && (c)->type == sen_db_raw_class)
#define NATIVE_METHODP(c) ((c) && (c)->type == sen_ql_native_method)
#define QUERYP(c) ((c) && (c)->type == sen_ql_query)
#define OBJ_SLOTP(c) ((c) && (c)->type == sen_db_obj_slot)
#define RA_SLOTP(c) ((c) && (c)->type == sen_db_ra_slot)
#define JA_SLOTP(c) ((c) && (c)->type == sen_db_ja_slot)
#define IDX_SLOTP(c) ((c) && (c)->type == sen_db_idx_slot)
#define SLOTP(c) ((c) && (((c)->type == sen_db_obj_slot) || ((c)->type == sen_db_ra_slot) || ((c)->type == sen_db_ja_slot) || ((c)->type == sen_db_idx_slot)))
#define OPP(c) ((c) && (c)->type == sen_ql_op)

#define SETNIL(c) ((c)->type = sen_ql_void)
#define SETINT(c,v) ((c)->type = sen_ql_int, (c)->u.i.i = (v))
#define SETBULK(c,v,s) ((c)->type = sen_ql_bulk, (c)->u.b.value = (v), (c)->u.b.size = (s))

#define CAR(c) ((c)->u.l.car)
#define CDR(c) ((c)->u.l.cdr)
#define CAAR(c) (CAR(CAR(c)))
#define CADR(c) (CAR(CDR(c)))
#define CDAR(c) (CDR(CAR(c)))
#define CDDR(c) (CDR(CDR(c)))
#define CADDR(c) (CAR(CDDR(c)))
#define POP(x,c) ((x) = CAR(c), (c) = CDR(c), (x))

#ifdef __cplusplus
}
#endif

#endif /* _SEN_QL_H */
