/*
 *  iReport  --  Visual designer for generating JasperReports Documents
 *  Copyright (C) 2002  Giulio Toffoli gt@businesslogic.it
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Giulio Toffoli
 *  Via T.Aspetti, 233
 *  35100 Padova ITALY
 *  gt@businesslogic.it
 */
package com.businesslogic.ireport.ui.directorydialog;

import com.ms.wfc.app.*;
import com.ms.wfc.core.*;
import com.ms.wfc.ui.*;
import com.ms.wfc.io.*;
import com.ms.wfc.util.*;
import com.ms.win32.*;
import com.ms.wfc.win32.*;

/**
 * This File Explorer supports drag and drop copying to and from the directory
 * tree, and to and from the filelist. The user cannot move/delete/overwrite
 * anything using this sample. If files are created or deleted in another 
 * application, selecting "Refresh" from the "View" menu will update the view.
 */
public class FileExplorer extends Form  { 

	private String selectedDir="";
    private int mouseDownX, mouseDownY;
    
    private static final int SHGFI_TYPENAME = 0x0400;
    private static final int SHGFI_SYSICONINDEX = 0x4000;
    private static final int SHGFI_SMALLICON = 0x0001;
    private static final String TITLE_DEFAULT = "Seleziona una directoy";
    
    private SysImageList sysLargeImageList;
    private SysImageList sysSmallImageList;
    private boolean freshDir;

    private MenuItem[] views;
    
    public FileExplorer() { 
        // Required for Visual J++ Form Designer support
        initForm();		

       // views = new MenuItem[] {viewLargeIcons, viewDetails, viewSmallIcons, 
       //            viewList};
        fillDirectoryTree();
    }

	public String getDirectory()
	{
		return this.selectedDir;
	}
    /**
     * FileExplorer overrides dispose so it can clean up the
     * component list.
     */
    public void dispose() { 
        sysLargeImageList = null;
        sysSmallImageList = null;
        super.dispose();
        components.dispose();
    }
    
    /**
     * Returns a HTML description for this project.
     * 
     * @see WFCRocks.ISample
     */
    public com.ms.wfc.io.IDataStream getHTML() { 
        /*
        * By referencing the file as a resource, it can be embedded in an
        * EXE, DLL, or even be located on a server.  As long as it's in the
        * same location as this class file, we will be able to get it.
        */
        
        return ResourceManager.getResource(getClass(), "FileExplorer.html");
    }
    
    /**
     * For a given root directory (or drive), add the directories to the 
     * directoryTree.
     */
    private void addDirectories(TreeNode node) {
        FileEnumerator e = new FileEnumerator( //getPathFromNode(node));
											  File.combine(getPathFromNode(node), "*.*"));
        while (e.hasMoreFiles()) {
            String name = e.getName();
            
            /*
            * Keep hidden directories hidden.
            */
            if ((e.getAttributes() & FileAttribute.DIRECTORY) != 0 &&
                (e.getAttributes() & FileAttribute.HIDDEN) == 0) {
                if (!name.equals(".") && !name.equals("..")) {
                    node.addNode(new DirectoryNode(name));
                }
            }
            e.getNextFile();
        }
    }
    
    /**
     * For a given node, add the sub-directories for node's children in the 
     * directoryTree.
     */
    private void addSubDirectories(DirectoryNode node) {
        // Get array of child nodes
        TreeNode[] nodes = node.getNodes();
        for (int i = 0; i < nodes.length; i++) {
            addDirectories(nodes[i]);
        }
        node.subDirectoriesAdded = true;
    }
    

    /* 
    * There are 4 view options for the filelist: 
    *   1. Large Icons, 
    *   2. Small Icons, 
    *   3. List, and 
    *   4. Details
    */
    private void changeView(Object source, Event e) {
        for (int i = 0; i < views.length; i++) {
            if (views[i] == (MenuItem)source) {
                views[i].setChecked(true);
                //fileList.setView(i);
            }
            else
                views[i].setChecked(false);
        }
    }

    /**
     *  Internal helper to copy all contents of a directory, including the 
     * contents of subdirectories.
     */ 
    private boolean copyDirectory(String sourceDir, String destDir) {
        int i = 0;
        File.createDirectories(destDir);
        String path = File.combine(sourceDir, "*.*");
        String[] files = File.getFiles(path);
        try {
            /* 
            * If the directory already exists, it is an exception. This sample
            * will not overwrite anything. 
            */
            for (i = 0; i < files.length; i++) {
                if (!files[i].equals(".") && !files[i].equals("..")) {
                    if (File.isDirectory(File.combine(sourceDir, files[i]))) {
                        copyDirectory(File.combine(sourceDir, files[i]), 
                            File.combine(destDir, files[i]));
                    } else {
                        File.copy(File.combine(sourceDir, files[i]), 
                            File.combine(destDir, files[i]));
                    }
                }
            }
            return true;
        }
        catch (WinIOException winIOE) {
            StringBuffer sMsg = new StringBuffer(
                "This FileExplorer will not overwrite your files. Copy of ");
            sMsg.append(File.combine(destDir, files[i]));
            sMsg.append(" will be aborted.");
            MessageBox.show(sMsg.toString(), "File already exists.", MessageBox.ICONEXCLAMATION);
            return false;
        }
    }

    /** 
     * Event handler for the afterSelect event on the directoryTree. Change the
     * title bar to show the path of the selected directoryNode.
     */
    private void directoryTree_afterSelect(Object source, TreeViewEvent e) {
        StringBuffer text = new StringBuffer(TITLE_DEFAULT);
        text.append(" - ");
        text.append(e.node.getText());
        setText(text.toString());
        //fillFileList(getPathFromNode(e.node));
		
		selectedDir = getPathFromNode(e.node);
    }

    /**
     * Event handler for the beforeExpand event on the directoryTree. If the 
     * node is not already expanded, expand it.
     */
    private void directoryTree_beforeExpand(Object source, TreeViewCancelEvent e) {
        DirectoryNode nodeExpanding = (DirectoryNode)e.node;
        if (!nodeExpanding.subDirectoriesAdded)
            addSubDirectories(nodeExpanding);            
    }

    
    /**
     *  Event handler for dropping into the directoryTree.
     */
    private void directoryTree_dragDrop(Object source, DragEvent e) { 
        String sourcePath;
        String destPath;
        String destFileName;
        String[] sourceFileName;

        TreeNode destNode = findNode(e.x, e.y);
        
        if (e.data.getDataPresent(DataFormats.CF_HDROP)) {
            sourceFileName = (String[])e.data.getData(DataFormats.CF_HDROP);
            destPath = destNode.getFullPath();
            boolean abort = false;            
            
            for (int i = 0; i < sourceFileName.length; ++i){
                // copy items over using system calls
                abort = false;
                sourcePath = File.getFullPath(sourceFileName[i]);
                destFileName = File.combine(destPath, 
                    File.getName(sourceFileName[i]));
                try  {
                    if (File.isDirectory(sourceFileName[i])) {
                        if(copyDirectory(sourceFileName[i], destFileName)) {
                            // Updating the directoryTree
                            TreeNode sourceNode = (TreeNode) findNode(mouseDownX, mouseDownY).clone();
                            destNode.addNode(sourceNode);
                            this.refresh(destNode);
                            
                        }
                    } else {
                        File.copy(sourceFileName[i], destFileName);
                    }
                }
                catch (WinIOException winIOE) {
                    // Don't let the user overwrite anything.
                    abort = true;
                    StringBuffer buffy = new StringBuffer("There is already a file, \"");
                    buffy.append(destFileName);
                    buffy.append("\". This file will not be copied.");
                    MessageBox.show( buffy.toString(), "File Already Exists!", 
                        MessageBox.ICONEXCLAMATION);
                }
            }
        }
    }
    
    /** 
     * Event handler for dragging directoryTree items from the directoryTree 
     * to somewhere else - either the directoryTree (handled by 
     * directoryTree_dragDrop), fileList (handled by fileList_dragDrop),
     * or the system (handled by Windows).
     */
    private void directoryTree_itemDrag(Object source, ItemDragEvent e) {
        TreeNode sourceNode = findNode(mouseDownX, mouseDownY);
        if (sourceNode != null) {
            // Try to get the node last clicked on
            DataObject dataObject = new DataObject(DataFormats.CF_HDROP, new String[] 
                { getPathFromNode(sourceNode) } );        
            this.doDragDrop(dataObject, DragDropEffect.COPY); 
        } 
    }

    /*
    * Save the mouseDown coordinates so that when draggind from the 
    * directoryTree, we can get the source node properly. Otherwise, we can 
    * only access the selectedNode, which is not necessarily the one we're 
    * dragging.
    */
    private void directoryTree_mouseDown(Object source, MouseEvent e) { 
        RECT rect = new RECT();
        Windows.GetWindowRect(directoryTree.getHandle(), rect);
        /* 
        * We want desktop coordinates and the mouseEvent gives us client 
        * coordinates so we need to convert them. 
        */
        mouseDownX = e.x + rect.left;
        mouseDownY = e.y + rect.top;
    }

    /**
     * Checks to see if the sysSmallImageList changes (it shouldn't). In this 
     * sample, this method is only called once so it really serves out its 
     * purpose upon creation of the sysSmallImageList.
     */ 
    private void ensureSmallImageList(int himl) {
        if (sysSmallImageList == null || sysSmallImageList.getHandle() != himl) {
            sysSmallImageList = new SysImageList(new Point(16, 16), himl);
            //fileList.setSmallImageList(sysSmallImageList);
        }
    }

    /**
     *  Checks to see if the sysLargeImageList changes (it shouldn't).
     */
    private void ensureLargeImageList(int himl) {
        if (sysLargeImageList == null || sysLargeImageList.getHandle() != himl) {
            sysLargeImageList = new SysImageList(new Point(32, 32), himl);
            //fileList.setLargeImageList(sysLargeImageList);
        }
    }

    private void fileExit_click(Object source, Event e) {
        Application.exit();
    }

    /** 
     * Event handler for dropping stuff into the filelist. Directories will 
     * not be accepted.   
     */
    private void fileList_dragDrop(Object source, DragEvent e) {
        if (e.data.getDataPresent(DataFormats.CF_HDROP)) {
            String sourcePath;
            String[] sourceFileName = (String[])e.data.getData(DataFormats.CF_HDROP);
            String destPath = getPathFromNode(directoryTree.getSelectedNode());
            String destFileName;
            boolean abort = false;            

            for (int i = 0; i < sourceFileName.length; ++i){
                // copy items over using system calls
                abort = false;
                sourcePath = File.getDirectory(sourceFileName[i]);

                /*
                * Disallow dragging and dropping into the same folder because that
                * would overwrite                 
                */
                if (sourcePath.equals(destPath)) {
                    return;
                }
                
                destFileName = File.combine(destPath, 
                    File.getName(sourceFileName[i]));
                try  {
                    if (File.isDirectory(sourceFileName[i])) {
                        MessageBox.show( "This is a filelist. You cannot drop a directory here."
                            , "No Directories Allowed!"
                              , MessageBox.ICONEXCLAMATION);
                    } else {
                        File.copy(sourceFileName[i], destFileName);
                    }
                }
                catch (WinIOException winIOE) {
                    // Don't let the user overwrite anything.
                    abort = true;
                    StringBuffer buffy = new StringBuffer("There is already a file, \"");
                    buffy.append(destFileName);
                    buffy.append("\". This file will not be copied.");
                    MessageBox.show( buffy.toString(), "File Already Exists!", 
                        MessageBox.ICONEXCLAMATION);
                }
            }
        }

        // Update the fileList.
        //fillFileList(getPathFromNode(directoryTree.getSelectedNode()));
    }

  

    /** 
     * Event handler for dragging filelist items from the filelist to somewhere else - 
     * either the directoryTree (handled by directoryTree_dragDrop) or the system (handled
     * by Windows).
     */
    
    
    
    /* For initializing the directoryTree upon creation of the FileExplorer. */
    private void fillDirectoryTree() {
        String[] drives = File.getLogicalDrives();
        for (int i = 0; i < drives.length; i++) {
            if (Kernel32.GetDriveType(drives[i]) == win.DRIVE_FIXED) {
                DirectoryNode cRoot = new DirectoryNode(drives[i]);
				cRoot.setImageIndex(2);
                directoryTree.addNode(cRoot);
                addDirectories(cRoot);
            }
			else if (Kernel32.GetDriveType(drives[i]) == win.DRIVE_CDROM) {
				DirectoryNode cRoot = new DirectoryNode(drives[i]);
				cRoot.setImageIndex(3);
                directoryTree.addNode(cRoot);
                addDirectories(cRoot);
			}
		//	else if (Kernel32.GetDriveType(drives[i]) == win.DRIVE_REMOVABLE) {
		//		DirectoryNode cRoot = new DirectoryNode(drives[i]);
		//		cRoot.setImageIndex(3);
        //        directoryTree.addNode(cRoot);
        //        addDirectories(cRoot);
		//	}
        }
    }
    
    /* 
    * For the given path, add all files to the fileList, with 
    * the appropriate icons.
    */
   

    /**
     * Internal helper to find the node under the cursor in the directoryTree given the 
     * cursor coordinates.
     * 
     * @param - x and y are screen coordinates of the cursor position.
     * @return - node of the directoryTree at (x, y)  
     */
    private TreeNode findNode(int x, int y) { 
        int nodeDesktopX, nodeDesktopY;
        int index = 0;
        boolean foundDir = false;
        boolean foundDrive = false;
        int iconWidth = sysSmallImageList.getImageSize().x;

        TreeNode[] nodes = directoryTree.getNodes();

        /*
        * TreeNode.getBounds() returns client coordinates relative to the form. Since these
        * client coordinates do not take into account the title bar and menu, 
        * we must convert relative coordinates to screen coordinates.
        */ 
        RECT rect = new RECT();
        Windows.GetWindowRect(directoryTree.getHandle(), rect);

        // Find the destination drive and store in nodes[index]
        while (!foundDrive) {
            nodeDesktopY = nodes[index].getBounds().y + rect.top;
            if ((y < nodeDesktopY) && (index < nodes.length)) {
                index++;
            } else {
                foundDrive = true;
            }
        }

        /*
        * This should never happen because findNode(x, y) is only called with
        * (x, y) being a point on the directoryTree.
        */
        if (!foundDrive) {
            return null;
        }
        
        // Find the destination directory and store in next
        TreeNode next = nodes[index];
        while ((next != null) && (foundDir == false)) {
            nodeDesktopY = next.getBounds().y + rect.top;
            nodeDesktopX = next.getBounds().x - iconWidth + rect.left;
            if ((y > nodeDesktopY) && (y < (nodeDesktopY + 
                                            next.getBounds().height))) {
                if((x > nodeDesktopX) && (x < (nodeDesktopX + 
                                               next.getBounds().width + 
                                               iconWidth))) {
                    foundDir = true;
                } else {
                    return null; // Didn't drop on a valid folder
                }
            } else {
                next = (DirectoryNode) next.getNextVisibleNode();
            }
        }            
        return next;
    }

    /**
     *  Returns the directory path of the node.
     */
    private String getPathFromNode(TreeNode node) {
        if (node.getParent() == null) {
            return node.getText();
        }
        return File.combine(getPathFromNode(node.getParent()), node.getText());
    }
    
    /**
     * Returns a string containing the project name
     * 
     * @see WFCRocks.ISample
     */
    public String getProjectName() { 
        return "WFCFileExplorer.vjp";
    }
    
    /**
     * Overrides control because the samples hash table only recognizes the
     * default title, "WFC File Explorer", in its hash table. You can still
     * explicitly call Control.getText() to get the full text value, which would
     * include the currently selected node.
     *
     * @return default text of the control.
     */
    public String getText() {
        return TITLE_DEFAULT;
    }

    /** 
     * Refresh helper functions to get all expanded nodes under the given
     * node. 
     * 
     * @param expandedNodes - reference to an array of paths containing all 
     * nodes which were in the expanded state when Refresh was requested.
     * 
     * @param startIndex - array index of expandedNodes to start adding entries 
     * to.
     * @return - new startIndex, i.e. given value of startIndex + number of 
     * entries added to expandedNodes.
     */
    private final int refresh_getExpanded(TreeNode node, String[] expandedNodes, int startIndex) {
        if (startIndex < expandedNodes.length) {
            if (node.isExpanded()) {
                expandedNodes[startIndex] = node.getText();
                startIndex++;
                TreeNode[] children = node.getNodes();
                for (int i = 0; i < children.length; i++) {
                    startIndex = refresh_getExpanded(children[i], 
                        expandedNodes,
                        startIndex);
                }
            }
            return startIndex;
        } 
        return -1;
    }
    
    /** 
     * Refresh helper function to expand all nodes whose paths are in paramter 
     * expandedNodes.
     * 
     * @param node - node from which to start expanding.
     * @param expandedNodes - Array of Strings with the path names of all 
     * nodes to expand
     */
    private final void refresh_expand(TreeNode node, String[] expandedNodes) {
        for (int i = expandedNodes.length - 1; i >= 0; i--) {
            if (expandedNodes[i].equals(node.getText())) {
                /*
                * For the expand button to show properly, one level of 
                * invisible children have to be added to the tree. 
                */
                addSubDirectories((DirectoryNode) node);
                node.expand();
                
                /* If the node is expanded, expand any children that were 
                * expanded before the refresh. 
                */
                TreeNode[] children = node.getNodes();
                for (int j = 0; j < children.length; j++) {
                    refresh_expand(children[j], expandedNodes);
                }
                
                return;
            }
        }
    }

    /**
     * Refresh helper function to check if the selected node is one 
     * of the children of a specified node.
     */
    private TreeNode refresh_selected_checkChildren(String selectedNodeString, TreeNode parentNode) {
        if (parentNode.getNodeCount() > 0) {
            if (parentNode.isExpanded()) {
                TreeNode[] children = parentNode.getNodes();
                for (int i = 0; i < children.length; i++) {
                    if (selectedNodeString.equals(children[i].getFullPath())) {
                        return children[i];
                    } else {
                        TreeNode ret = refresh_selected_checkChildren(selectedNodeString, children[i]);
                        if (ret != null) {
                            return ret;
                        }
                    }
                }
            }
        }
        return null;
    }
    
    /**
     * Refreshes the view by deleting all the nodes and restoring them by 
     * reading the disk(s). Any expanded nodes in the directoryView will be 
     * expanded after the refresh.
     * 
     * @param node - Node from which the refresh begins. Generally, this is 
     * the root.
     */
    private void refresh(TreeNode node) {
        // Keep track of which node was selected.
        TreeNode selectedTreeNode = directoryTree.getSelectedNode();
        String selectedTreeNodeString = selectedTreeNode.getFullPath();
        
        /*
        * Update the directoryTree. If there are children, then we must 
        * store the "expanded state" of each. Then we remove all the 
        * children nodes and read them from the system disk again in 
        * case any external application, like the  Windows Explorer, 
        * created/deleted any files/directories.
        */
        if (node.getNodeCount() > 0) {
            if (node.isExpanded()) {
                
                /*
                * Save all expanded nodes rooted at node, even those 
                * that are indirectly rooted. 
                */
                String[] tooBigExpandedNodes = new String[node.getNodeCount(true)];
                int iExpandedNodes = refresh_getExpanded(node, 
                    tooBigExpandedNodes, 
                    0);
                String[] expandedNodes = new String[iExpandedNodes];
                System.arraycopy(tooBigExpandedNodes, 0, expandedNodes, 0, 
                    iExpandedNodes);
                
                node.removeAll();
                addDirectories(node);
                
                /*
                * Make children with subdirectories show up with  
                * expand/collapse button. 
                */
                addSubDirectories((DirectoryNode)node); 
                node.expand();
                
                TreeNode[] children = node.getNodes();
                /*
                * check all children. Some might have had sub-directories 
                * added from an external application so previous childless 
                * nodes might now have children. 
                */
                for (int j = 0; j < children.length; j++) {
                    if (children[j].getNodeCount() > 0) {
                        /*
                        * If the child has subdirectories and it was expanded
                        * before the refresh, then expand after the refresh.                         
                        */
                        refresh_expand(children[j], expandedNodes);
                    }
                }
            } else {
                /* 
                * If the node is not expanded, then there is no need to check
                * if any of the children were expanded. However, we should
                * update the tree by reading the drive in case an external 
                * application add/removed any directories. 
                */
                node.removeAll();
                addDirectories(node);
            }
        } else {
            /* 
            * If there are no children, then again there is no need to 
            * worry about expanded nodes but if an external application
            * add/removed any directories we should reflect that. 
            */
            node.removeAll();
            addDirectories(node);
        }
        
        // setSelectedNode(...) to the node that was selected before the refresh.
        selectedTreeNode 
            = refresh_selected_checkChildren(selectedTreeNodeString, node);
        if (selectedTreeNode != null) {
            directoryTree.setSelectedNode(selectedTreeNode);
        }

        // Update the fileList.
        //fillFileList(getPathFromNode(directoryTree.getSelectedNode()));
    }

    private void viewRefresh_click(Object source, Event e) { 
        TreeNode[] nodes = directoryTree.getNodes();
        for (int i = 0; i < nodes.length; i++) {
            refresh(nodes[i]);
        }
    }


	private void okButton_click(Object source, Event e)
	{
	}

	/**
	 * NOTE: The following code is required by the Visual J++ form
	 * designer.  It can be modified using the form editor.  Do not
	 * modify it using the code editor.
	 */
	Container components = new Container();
	TreeView directoryTree = new TreeView();
	Splitter splitter1 = new Splitter();
	ImageList imageList1 = new ImageList();
	Panel panel1 = new Panel();
	Button cancelButton = new Button();
	Button okButton = new Button();

	private void initForm()
	{
		// NOTA: questo form contiene informazioni relative alle risorse in
		// un file esterno. Non modificare il parametro di tipo String delle
		// chiamate alla funzione resources.getObject(). Ad esempio, non
		// modificare"foo1_location" nella seguente riga di codice,
		// nemmeno se il nome dell'oggetto Foo viene modificato: 
		//	 foo1.setLocation((Point)resources.getObject("foo1_location"));

		IResourceManager resources = new ResourceManager(this, "FileExplorer");
		splitter1.setDock(ControlDock.LEFT);
		splitter1.setSize(new Point(2, 258));
		splitter1.setTabIndex(1);
		splitter1.setTabStop(false);

		imageList1.setImageStream((ImageListStreamer)resources.getObject("imageList1_imageStream"));
		/* @designTimeOnly imageList1.setLocation(new Point(56, 184)); */

		directoryTree.setAllowDrop(true);
		directoryTree.setDock(ControlDock.FILL);
		directoryTree.setLocation(new Point(2, 0));
		directoryTree.setSize(new Point(273, 258));
		directoryTree.setTabIndex(0);
		directoryTree.setText("treeView1");
		directoryTree.setImageList(imageList1);
		directoryTree.setIndent(19);
		directoryTree.setSelectedImageIndex(1);
		directoryTree.setSorted(true);
		directoryTree.addOnMouseDown(new MouseEventHandler(this.directoryTree_mouseDown));
		directoryTree.addOnDragDrop(new DragEventHandler(this.directoryTree_dragDrop));
		directoryTree.addOnAfterSelect(new TreeViewEventHandler(this.directoryTree_afterSelect));
		directoryTree.addOnBeforeExpand(new TreeViewCancelEventHandler(this.directoryTree_beforeExpand));
		directoryTree.addOnItemDrag(new ItemDragEventHandler(this.directoryTree_itemDrag));

		panel1.setDock(ControlDock.BOTTOM);
		panel1.setLocation(new Point(0, 258));
		panel1.setSize(new Point(275, 40));
		panel1.setTabIndex(3);
		panel1.setText("panel1");

		cancelButton.setAnchor(ControlAnchor.BOTTOMRIGHT);
		cancelButton.setLocation(new Point(186, 8));
		cancelButton.setSize(new Point(80, 24));
		cancelButton.setTabIndex(1);
		cancelButton.setText("Annulla");
		cancelButton.setDialogResult(DialogResult.CANCEL);

		okButton.setAnchor(ControlAnchor.BOTTOMRIGHT);
		okButton.setLocation(new Point(104, 8));
		okButton.setSize(new Point(72, 24));
		okButton.setTabIndex(0);
		okButton.setText("&Seleziona");
		okButton.setDialogResult(DialogResult.OK);
		okButton.addOnClick(new EventHandler(this.okButton_click));

		this.setText("Seleziona la diectory");
		this.setAcceptButton(okButton);
		this.setAutoScaleBaseSize(new Point(5, 13));
		this.setCancelButton(cancelButton);
		this.setClientSize(new Point(275, 298));
		this.setIcon((Icon)resources.getObject("this_icon"));

		this.setNewControls(new Control[] {
							panel1, 
							splitter1, 
							directoryTree});
		panel1.setNewControls(new Control[] {
							  cancelButton, 
							  okButton});
	}

    /**
     * The main entry point for the application. 
     *
     * @param args Array of parameters passed to the application
     * via the command line.
     */
    public static void main(String args[]) { 
        Application.run(new FileExplorer());
    }
}
