/*
 * DatabaseConnectionManager.java
 *
 * Created on February 12, 2004, 2:15 PM
 */

package dori.jasper.util;

/**
 * This is a class used to keep track of database connection information.
 * This class works sort of like a database since instances of this class can
 * be serialized using java 1.4 or above to an XML file then simply reloaded.
 * This class is not thread safe.  This class should operate best on capacity or less
 * DatabaseConnectionInfo objects.  Any more than that and the internal structures
 * have to be resized after each addition the copied over.  This is a slow process
 * once the data set is large.
 * @author  Wade Chandler
 * @version 1.0
 */
public class DatabaseConnectionInfoManager
implements java.io.Serializable
{
   
   /** used to hold the DatabaseConnectionInfo Objects */
   protected java.util.LinkedHashMap infos = null;
   
   /** used to hold the DatabaseConnectionInfo names and driver names. */
   protected java.util.ArrayList tableHelper = null;
   
   /** used to keep track of the indexes of the infos. */
   protected java.util.ArrayList infosIndexes = null;
   
   /** used to hold the name of the active connection */
   protected String activeConnection = null;
   
   /** Creates a new instance of DatabaseConnectionManager */
   public DatabaseConnectionInfoManager()
   {
      infos = new java.util.LinkedHashMap(30);
      infosIndexes = new java.util.ArrayList(30);
      tableHelper = new java.util.ArrayList(30);
      this.activeConnection = "";
   }//end DatabaseConnectionInfoManager
   
   /**
    *Creates a new instance of DatabaseConnectionManager with the capacity
    *set to the value supplied.  This can affect performance if the user were
    *to have many connections.  However, it should seem unlikely that a user
    *would have over 1000 different connection settings though nothing is
    *impossible.
    *@param intialCapacity the initial size of the objects.
    */
   public DatabaseConnectionInfoManager(int initialCapacity)
   {
      this.infos = new java.util.LinkedHashMap(initialCapacity);
      this.infosIndexes = new java.util.ArrayList(initialCapacity);
      this.tableHelper = new java.util.ArrayList(initialCapacity);
      this.activeConnection = "";
   }//end DatabaseConnectionInfoManager
   
   /**
    *Method used to ensure the capacity of the instance.  This operates
    *on all internal data structures.  So, this is not a method to be called
    *frequently.
    *@param capacity the capacity to ensure
    */
   public void ensureCapacity(int capacity)
   {
      java.util.Map oldMap = this.infos;
      this.infos = new java.util.LinkedHashMap(capacity);
      this.infos.putAll(oldMap);
      this.infosIndexes.ensureCapacity(capacity);
      tableHelper.ensureCapacity(capacity);
   }//end ensureCapacity
   
   /**
    *Used to obtain an index of a key.
    *@param key the key to find the index for
    *@return the index for the key/name
    */
   public int getIndexForKey(String key)
   {
      int ret = this.infosIndexes.indexOf(key);
      return ret;
   }//end getIndexForKey
   
   /**
    *Method modifies or adds a DatabaseConnectionInfo to this DatabaseConnectionManager
    *Object.
    *@param key the name or key of the info.
    *@param info the information for the connection itself.
    */
   public void setDatabaseConnectionInfo(String key, DatabaseConnectionInfo info)
   {
      if( key != null && key.trim().length() > 0 )
      {
         
         if( !this.infos.containsKey(key) )
         {
            java.util.Vector os = new java.util.Vector(2);
            Object driver = info.getDriver();
            Object okey = key;
             os.addElement(okey);
            os.addElement(driver);
            this.tableHelper.add( os );
            //this should be the index of our
            //next component sense we are using LinkedHashMap.
            //it is an ordered set.
            int index = this.infos.size();
            this.infosIndexes.add(key);
         }
         this.infos.put(key, info);
      }
      else
      {
         throw new java.lang.IllegalArgumentException("The key can not be null or empty when adding a database connection info object.");
      }
   }//end setDatabaseConnectionInfo
   
   /**
    *Removes a DatabaseConnectionInfo object from this instance.
    *@param key the info to remove
    *@return the DatabaseConnectionInfo which was removed
    */
   public DatabaseConnectionInfo removeDatabaseConnectionInfo(String key)
   {
      DatabaseConnectionInfo ret = null;
      ret = (DatabaseConnectionInfo)this.infos.remove(key);
      int index = this.infosIndexes.indexOf(key);
      this.tableHelper.remove(index);
      this.infosIndexes.remove(key);
      if(key.equals(this.activeConnection))
      {
         this.activeConnection = "";
      }
      return ret;
   }//end removeDatabaseConnectionInfo
   
   /**
    *Used to retrieve a DatabaseConnectionInfo object by name/key.  This method
    *returns null if one is not found.
    *@param key the name/key to use to find the info
    *@return the info
    */
   public DatabaseConnectionInfo getDatabaseConnectionInfo(String key)
   {
      DatabaseConnectionInfo ret = (DatabaseConnectionInfo)this.infos.get(key);
      return ret;
   }//end getDatabaseConnectionInfo
   
   /**
    *Used to set the active connection.
    *@param key the key/name to make the active connection.
    */
   public void setActiveDatabaseConnectionInfo(String key)
   {
      if(!this.infos.containsKey(key))
      {
         throw new java.lang.IllegalArgumentException("The connection can not be found.  The key used to set the active connection must exists in the connection info.");
      }
      this.activeConnection = key;
   }//end setActiveConnection
   
   /**
    *Method used to get the active connection.  If there is no active connection
    *then null is returned.
    *@return the DatabaseConnectionInfo object for the active connection.
    */
   public DatabaseConnectionInfo getActiveDatabaseConnectionInfo()
   {
      DatabaseConnectionInfo ret = null;
      if(this.activeConnection != null && this.activeConnection.length() > 0)
      {
         if(!this.infos.containsKey(this.activeConnection))
         {
            throw new java.lang.RuntimeException("Something has been configured incorrectly.  The active connection can not be found.");
         }
         ret = (DatabaseConnectionInfo)this.infos.get(this.activeConnection);
      }
      else
      {
         ret = null;
      }
      return ret;
   }//end getActiveDatabaseConnectionInfo
   
   /**
    *Method used to get all of the database connection  infos keys/names for an instance.
    *@return all the keys/names of the connection info objects.
    */
   public Object[] getDatabaseConnectionInfoKeys()
   {
      Object[] ret = this.infos.keySet().toArray();
      return ret;
   }
   
   /**
    *Helper method used to get an array of keys and drivers.  This is a simple
    *method used to allow the info in this class to be used in a DefaultTableModel.
    *One needs to override the remove/setDatabaseConnectionInfo methods to have this
    *method operate differently.  Currently this operates on info stored in tableHelper
    *a protected variable which is used for nothing more than to store an array of array
    *info about columns to be used by a table.  See the source for more information.
    *@return an array of arrays where Object[a][b] a = rows and b = connection names and drivers
    *        index 0 = key/name and index 1 = driver respectfully in a vector
    */
   public java.util.Vector getDatabaseConnectionInfoKeysAndDrivers()
   {
      java.util.Vector ret = new java.util.Vector( this.tableHelper );
      return ret;
   }//end getDatabaseConnectionInfoKeysAndDrivers
   
   /**
    *Method used to obtain a database connection by a named DatabaseConnectionInfo instance.
    *@param key the key/name of the database connection.
    *@return a JDBC Connection to the database referenced from the DatabaseConnectionInfo class
    *@throws java.sql.SQLException thrown on JDBC errors.
    */
   public java.sql.Connection getConnection(String key)
   throws java.sql.SQLException
   {
      java.sql.Connection ret = this.getDatabaseConnectionInfo(key).getConnection();
      return ret;
   }//end getConnection
   
}//end class DatabaseConnectionInfoManager
