#include <linux/interrupt.h>
#include <linux/io.h>

#include <mach-ms104sh4ag/mach/ms104sh4ag.h>

static struct reg_data {
	unsigned long addr;
	unsigned long bit;
} reg_data[] = {
	[3]     = {CPLD4, 1},
	[4]     = {CPLD4, 2},
	[5]     = {CPLD4, 4},
	[6]     = {CPLD4, 8},
	[7]     = {CPLD5, 1},
	[9]     = {CPLD5, 2},
	[10]    = {CPLD5, 4},
	[11]    = {CPLD5, 8},
	[12]    = {CPLD6, 1},
	[14]    = {CPLD6, 2},
	[15]    = {CPLD6, 4},
	[16]    = {CPLD10, 1},
	[17]    = {CPLD10, 2}
};

static struct reg_data *get_rd(unsigned int irq)
{
	return &reg_data[irq - IRQ_CASCADE_BASE];
}

static void enable_ms104sh4ag_irq(unsigned int irq)
{
	struct reg_data *rd = get_rd(irq);
#if 0
	pr_info("%s: irq=%d addr=%08lx bit=%08lx st=%08x flags=%08lx\n", __func__, irq, rd->addr, rd->bit, irq_desc[irq].status, __raw_local_save_flags());
#endif
	ctrl_outb(ctrl_inb(rd->addr) | rd->bit, rd->addr);
}

static void disable_ms104sh4ag_irq(unsigned int irq)
{
	struct reg_data *rd = get_rd(irq);
#if 0
	pr_info("%s: irq=%d addr=%08lx bit=%08lx st=%08x flags=%08lx\n", __func__, irq, rd->addr, rd->bit, irq_desc[irq].status, __raw_local_save_flags());
#endif
	ctrl_outb(ctrl_inb(rd->addr) & ~rd->bit, rd->addr);
}

static void mask_ms104sh4ag_irq(unsigned int irq)
{
#if 0
	printk("%s: >>\n", __func__);
#endif
	disable_ms104sh4ag_irq(irq);
}

static void mask_ack_ms104sh4ag_irq(unsigned int irq)
{
	disable_ms104sh4ag_irq(irq);
}

static void end_ms104sh4ag_irq(unsigned int irq)
{
#if 0
	printk("%s: irq=%d\n", __func__, irq);
#endif
}

static int set_type_ms104sh4ag_irq(unsigned int irq, unsigned int flow_type)
{
#if 0
	printk("%s: irq=%d flow_type=%08x\n", __func__, irq, flow_type);
#endif
	//      ctrl_outl((inl(ICR1) & ~0xC0000000) | 0x10000000, ICR1);
	return 0;
}

static struct irq_chip ms104sh4ag_irq_chip __read_mostly = {
	.name           = "MS104-SH4AG-CPLD",
	.mask           = mask_ms104sh4ag_irq,
	.unmask         = enable_ms104sh4ag_irq,
	.mask_ack       = mask_ack_ms104sh4ag_irq,
	.end            = end_ms104sh4ag_irq,
	.set_type       = set_type_ms104sh4ag_irq,
};

static void make_ms104sh4ag_irq(unsigned int irq)
{
	disable_irq_nosync(irq);
	set_irq_chip_and_handler_name(irq, &ms104sh4ag_irq_chip,
				      handle_level_irq, "level");
	enable_irq(irq);
}

static int find_active_irq0(int irq)
{
	u8 v = ctrl_inb(CPLD10);
#if 0
	printk("%s: v = %02x\n", __func__, v);
#endif
	if (v & TP_INT_STATUS)
		return TP_IRQ;
	if (v & RTC_INT_STATUS)
		return RTC_IRQ;
#if 0
       	pr_warning("IRQ0 failed...\n");
#endif

	return irq;
}

static int find_active_irq1(int irq)
{
	int i, j;
	unsigned char v;
	struct {
		unsigned long addr;
		int num;
		int offset[4];
	} regs[] = {
		{0xAF000000, 4, {3, 4, 5, 6}},
		{0xAF400000, 4, {7, 9, 10, 11}},
		{0xAF800000, 3, {12, 14, 15}}
	};

	for (i = ARRAY_SIZE(regs) - 1; i >= 0; i--) {
		v = ctrl_inb(regs[i].addr);

		for (j = regs[i].num - 1; j >= 0; j--)
			if (v & (1 << j))
				return PC104_IRQ_BASE + regs[i].offset[j];
	}
#if 0
	pr_warning("IRQ1 failed...\n");
#endif
	return irq;
}

int ms104sh4ag_irq_demux(int irq)
{
	switch (irq) {
	case IRQ0_IRQ:
		return find_active_irq0(irq);
	case IRQ1_IRQ:
		return find_active_irq1(irq);
	}
	return irq;
}

static struct irqaction irq0 = {
	.handler        = no_action,
	.mask           = CPU_MASK_NONE,
        .name           = "IRQ0-casecade"
};

static struct irqaction irq1 = {
	.handler        = no_action,
	.mask           = CPU_MASK_NONE,
        .name           = "IRQ1-casecade"
};

void init_ms104sh4ag_IRQ(void)
{
        printk("MS104-SH4AG IRQ setup CPLD10=%02x INTREQ=%08x\n", ctrl_inb(0xafc00000), ctrl_inl(INTREQ));

        ctrl_outw(ctrl_inw(PTSEL_S) & 0x7FFF, PTSEL_S);
        ctrl_outw((ctrl_inw(PTSEL_K) & 0x3FFF) | 0x4000, PTSEL_K);

        ctrl_outl(0xC0000000, INTMSKCLR);
        ctrl_outl(0x00000000, INTREQ);
        ctrl_outl((ctrl_inl(ICR1) & ~0xF0000000) | 0xA0000000, ICR1);

//      printk(" INTMSK=%08x\n", ctrl_inl(INTMSK));

        make_ms104sh4ag_irq(PC104_IRQ3);
        make_ms104sh4ag_irq(PC104_IRQ4);
        make_ms104sh4ag_irq(PC104_IRQ5);
        make_ms104sh4ag_irq(PC104_IRQ6);
        make_ms104sh4ag_irq(PC104_IRQ7);
        make_ms104sh4ag_irq(PC104_IRQ9);
        make_ms104sh4ag_irq(PC104_IRQ10);
        make_ms104sh4ag_irq(PC104_IRQ11);
        make_ms104sh4ag_irq(PC104_IRQ12);
        make_ms104sh4ag_irq(PC104_IRQ14);
        make_ms104sh4ag_irq(PC104_IRQ15);
        make_ms104sh4ag_irq(RTC_IRQ);
        make_ms104sh4ag_irq(TP_IRQ);
	ctrl_outb(0, CPLD10);
        ctrl_outl(ctrl_inl(INTPRI) | IRQ0_PRIO<<28 | IRQ1_PRIO<<24, INTPRI);

        setup_irq(IRQ0_IRQ, &irq0);
        setup_irq(IRQ1_IRQ, &irq1);
}
