/*
 * I2C bus driver for the SH7764 I2C Interfaces.
 *
 * (c) 2005-2008 MSC Vertriebsges.m.b.H, Manuel Lauss <mlau@msc-ge.com>
 * (c) 2008 ALPHAPROJECT
 */

#include <linux/completion.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>
#include <linux/slab.h>

#include <asm/clock.h>
#include <asm/i2c-sh7764.h>
#include <asm/io.h>

/* register offsets */
#define ICSCR		0x0		/* slave ctrl		*/
#define ICMCR		0x4		/* master ctrl		*/
#define ICSSR		0x8		/* slave status		*/
#define ICMSR		0xC		/* master status	*/
#define ICSIER		0x10		/* slave irq enable	*/
#define ICMIER		0x14		/* master irq enable	*/
#define ICCCR		0x18		/* clock dividers	*/
#define ICSAR		0x1c		/* slave address	*/
#define ICMAR		0x20		/* master address	*/
#define ICRXD		0x24		/* data port		*/
#define ICTXD		0x24		/* data port		*/

#define MCR_MDBS	0x80
#define MCR_FSCL	0x40
#define MCR_FSDA	0x20
#define MCR_OBPC	0x10
#define MCR_MIE		0x08
#define MCR_TSBE	0x04
#define MCR_FSB		0x02
#define MCR_ESG		0x01

#define MSR_MNR		0x40
#define MSR_MAL		0x20
#define MSR_MST		0x10
#define MSR_MDE		0x08
#define MSR_MDT		0x04
#define MSR_MDR		0x02
#define MSR_MAT		0x01

#define MIE_MNRE	0x40
#define MIE_MALE	0x20
#define MIE_MSTE	0x10
#define MIE_MDEE	0x08
#define MIE_MDTE	0x04
#define MIE_MDRE	0x02
#define MIE_MATE	0x01

struct sh7764_i2c_drvdata {
	void __iomem *iobase;
	struct i2c_adapter adapter;
	struct i2c_msg *msg;
#define SH7764_I2C_SEND 1
#define SH7764_I2C_RECV 2
#define SH7764_I2C_STOP 4
	u8 flags;
#define SH7764_I2C_DONE		1
#define SH7764_I2C_ARBLOST	2
#define SH7764_I2C_NACK		4
#define SH7764_I2C_IN_INTERRUPT 8
	u8 status;
	struct completion xfer_done;
};

static inline u8 sh7764_i2c_inb(struct sh7764_i2c_drvdata *drvdata, int reg)
{
	return ctrl_inb((unsigned long)drvdata->iobase + reg);
}

static inline void sh7764_i2c_outl(struct sh7764_i2c_drvdata *drvdata,
				    int reg, u32 val)
{
	ctrl_outl(val, (unsigned long)drvdata->iobase + reg);
}

static inline void sh7764_i2c_outb(struct sh7764_i2c_drvdata *drvdata,
				    int reg, u8 val)
{
	ctrl_outb(val, (unsigned long)drvdata->iobase + reg);
}

static void sh7764_i2c_check(struct sh7764_i2c_drvdata *drvdata)
{
	struct i2c_msg *msg = drvdata->msg;
	u32 mcr = sh7764_i2c_inb(drvdata, ICMCR);
	u32 scr = sh7764_i2c_inb(drvdata, ICSCR);
	u32 msr = sh7764_i2c_inb(drvdata, ICMSR);
	u32 ssr = sh7764_i2c_inb(drvdata, ICSSR);
	int mat = 0;

	//	printk("%s: MCR=%02x MSR=%02x SCR=%02x SSR=%02x msg{len=%d flags=%04x} flg=%1x st=%1x\n", __func__, mcr, msr, scr, ssr, msg->len, msg->flags, drvdata->flags, drvdata->status);

	if (msr & MSR_MAL) {	/* Arbitration lost */
		sh7764_i2c_outb(drvdata, ICMCR, 0);
		sh7764_i2c_outb(drvdata, ICSCR, 0);
		sh7764_i2c_outb(drvdata, ICSAR, 0);
		drvdata->status |= SH7764_I2C_DONE | SH7764_I2C_ARBLOST;
		goto out;
	}

	if (msr & MSR_MNR) {	/* NACK */
	  //		udelay(100);
		sh7764_i2c_outb(drvdata, ICMCR, MCR_MIE | MCR_FSB);
		sh7764_i2c_outb(drvdata, ICMIER, MIE_MSTE);
		sh7764_i2c_outb(drvdata, ICSCR, 0);
		sh7764_i2c_outb(drvdata, ICSAR, 0);
		drvdata->status |= SH7764_I2C_NACK;
		msr &= ~MSR_MAT;
	}

	if (msr & MSR_MST) {
		drvdata->status |= SH7764_I2C_DONE;
		goto out;
	}

	if (msr & MSR_MAT) {
		if (msg->len) {
			/* clear ESG */
			sh7764_i2c_outb(drvdata, ICMCR, MCR_MIE);
			/* clear MDE_MAT */
			if (msg->flags & I2C_M_RD) {
				sh7764_i2c_outb(drvdata, ICMSR, 
						msr & ~(MSR_MDR|MSR_MAT));
			} else {
				sh7764_i2c_outb(drvdata, ICMSR, 
						msr & ~(MSR_MDE|MSR_MAT));
			}
			goto end;
		} else {
			sh7764_i2c_outb(drvdata, ICMIER, 
					MIE_MNRE | MIE_MALE | MIE_MSTE);
			sh7764_i2c_outb(drvdata, ICMCR,
					MCR_MIE | MCR_FSB);
			goto out;
		}
	}

	if (msr & MSR_MDE) {
		struct i2c_msg *msg = drvdata->msg;

		if (drvdata->flags & SH7764_I2C_SEND) {
			if (msg->len) {
				sh7764_i2c_outb(drvdata, ICTXD, *msg->buf);
				msg->buf++;
				msg->len--;
				if (mat)
					;//return IRQ_HANDLED;
			} else {
//				printk(" send FSB\n");
				sh7764_i2c_outb(drvdata, ICMCR,
						MCR_MIE | MCR_FSB);
			}
		} else {
			printk("%s:%d Not implemeneted ??\n", __func__, __LINE__);
		}
	}

	if (msr & MSR_MDR) {
		if (drvdata->flags & SH7764_I2C_RECV) {
			if (msg->len) {
				*msg->buf = sh7764_i2c_inb(drvdata, ICRXD);
				msg->buf++;
				msg->len--;
			} else {
//				printk(" send FSB\n");
				sh7764_i2c_outb(drvdata, ICMCR,
						MCR_MIE | MCR_FSB);
			}
		}
	}

out:
	if (drvdata->status & SH7764_I2C_DONE) {
		sh7764_i2c_outb(drvdata, ICMIER, 0);
		if (!(drvdata->status & SH7764_I2C_IN_INTERRUPT))
			complete(&drvdata->xfer_done);
	}

	sh7764_i2c_outb(drvdata, ICMSR, ~msr);
	sh7764_i2c_outb(drvdata, ICSSR, 0);
end:
	;
}

static irqreturn_t sh7764_i2c_irq(int irq, void *ptr)
{
//	printk("%s: >>\n", __func__);

	sh7764_i2c_check(ptr);

	return IRQ_HANDLED;
}

static void sh7764_i2c_mrecv(struct sh7764_i2c_drvdata *drvdata,
			     struct i2c_msg *msg)
{
  //	pr_debug("%s: msg{addr=%02x len=%d}\n", __func__, msg->addr, msg->len);

	drvdata->flags |= SH7764_I2C_RECV;

	//sh7764_i2c_outb(drvdata, ICSAR, 0x7f);
	sh7764_i2c_outb(drvdata, ICMAR, msg->addr << 1 | 1);

	sh7764_i2c_outb(drvdata, ICMSR, 0);
	sh7764_i2c_outb(drvdata, ICMCR, MCR_MIE | MCR_ESG);
	sh7764_i2c_outb(drvdata, ICMIER,
			MIE_MNRE | MIE_MALE | MIE_MSTE | MIE_MDRE | MIE_MATE);
}

static void sh7764_i2c_msend(struct sh7764_i2c_drvdata *drvdata,
			     struct i2c_msg *msg)
{
	int i;

	//	pr_debug("%s: msg{addr=%02x len=%d}\n", __func__, msg->addr, msg->len);
#if 0
	for (i = 0; i < msg->len; i++)
		printk("%02x ", msg->buf[i]);
	printk("\n");
#endif
	drvdata->flags |= SH7764_I2C_SEND;

//	sh7764_i2c_outb(drvdata, ICSAR, 0x7F);
	sh7764_i2c_outb(drvdata, ICMAR, msg->addr << 1 | 0);
	sh7764_i2c_outb(drvdata, ICTXD, *msg->buf);

	msg->buf++;
	msg->len--;

	sh7764_i2c_outb(drvdata, ICMSR, 0);
	sh7764_i2c_outb(drvdata, ICMCR, MCR_MIE | MCR_ESG);
	sh7764_i2c_outb(drvdata, ICMIER, 
			MIE_MNRE | MIE_MALE | MIE_MSTE | MIE_MDEE | MIE_MATE);
//	sh7764_i2c_outb(drvdata, ICSIER, 0x1F);
	sh7764_i2c_outb(drvdata, ICSSR, 0x00);
	sh7764_i2c_outb(drvdata, ICSCR, 0x00);
}

static int sh7764_i2c_master_xfer(struct i2c_adapter *adap,
				  struct i2c_msg *msg, int num)
{
	struct sh7764_i2c_drvdata *drvdata = adap->algo_data;
	int i = 0;

	//	printk("\n%s: >> adap{nr=%d} num=%d\n\n", __func__, adap->nr, num);

	if (sh7764_i2c_inb(drvdata, ICMCR) & MCR_FSDA) {
		dev_err(&adap->dev, "sh7764-i2c%d: bus busy!\n", adap->nr);
		return -EBUSY;
	}

	while (i < num) {
	  //		printk("%d < %d\n", i++, num);continue;
		int retr = adap->retries;
	retry:
		drvdata->flags = i == num-1 ? SH7764_I2C_STOP : 0;
		drvdata->status = 0;
		drvdata->msg = msg;
		if (in_interrupt())
			drvdata->status |= SH7764_I2C_IN_INTERRUPT;
		else
			init_completion(&drvdata->xfer_done);

		if (msg->flags & I2C_M_RD)
			sh7764_i2c_mrecv(drvdata, msg);
		else
			sh7764_i2c_msend(drvdata, msg);

		if (in_interrupt()) {
			for ( ; ; ) {
				sh7764_i2c_check(drvdata);
				if (drvdata->status & 0x7)
					break;
				cpu_relax();
			}
			
		} else
			wait_for_completion(&drvdata->xfer_done);

//g		printk("Complete !!\n");

		if (drvdata->status == 0) {
			num = -EIO;
			break;
		}

		if (drvdata->status & SH7764_I2C_NACK) {
			mdelay(1);
			num = -EREMOTEIO;
			break;
		}

		if (drvdata->status & SH7764_I2C_ARBLOST) {
			if (retr--) {
				mdelay(2);
				goto retry;
			}
			num = -EREMOTEIO;
			break;
		}

		msg++;
		i++;
	}
	drvdata->msg = NULL;
	drvdata->flags = 0;
	drvdata->status = 0;

	sh7764_i2c_outb(drvdata, ICMCR, 0);
	sh7764_i2c_outb(drvdata, ICMSR, 0);
	sh7764_i2c_outb(drvdata, ICMIER, 0);
	sh7764_i2c_outb(drvdata, ICSCR, 0);
	sh7764_i2c_outb(drvdata, ICSSR, 0);
	sh7764_i2c_outb(drvdata, ICSAR, 0);
	
	return num;
}

static u32 sh7764_i2c_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | (I2C_FUNC_SMBUS_EMUL & ~I2C_FUNC_SMBUS_QUICK);
}

static const struct i2c_algorithm sh7764_i2c_algo = {
	.master_xfer	= sh7764_i2c_master_xfer,
	.functionality	= sh7764_i2c_func
};

static u8 calc_CCR(unsigned long hz)
{
	struct clk *mclk;
	unsigned long clock;
	int cdf, scgd;

	mclk = clk_get(NULL, "module_clk");
	if (IS_ERR(mclk))
		return PTR_ERR(mclk);

	clock = mclk->rate;
	clk_put(mclk);

	cdf = 1;
	while ( (clock / cdf) >= 20000000 )
		cdf++;

	clock /= cdf;

	cdf--;

	scgd = ((clock - 1) - hz * 20) / (hz * 8) + 1;
	printk(" CDF=%d SCGD=%d\n", cdf, scgd);

	return (scgd << 2) | cdf;
}

static int sh7764_i2c_probe(struct platform_device *pdev)
{
	int ret = 0;
	int irq;
	struct sh7764_i2c_platdata *pdata = pdev->dev.platform_data;
	struct sh7764_i2c_drvdata *drvdata;
	struct resource *res;
	struct i2c_adapter *adap;
	int ccr;

	if (! pdata) {
		dev_err(&pdev->dev, "no platform_data!\n");
		ret = -ENODEV;
		goto out0;
	}

	drvdata = kzalloc(sizeof(struct sh7764_i2c_drvdata), GFP_KERNEL);
	if (! drvdata) {
		dev_err(&pdev->dev, "no mem for private data\n");
		ret = -ENOMEM;
		goto out0;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (! res) {
		dev_err(&pdev->dev, "no mmio resources\n");
		ret = -ENODEV;
		goto out1;
	}

	drvdata->iobase = (void *)res->start;

	adap = &drvdata->adapter;
	adap->nr = pdev->id;
	adap->algo = &sh7764_i2c_algo;
	//	adap->class = I2C_CLASS_ALL;
	adap->class = I2C_CLASS_HWMON | I2C_CLASS_SPD;
	adap->retries = 3;
	adap->algo_data = drvdata;
	adap->dev.parent = &pdev->dev;
	snprintf(adap->name, sizeof(adap->name),
		 "SH7764 I2C at %08lx", (unsigned long)res->start);

	sh7764_i2c_outb(drvdata, ICSCR, 0);
	sh7764_i2c_outb(drvdata, ICMCR, 0);
	sh7764_i2c_outb(drvdata, ICSSR, 0);
	sh7764_i2c_outb(drvdata, ICMSR, 0);
	sh7764_i2c_outb(drvdata, ICSIER, 0);
	sh7764_i2c_outb(drvdata, ICMIER, 0);
	sh7764_i2c_outb(drvdata, ICSAR, 0);
	sh7764_i2c_outb(drvdata, ICMAR, 0);
	
#ifdef CONFIG_SH_MS104SH4AG
	ccr = 3 << 2 | 2;
#else
	ccr = calc_CCR(pdata->speed_khz);
#endif

	if (ccr < 0) {
		dev_err(&pdev->dev, "invalid SCL clock: %dkHz\n",
			pdata->speed_khz);
		goto out3;
	}

	sh7764_i2c_outb(drvdata, ICCCR, ccr);

	irq = platform_get_irq(pdev, 0);

	if (request_irq(irq, sh7764_i2c_irq, IRQF_DISABLED, 
			SH7764_I2C_DEVNAME, drvdata)) {
		dev_err(&pdev->dev, "cannot get irq %d\n", irq);
		ret = -EBUSY;
		goto out3;
	}

	i2c_add_numbered_adapter(&drvdata->adapter);

	platform_set_drvdata(pdev, drvdata);

	dev_info(&pdev->dev, "%d kHz ICCCR=%02x mmio %08x-%08x irq %d\n", 
		 pdata->speed_khz, sh7764_i2c_inb(drvdata, ICCCR),
		 res->start, res->end, irq);
	return 0;
out3:
	
out1:
	kfree(drvdata);
	
out0:
	return ret;
}

static int sh7764_i2c_remove(struct platform_device *pdev)
{
	struct sh7764_i2c_drvdata *drvdata = platform_get_drvdata(pdev);

	i2c_del_adapter(&drvdata->adapter);

	free_irq(platform_get_irq(pdev, 0), drvdata);
	kfree(drvdata);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static struct platform_driver sh7764_i2c_drv = {
	.driver	= {
		.name	= SH7764_I2C_DEVNAME,
		.owner	= THIS_MODULE,
	},
	.probe		= sh7764_i2c_probe,
	.remove		= __devexit_p(sh7764_i2c_remove),
};

static int __init sh7764_i2c_init(void)
{
	return platform_driver_register(&sh7764_i2c_drv);
}

static void __exit sh7764_i2c_exit(void)
{
	platform_driver_unregister(&sh7764_i2c_drv);
}

module_init(sh7764_i2c_init);
module_exit(sh7764_i2c_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SH7764 I2C bus driver");
MODULE_AUTHOR("ALPHAPROJECT");
