/*
 * SuperH MS104-SH4AG IDE host driver
 *
 * Copyright (C) 2008 ALPHAPROJECT.
 *
 * Based on Lineo Solutions panther-ide.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 */

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/blkdev.h>
#include <linux/errno.h>
#include <linux/hdreg.h>
#include <linux/ide.h>
#include <linux/dma-mapping.h>
#include <linux/device.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/platform_device.h>

#include <asm/dma.h>
#include <asm/io.h>
#include <asm/irq.h>

/* IDE Defination */
#define MS104SH4AG_IDE              0xfff00000      /* HDD IF */
#define MS104SH4AG_IDE_IF_BASE      0xfff00080      /* MS104SH4AG IF */
#define ATAPI_CONTROL            (MS104SH4AG_IDE_IF_BASE+0x0)
#define ATAPI_STATUS             (MS104SH4AG_IDE_IF_BASE+0x4)
#define ATAPI_INT_ENABLE         (MS104SH4AG_IDE_IF_BASE+0x8)
#define ATAPI_PIO_TIMING         (MS104SH4AG_IDE_IF_BASE+0xc)
#define ATAPI_MULTI_TIMING       (MS104SH4AG_IDE_IF_BASE+0x10)
#define ATAPI_ULTRA_TIMING       (MS104SH4AG_IDE_IF_BASE+0x14)
#define ATAPI_DTB_ADR            (MS104SH4AG_IDE_IF_BASE+0x18)
#define ATAPI_DMA_START_ADR      (MS104SH4AG_IDE_IF_BASE+0x1c)
#define ATAPI_DMA_TRANS_CNT      (MS104SH4AG_IDE_IF_BASE+0x20)
#define ATAPI_CONTROL2           (MS104SH4AG_IDE_IF_BASE+0x24)
#define ATAPI_SIG_ST             (MS104SH4AG_IDE_IF_BASE+0x30)
#define ATAPI_BYTE_SWAP          (MS104SH4AG_IDE_IF_BASE+0x3c)
#define ATAPI_READ
#define ATAPI_WRITE

#define ATAPI_CONTROL_RESET     0x00000080
#define ATAPI_CONTROL_MS        0x00000040
#define ATAPI_CONTROL_BUSSEL    0x00000020
#define ATAPI_CONTROL_UDMAEN    0x00000010
#define ATAPI_CONTROL_DESE      0x00000008
#define ATAPI_CONTROL_RW        0x00000004
#define ATAPI_CONTROL_STOP      0x00000002
#define ATAPI_CONTROL_START     0x00000001

#define ATAPI_CONTROL2_WORDSWAP 0x00000002
#define ATAPI_CONTROL2_IFEN     0x00000001

/* STATUS */
#define STATUS_ACT      (0x1<<0)
#define STATUS_NEND     (0x1<<1)
#define STATUS_ERR      (0x1<<2)
#define STATUS_TOUT     (0x1<<3)
#define STATUS_DEVINT   (0x1<<4)
#define STATUS_DEVTERM  (0x1<<5)
#define STATUS_DNEND    (0x1<<6)
#define STATUS_IFERR    (0x1<<7)
#define STATUS_SWERR    (0x1<<8)

/* CONTROL */
#define CONTROL_START   (0x1<<0)
#define CONTROL_STOP    (0x1<<1)
#define CONTROL_RW      (0x1<<2)
#define CONTROL_DESE    (0x1<<3)
#define CONTROL_UDMAEN  (0x1<<4)
#define CONTROL_BUSSEL  (0x1<<5)
#define CONTROL_MS      (0x1<<6)
#define CONTROL_RESET   (0x1<<7)
#define CONTROL_DTCD    (0x1<<9)

/* INT ENABLE */
#define INT_IACT        (0x1<<0)
#define INT_INEND       (0x1<<1)
#define INT_IERR        (0x1<<2)
#define INT_ITOUT       (0x1<<3)
#define INT_IDEVINT     (0x1<<4)
#define INT_IDEVTRM     (0x1<<5)
#define INT_IDNEND      (0x1<<6)
#define INT_IIFERR      (0x1<<7)
#define INT_ISWERR      (0x1<<8)

#if 1 /* Do not care DMA Interrupt. */
#define INT_ENABLE_MASK  (INT_IERR | INT_ITOUT | INT_IDEVINT | INT_IIFERR | INT_ISWERR)
#else
#define INT_ENABLE_MASK (INT_INEND | INT_IERR | INT_ITOUT | INT_IDEVINT | INT_IDEVTRM | INT_IDNEND | INT_IIFERR | INT_ISWERR)
#endif
#define INT_DISABLE_MASK        (INT_IERR | INT_ITOUT | INT_IDEVINT | INT_IDEVTRM |INT_IIFERR | INT_ISWERR)

/* SG */
#define SG_STOP         (0x1<<31)
#define SG_ADDR_MSK     0x1ffffffc
#define SG_COUNT_MSK    0x1ffffffe

#undef	IDEDEBUG
//#define IDEDEBUG
//#include "ide-timing.h"
#if defined(IDEDEBUG)
#define	 PRINTK printk
#else
#define	 PRINTK(a,...)
#endif
#define USE_DISCRIPTER

#ifdef IDEDEBUG
static void ms104sh4ag_dma_dbg(void)
{
	printk("========= CTRL =========\n");
	printk(" CONTROL       0x%08x\n", ctrl_inl(ATAPI_CONTROL));
	printk(" STATUS        0x%08x\n", ctrl_inl(ATAPI_STATUS));
	printk(" INT_ENABLE    0x%08x\n", ctrl_inl(ATAPI_INT_ENABLE));
	printk(" PIO_TIMING    0x%08x\n", ctrl_inl(ATAPI_PIO_TIMING));
	printk(" MULTI_TIMING  0x%08x\n", ctrl_inl(ATAPI_MULTI_TIMING));
	printk(" ULTRA_TIMING  0x%08x\n", ctrl_inl(ATAPI_ULTRA_TIMING));
	printk(" DTB_ADR       0x%08x\n", ctrl_inl(ATAPI_DTB_ADR));
	printk(" DMA_START_ADR 0x%08x\n", ctrl_inl(ATAPI_DMA_START_ADR));
	printk(" DMA_TRANS_CNT 0x%08x\n", ctrl_inl(ATAPI_DMA_TRANS_CNT));
	printk(" CONTROL2      0x%08x\n", ctrl_inl(ATAPI_CONTROL2));
	printk(" SIG_ST        0x%08x\n", ctrl_inl(ATAPI_SIG_ST));
	printk(" BYTE_SWAP     0x%08x\n", ctrl_inl(ATAPI_BYTE_SWAP));
	printk("========================\n");
}
#endif

static int dma_channel_active(unsigned int channel)
{
	return ctrl_inl(ATAPI_STATUS) & STATUS_ACT;
}

static void set_dma_mode(unsigned int chan, char mode)
{
	unsigned int control = ctrl_inl(ATAPI_CONTROL);

	if (chan)
		control &= ~CONTROL_MS; /* slave */
	else
		control |= CONTROL_MS;
	if (mode == DMA_MODE_WRITE)
		control &= ~CONTROL_RW;
	else
		control |= CONTROL_RW;
	ctrl_outl(control, ATAPI_CONTROL);
}

/* set pixlbus wait (100MHz base) */ 
static void set_dma_speed(unsigned int channel, ide_drive_t *drive,
			  int cycle_ns)
{
	ide_hwif_t *hwif = drive->hwif;
	int is_slave = (hwif->devices[1] == drive);
	int wait_mode = 0;	/* 0:PIO 1:MW 2:UDMA */
	unsigned int wait_val, wait_mask;
	unsigned int wait_cycle = 0x0d44;

//	printk("%s: cur_speed=%02xh\n", __func__, drive->current_speed);

	switch (drive->current_speed) {
        case XFER_UDMA_4:
		wait_mode = 2;
		wait_cycle = 0x00af;
		break;
        case XFER_UDMA_3:
		wait_mode = 2;
		wait_cycle = 0x00ab;
		break;
        case XFER_UDMA_2:
		wait_mode = 2;
		wait_cycle = 0x00cb;
		break;
        case XFER_UDMA_1:
		wait_mode = 2;
		wait_cycle = 0x010e;
		break;
        case XFER_UDMA_0:
		wait_mode = 2;
		wait_cycle = 0x0192;
		break;
	case XFER_MW_DMA_2:
		wait_mode = 1;
		wait_cycle = 0x01a8;
		break;
	case XFER_MW_DMA_1:
		wait_mode = 1;
		wait_cycle = 0x0209;
		break;
	case XFER_MW_DMA_0:
		wait_mode = 1;
		wait_cycle = 0x0637;
		break;
	case XFER_SW_DMA_2:
	case XFER_SW_DMA_1:
	case XFER_SW_DMA_0:
		break;
        case XFER_PIO_4:
		wait_mode = 0;
		wait_cycle = 0x0d44;
		break;
        case XFER_PIO_3:
		wait_mode = 0;
		wait_cycle = 0x134c;
		break;
        case XFER_PIO_2:
		wait_mode = 0;
		wait_cycle = 0x22f4;
		break;
	case XFER_PIO_1:
		wait_mode = 0;
		wait_cycle = 0x28f6;
		break;
	case XFER_PIO_0:
	case XFER_PIO_SLOW:
		wait_mode = 0;
		wait_cycle = 0x3df7;
		break;
	}
	if (is_slave) {
		wait_mask = 0x0000ffff;
		wait_cycle <<= 16;
	} else {
		wait_mask = 0xffff0000;
	}
	switch(wait_mode) {
	case 2:
		wait_val = (ctrl_inl(ATAPI_ULTRA_TIMING) & wait_mask) |
				     wait_cycle;	
		ctrl_outl(wait_val, ATAPI_ULTRA_TIMING);
		break;
	case 1:
		wait_val = (ctrl_inl(ATAPI_MULTI_TIMING) & wait_mask) |
				     wait_cycle;	
		ctrl_outl(wait_val, ATAPI_MULTI_TIMING);
		break;
	case 0:
	default:
		wait_val = (ctrl_inl(ATAPI_PIO_TIMING) & wait_mask) |
				     wait_cycle;	
		ctrl_outl(wait_val, ATAPI_PIO_TIMING);
		break;
	}
}

/* Set DMA Scatter-Gather list
 */
void set_dma_sg(unsigned int channel, ide_drive_t *drive, 
		struct scatterlist *sg, int nr_sg)
{
	ide_hwif_t *hwif = drive->hwif;
	unsigned long *buffer, *phys;	
	int i;

	buffer = (unsigned long *)hwif->dmatable_cpu;
	phys = (unsigned long *)hwif->dmatable_dma;

	//printk("%s: nr_sg=%d cpu=%p dma=%p\n", __func__, nr_sg, buffer, phys);

	for (i = 0; i < nr_sg; i++, sg++) {
		*buffer++ = sg->dma_address;
		*buffer++ = sg->length;
	}
	buffer -= 2;
	*buffer |= SG_STOP; 

	ctrl_outl((unsigned int)phys, ATAPI_DTB_ADR);
	/* discripter */
	if (drive->current_speed >= XFER_UDMA_0)
		ctrl_outl(ctrl_inl(ATAPI_CONTROL) |
			  CONTROL_DESE |
			  CONTROL_UDMAEN,
			  ATAPI_CONTROL);
	else
		ctrl_outl((ctrl_inl(ATAPI_CONTROL) |
			   CONTROL_DESE) &
			  ~CONTROL_UDMAEN,
			  ATAPI_CONTROL);
}

/*
 * Handle routing of interrupts.  This is called before
 * we write the command to the drive.
 */
static void sh7764atapi_maskproc(ide_drive_t *drive, int mask)
{
	ide_hwif_t *hwif = drive->hwif;
	unsigned long flags;
	int is_slave, control;

	local_irq_save(flags);
	is_slave  = (hwif->devices[1] == drive);
	control = ctrl_inl(ATAPI_CONTROL);
	if (is_slave)
		control &= ~CONTROL_MS; /* slave */
	else
		control |= CONTROL_MS;
	ctrl_outl(control, ATAPI_CONTROL);
	if (mask)
		ctrl_outl(0, ATAPI_INT_ENABLE);
	else
		ctrl_outl(INT_ENABLE_MASK, ATAPI_INT_ENABLE);

	local_irq_restore(flags);
}

#ifdef CONFIG_BLK_DEV_IDEDMA_MS104SH4AG
/*
 * SG-DMA support.
 *
 * Similar to the BM-DMA, but we use the RiscPCs IOMD DMA controllers.
 * There is only one DMA controller per card, which means that only
 * one drive can be accessed at one time.  NOTE! We do not enforce that
 * here, but we rely on the main IDE driver spotting that both
 * interfaces use the same IRQ, which should guarantee this.
 */
#define NR_ENTRIES 256
#define TABLE_SIZE (NR_ENTRIES * 8)

static void sh7764atapi_set_dma_mode(ide_drive_t *drive, const u8 mode)
{
	int cycle_time = 0, use_dma_info = 0;
	int use_udma_info = 0;
	u8 xfer_mode = mode;

//	printk("%s: 0x%02x\n", __func__, xfer_mode);

	/*
	 * Limit the transfer speed to XFER_UDMA_4
	 */
	if (xfer_mode > XFER_UDMA_4)
		xfer_mode = XFER_UDMA_4;
	switch (xfer_mode) {
        case XFER_UDMA_4:
		cycle_time = 30;
		use_dma_info = 1;
		use_udma_info = 1;
		break;
        case XFER_UDMA_3:
		cycle_time = 50;
		use_dma_info = 1;
		use_udma_info = 1;
		break;
        case XFER_UDMA_2:
		cycle_time = 60;
		use_dma_info = 1;
		break;
        case XFER_UDMA_1:
		cycle_time = 80;
		use_dma_info = 1;
		break;
        case XFER_UDMA_0:
		cycle_time = 120;
		use_dma_info = 1;
		break;
	case XFER_MW_DMA_2:
		cycle_time = 130;
		use_dma_info = 1;
		break;
	case XFER_MW_DMA_1:
		cycle_time = 320;
		use_dma_info = 1;
		break;
	case XFER_MW_DMA_0:
		cycle_time = 490;
		use_dma_info = 1;
		break;
        case XFER_PIO_4:
		cycle_time = 130;
		break;
        case XFER_PIO_3:
		cycle_time = 190;
		break;
        case XFER_PIO_2:
		cycle_time = 340;
		break;
	case XFER_PIO_1:
		cycle_time = 400;
		break;
	case XFER_PIO_0:
	case XFER_PIO_SLOW:
		cycle_time = 610;
		break;
	case XFER_SW_DMA_2: /* no support */
	case XFER_SW_DMA_1:
	case XFER_SW_DMA_0:
		xfer_mode = XFER_PIO_0;
		cycle_time = 610;
		break;
	}

	drive->drive_data = cycle_time;

	printk("%s: %s selected (peak %dMB/s) %d ns\n", drive->name,
		ide_xfer_verbose(xfer_mode), 2000 / drive->drive_data,
		cycle_time);
}

static u8 sh7764atapi_inb(unsigned long port)
{
	return ctrl_inl(port);
}

static u16 sh7764atapi_inw(unsigned long port)
{
	return ctrl_inl(port);
}

static void sh7764atapi_insw(unsigned long port, void *addr, u32 count)
{
	unsigned long reg = port;
	u16 *buf = addr;

	while (count--)
		*buf++ = (u16)ctrl_inl(reg);
}

static void sh7764atapi_outb(u8 value, unsigned long port)
{
	ctrl_outl(value, port);
}

static void sh7764atapi_outw(u16 value, unsigned long port)
{
	ctrl_outl(value, port);
}

static void sh7764atapi_outsw(unsigned long port, void *addr, u32 count)
{
	unsigned long reg = port;
	u16 *buf = addr;

	while (count--)
		ctrl_outl((u32)*buf++, reg);
}

static void sh7764atapi_input_data(ide_drive_t *drive, struct request *rq,
				   void *buf, unsigned int len)
{
//	printk("%s: len=%d\n", __func__, len);
	sh7764atapi_insw(drive->hwif->io_ports.data_addr, buf, (len + 1) / 2);
}

static void sh7764atapi_output_data(ide_drive_t *drive, struct request *rq,
				    void *buf, unsigned int len)
{
//	printk("%s: len=%d\n", __func__, len);
	sh7764atapi_outsw(drive->hwif->io_ports.data_addr, buf, (len + 1) / 2);
}


static void sh7764atapi_set_pio_mode(ide_drive_t *drive, const u8 mode)
{
	sh7764atapi_set_dma_mode(drive, mode + XFER_PIO_0);
}

static void sh7764atapi_dma_host_set (ide_drive_t *drive, int on)
{
	pr_debug("%s: %s\n", __func__, on?"on":"off");
}

static int sh7764atapi_dma_end(ide_drive_t *drive)
{
	unsigned int sts, ctrl;

	drive->waiting_for_dma = 0;

	/* disable dma */
//	ctrl_outl(0, ATAPI_INT_ENABLE);
	ctrl = ctrl_inl(ATAPI_CONTROL); /* Get Status */
	
	ctrl |= CONTROL_STOP;                  /* SET DMA stop bit */
	ctrl &= ~(CONTROL_START);              /* Disable DMA Start bit */
	ctrl_outl(ctrl, ATAPI_CONTROL); /* Update control register */
	
	sts = ctrl_inl(ATAPI_STATUS);   /* Get Status register val */
	ctrl_outl(0, ATAPI_STATUS);     /* Clear All Status Register */

	/* Teardown mappings after DMA has completed. */

	ide_destroy_dmatable(drive);

	if (sts & (STATUS_SWERR | STATUS_IFERR | STATUS_TOUT)) {
		printk(KERN_ERR "%s:DMA Status:{%s %s %s}\n",drive->name,
		       ((sts&STATUS_SWERR)?"SWERR":""),
		       ((sts&STATUS_IFERR)?"IFERR":""),
		       ((sts&STATUS_TOUT)?"TOUT":""));
		return 1;
	}
	
	return !(ctrl_inl(ATAPI_STATUS) & STATUS_DEVINT);
}

static void sh7764atapi_dma_start(ide_drive_t *drive)
{
	BUG_ON(dma_channel_active(0/*hwif->hw.dma*/));

	if ( ctrl_inl(ATAPI_STATUS )) {
		printk(KERN_WARNING "IDE:DMA abnormal status 0x%08x\n" ,
		       ctrl_inl(ATAPI_STATUS));
		ctrl_outl(0,ATAPI_STATUS ); /* All Status Clear */
	}
	
	ctrl_outl(INT_ENABLE_MASK, ATAPI_INT_ENABLE);
#ifdef IDEDEBUG
	ms104sh4ag_dma_dbg();
#endif
	ctrl_outl(ctrl_inl(ATAPI_CONTROL) | CONTROL_START | CONTROL_BUSSEL,
		  ATAPI_CONTROL);
}

static int sh7764atapi_dma_setup(ide_drive_t *drive)
{
	ide_hwif_t *hwif = drive->hwif;
	int is_slave  = (hwif->devices[1] == drive);
	struct request *rq = hwif->rq;
	unsigned int dma_mode;

	if (rq_data_dir(rq))
		dma_mode = DMA_MODE_WRITE;
	else
		dma_mode = DMA_MODE_READ;
	/*
	 * We can not enable DMA on both channels.
	 */
	BUG_ON(dma_channel_active(0/*hwif->hw.dma*/));

	hwif->sg_nents = ide_build_sglist(drive, rq);
	/*
	 * Ensure that we have the right interrupt routed.
	 */
	sh7764atapi_maskproc(drive, 0);

	/*
	 * Select the correct timing for this drive.
	 */
	set_dma_speed(0/*hwif->hw.dma*/, drive, drive->drive_data);

	/*
	 * Tell the DMA engine about the SG table and
	 * data direction.
	 */
	set_dma_sg(0/*hwif->hw.dma*/, drive, hwif->sg_table, hwif->sg_nents);
	set_dma_mode(is_slave, dma_mode);

	drive->waiting_for_dma = 1;

	return 0;
}

static void sh7764atapi_dma_exec_cmd(ide_drive_t *drive, u8 cmd)
{
	/* issue cmd to drive */
	pr_debug("%s: %02xh\n", __func__, cmd);
	ide_execute_command(drive, cmd, ide_dma_intr, 2 * WAIT_CMD, NULL);
}

static int sh7764atapi_dma_test_irq(ide_drive_t *drive)
{
	return (ctrl_inl(ATAPI_STATUS) & STATUS_DEVINT) ? 1 : 0;
}

static u8 sh7764atapi_udma_filter(ide_drive_t *drive)
{
	return 0x1F;
}

static u8 sh7764atapi_cable_detect(struct hwif_s *hwif)
{
	return ATA_CBL_PATA40_SHORT;
}

static const struct ide_dma_ops sh7764atapi_dma_ops = {
	.dma_host_set		= sh7764atapi_dma_host_set,
	.dma_setup		= sh7764atapi_dma_setup,
	.dma_exec_cmd		= sh7764atapi_dma_exec_cmd,
	.dma_start		= sh7764atapi_dma_start,
	.dma_end		= sh7764atapi_dma_end,
	.dma_test_irq		= sh7764atapi_dma_test_irq,
	.dma_lost_irq		= ide_dma_lost_irq,
	.dma_timeout		= ide_dma_timeout,
};

static int sh7764atapi_dma_init(ide_hwif_t *hwif,
				const struct ide_port_info *d)
{
//	int autodma = 1;

	printk("    %s: SG-DMA\n", hwif->name);

	hwif->ultra_mask	= 0x1f; /* U0..4 */
	hwif->mwdma_mask	= 7; /* MW0..2 */
	hwif->swdma_mask	= 0; /* SW0..2 */

	ctrl_outl(0x0, ATAPI_BYTE_SWAP);
	ctrl_outl(ATAPI_CONTROL2_WORDSWAP|ATAPI_CONTROL2_IFEN,
		  ATAPI_CONTROL2);

	ctrl_outl(0x0d440d44, ATAPI_PIO_TIMING);
	ctrl_outl(0x01ab01ab, ATAPI_MULTI_TIMING);
	ctrl_outl(0x006b006b, ATAPI_ULTRA_TIMING);

	ctrl_outl(INT_ENABLE_MASK | INT_DISABLE_MASK, ATAPI_INT_ENABLE);

	ide_allocate_dma_engine(hwif);
	//ctrl_outl(0, ATAPI_INT_ENABLE);

	return 0;
}

#else
 #define sh7764atapi_dma_init(hwif)	(0)
 #define sh7764atapi_dma_exit(hwif)	do { } while (0)
#endif

static void sh7764atapi_init_ports(hw_regs_t *hw,
				   unsigned long io_addr,
				   unsigned long ctl_addr)
{
	unsigned int i;

	for (i = 0; i <= 7; i++)
		hw->io_ports_array[i] = io_addr + i * 4;

	hw->io_ports.ctl_addr = ctl_addr;	
}

static void sh7764atapi_exec_command(ide_hwif_t *hwif, u8 cmd)
{
	sh7764atapi_outb(cmd, hwif->io_ports.command_addr);
}

static u8 sh7764atapi_read_status(ide_hwif_t *hwif)
{
	return sh7764atapi_inb(hwif->io_ports.status_addr);
}

static u8 sh7764atapi_read_altstatus(ide_hwif_t *hwif)
{
	return sh7764atapi_inb(hwif->io_ports.ctl_addr);
}

static void sh7764atapi_set_irq(ide_hwif_t *hwif, int on)
{
	u8 ctl = ATA_DEVCTL_OBS;

	if (on == 4) { /* hack for SRST */
		ctl |= 4;
		on &= ~4;
	}

	ctl |= on ? 0 : 2;

	sh7764atapi_outb(ctl, hwif->io_ports.ctl_addr);
}

static void sh7764atapi_tf_load(ide_drive_t *drive, ide_task_t *task)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;
	struct ide_taskfile *tf = &task->tf;
	void (*tf_outb)(u8 addr, unsigned long port) = sh7764atapi_outb;
	u8 HIHI = (task->tf_flags & IDE_TFLAG_LBA48) ? 0xE0 : 0xEF;

	if (task->tf_flags & IDE_TFLAG_FLAGGED)
		HIHI = 0xFF;

	if (task->tf_flags & IDE_TFLAG_OUT_DATA) {
		u16 data = (tf->hob_data << 8) | tf->data;

		sh7764atapi_outw(data, io_ports->data_addr);
	}

	if (task->tf_flags & IDE_TFLAG_OUT_HOB_FEATURE)
		tf_outb(tf->hob_feature, io_ports->feature_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_HOB_NSECT)
		tf_outb(tf->hob_nsect, io_ports->nsect_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_HOB_LBAL)
		tf_outb(tf->hob_lbal, io_ports->lbal_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_HOB_LBAM)
		tf_outb(tf->hob_lbam, io_ports->lbam_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_HOB_LBAH)
		tf_outb(tf->hob_lbah, io_ports->lbah_addr);

	if (task->tf_flags & IDE_TFLAG_OUT_FEATURE)
		tf_outb(tf->feature, io_ports->feature_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_NSECT)
		tf_outb(tf->nsect, io_ports->nsect_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_LBAL)
		tf_outb(tf->lbal, io_ports->lbal_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_LBAM)
		tf_outb(tf->lbam, io_ports->lbam_addr);
	if (task->tf_flags & IDE_TFLAG_OUT_LBAH)
		tf_outb(tf->lbah, io_ports->lbah_addr);

	if (task->tf_flags & IDE_TFLAG_OUT_DEVICE)
		tf_outb((tf->device & HIHI) | drive->select,
			 io_ports->device_addr);
}

static void sh7764atapi_tf_read(ide_drive_t *drive, ide_task_t *task)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;
	struct ide_taskfile *tf = &task->tf;
	void (*tf_outb)(u8 addr, unsigned long port) = sh7764atapi_outb;
	u8 (*tf_inb)(unsigned long port) = sh7764atapi_inb;

	if (task->tf_flags & IDE_TFLAG_IN_DATA) {
		u16 data;

		data = sh7764atapi_inw(io_ports->data_addr);

		tf->data = data & 0xff;
		tf->hob_data = (data >> 8) & 0xff;
	}

	tf_outb(ATA_DEVCTL_OBS & ~0x80, io_ports->ctl_addr);

	if (task->tf_flags & IDE_TFLAG_IN_FEATURE)
		tf->feature = tf_inb(io_ports->feature_addr);
	if (task->tf_flags & IDE_TFLAG_IN_NSECT)
		tf->nsect  = tf_inb(io_ports->nsect_addr);
	if (task->tf_flags & IDE_TFLAG_IN_LBAL)
		tf->lbal   = tf_inb(io_ports->lbal_addr);
	if (task->tf_flags & IDE_TFLAG_IN_LBAM)
		tf->lbam   = tf_inb(io_ports->lbam_addr);
	if (task->tf_flags & IDE_TFLAG_IN_LBAH)
		tf->lbah   = tf_inb(io_ports->lbah_addr);
	if (task->tf_flags & IDE_TFLAG_IN_DEVICE)
		tf->device = tf_inb(io_ports->device_addr);

	if (task->tf_flags & IDE_TFLAG_LBA48) {
		tf_outb(ATA_DEVCTL_OBS | 0x80, io_ports->ctl_addr);

		if (task->tf_flags & IDE_TFLAG_IN_HOB_FEATURE)
			tf->hob_feature = tf_inb(io_ports->feature_addr);
		if (task->tf_flags & IDE_TFLAG_IN_HOB_NSECT)
			tf->hob_nsect   = tf_inb(io_ports->nsect_addr);
		if (task->tf_flags & IDE_TFLAG_IN_HOB_LBAL)
			tf->hob_lbal    = tf_inb(io_ports->lbal_addr);
		if (task->tf_flags & IDE_TFLAG_IN_HOB_LBAM)
			tf->hob_lbam    = tf_inb(io_ports->lbam_addr);
		if (task->tf_flags & IDE_TFLAG_IN_HOB_LBAH)
			tf->hob_lbah    = tf_inb(io_ports->lbah_addr);
	}
}

static const struct ide_tp_ops sh7764atapi_tp_ops = {
	.exec_command		= sh7764atapi_exec_command,
	.read_status		= sh7764atapi_read_status,
	.read_altstatus		= sh7764atapi_read_altstatus,

	.set_irq		= sh7764atapi_set_irq,

	.tf_load		= sh7764atapi_tf_load,
	.tf_read		= sh7764atapi_tf_read,

	.input_data		= sh7764atapi_input_data,
	.output_data		= sh7764atapi_output_data,
};

static const struct ide_port_ops sh7764atapi_port_ops = {
	.set_pio_mode		= sh7764atapi_set_pio_mode,
	.set_dma_mode		= sh7764atapi_set_dma_mode,
	.udma_filter		= sh7764atapi_udma_filter,
	.cable_detect		= sh7764atapi_cable_detect,
};

static struct ide_port_info sh7764atapi_port_info = {
	.name		= "SH7764-ATAPI",
	.init_dma	= sh7764atapi_dma_init,
	.tp_ops		= &sh7764atapi_tp_ops,
	.port_ops	= &sh7764atapi_port_ops,
	.dma_ops	= &sh7764atapi_dma_ops,
	.host_flags	= IDE_HFLAG_MMIO,
	.pio_mask	= ATA_PIO4,
	.mwdma_mask	= ATA_MWDMA2,
	.udma_mask	= ATA_UDMA4,
};

#define DRV_NAME "ms104sh4ag-ide"
static int ms104sh4ag_ide_probe(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ide_host *host;
	struct resource *res;
	int ret = 0;
	hw_regs_t hw, *hws[] = {&hw, NULL, NULL, NULL};
	int irq;
	unsigned long base = 0x1F0;

	irq = platform_get_irq(pdev, 0);
	base = 0xfff00000;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (res == NULL) {
		pr_err("%s %d: no base address\n", DRV_NAME, pdev->id);
		ret = -ENODEV;
		goto out;
	}
	if (irq < 0) {
		pr_err("%s %d: no IRQ\n", DRV_NAME, pdev->id);
		ret = -ENODEV;
		goto out;
	}

	if (!request_mem_region(res->start, res->end - res->start + 1,
				pdev->name)) {
		pr_err("%s: request_mem_region failed\n", DRV_NAME);
		ret =  -EBUSY;
		goto out;
	}

	memset(&hw, 0, sizeof(hw));
	sh7764atapi_init_ports(&hw, base, base + 0x206);
	hw.irq = irq;
	hw.dev = dev;
	hw.chipset = ide_generic;

	ret = ide_host_add(&sh7764atapi_port_info, hws, &host);
	if (ret)
		goto out;

	dev_set_drvdata(dev, host);

	printk(KERN_INFO "MS104SH4AG IDE(builtin)\n" );

 out:
	return ret;
}

static int ms104sh4ag_ide_remove(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct resource *res;
	struct ide_host *host = dev_get_drvdata(dev);

	ide_host_remove(host);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	release_mem_region(res->start, res->end - res->start + 1);

	return 0;
}

static struct device_driver ms104sh4ag_ide_driver = {
	.name		= "ms104sh4ag-ide",
	.bus		= &platform_bus_type,
	.probe 		= ms104sh4ag_ide_probe,
	.remove		= ms104sh4ag_ide_remove,
};

static int __init sh7764atapi_init(void)
{
	return driver_register(&ms104sh4ag_ide_driver);
}

static void __exit sh7764atapi_exit(void)
{
	driver_unregister(&ms104sh4ag_ide_driver);
}


MODULE_AUTHOR("ALPHAPROJECT");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SH7764 ATAPI driver");

module_init(sh7764atapi_init);
module_exit(sh7764atapi_exit);
