/*
 * Ricoh RS5C316 RTC device/driver
 * Copyright (C) 2009 ALPHAPROJECT
 *
 * Based on the old drivers/rtc/rtc-rs5c313_rtc.c
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/rtc.h>
#include <linux/platform_device.h>
#include <linux/bcd.h>
#include <linux/delay.h>
#include <asm/io.h>

#define DRV_NAME	"rs5c316"
#define DRV_VERSION 	"0.1"

#ifdef CONFIG_SH_MS104SH4

#define PCTRA 0xff80002c
#define PDTRA 0xff800030
#endif

#define RS5C316_SEC1		0x0
#define RS5C316_SEC10		0x1
#define RS5C316_MIN1		0x2
#define RS5C316_MIN10		0x3
#define RS5C316_HOUR1		0x4
#define RS5C316_HOUR10		0x5
#define RS5C316_WDAY		0x6
#define RS5C316_DAY1		0x8
#define RS5C316_DAY10		0x9
#define RS5C316_MON1		0xa
#define RS5C316_MON10		0xb
#define RS5C316_YEAR1		0xc
#define RS5C316_YEAR10		0xd
#define RS5C316_CONTROL1	0xe
#define RS5C316_CONTROL2	0xf

static void rs5c316_chip_enable(void)
{
	ctrl_outw(ctrl_inw(PDTRA) | 0x8000, PDTRA);
}

static void rs5c316_chip_disable(void)
{
	ctrl_outw(ctrl_inw(PDTRA) & ~0x8000, PDTRA);
}

static void rs5c316_mode_wr(void)
{
	ctrl_outl(ctrl_inl(PCTRA) | 0x04000000, PCTRA);
}

static void rs5c316_mode_rd(void)
{
	ctrl_outl(ctrl_inl(PCTRA) & ~0x04000000, PCTRA);
}

static void rs5c316_sclk(int b)
{
	if (b)
                ctrl_outw(ctrl_inw(PDTRA) | 0x4000, PDTRA);
        else
                ctrl_outw(ctrl_inw(PDTRA) & ~0x4000, PDTRA);
}

static void rs5c316_write_bit(int bit)
{
	rs5c316_sclk(1);

	udelay(1);

	if (bit)
                ctrl_outw(ctrl_inw(PDTRA) | 0x2000, PDTRA);
        else
                ctrl_outw(ctrl_inw(PDTRA) & ~0x2000, PDTRA);

        udelay(1);

        rs5c316_sclk(0);

        udelay(1);

}

static bool rs5c316_read_bit(void)
{
        unsigned short val;

        rs5c316_sclk(1);

        udelay(1);

        val = ctrl_inw(PDTRA);

        udelay(1);

        rs5c316_sclk(0);

        udelay(1);

//      printf("%s val=%d\n", __func__, val & RS5C316_SIO);

        return val & 0x2000;
}


static int rs5c316_read(int addr)
{
	int data;

	rs5c316_mode_wr();
	rs5c316_write_bit(0);
	rs5c316_write_bit(1);
	rs5c316_write_bit(1);
	rs5c316_write_bit(0);
	
        rs5c316_write_bit((addr & 0x08) >> 3);
        rs5c316_write_bit((addr & 0x04) >> 2);
        rs5c316_write_bit((addr & 0x02) >> 1);
        rs5c316_write_bit(addr & 0x01);

        rs5c316_mode_rd();

        rs5c316_read_bit();     /* dummy */
        rs5c316_read_bit();
        rs5c316_read_bit();
        rs5c316_read_bit();
        data = rs5c316_read_bit() << 3;
        data |= rs5c316_read_bit() << 2;
        data |= rs5c316_read_bit() << 1;
        data |= rs5c316_read_bit();

        return data;

}

static void rs5c316_write(int data, int addr)
{
	rs5c316_mode_wr();

        rs5c316_write_bit(0);   /* dummy */
        rs5c316_write_bit(0);   /* R/W = 0 */
        rs5c316_write_bit(1);   /* AD  = 1 */
        rs5c316_write_bit(0);   /* DT  = 0 */

        rs5c316_write_bit((addr & 0x08) >> 3);
        rs5c316_write_bit((addr & 0x04) >> 2);
        rs5c316_write_bit((addr & 0x02) >> 1);
        rs5c316_write_bit(addr & 0x01);

        rs5c316_write_bit(0);   /* dummy */
        rs5c316_write_bit(0);   /* R/W = 0 */
        rs5c316_write_bit(0);   /* AD  = 0 */
        rs5c316_write_bit(1);   /* DT  = 1 */

        rs5c316_write_bit((data & 0x08) >> 3);
        rs5c316_write_bit((data & 0x04) >> 2);
        rs5c316_write_bit((data & 0x02) >> 1);
        rs5c316_write_bit(data & 0x01);
}

static void rs5c316_enable(void)
{
	int data;

	rs5c316_chip_enable();

	data = rs5c316_read(RS5C316_CONTROL1);
//	printk("%s: data=%x\n", __func__, data);

	rs5c316_write(0x9, RS5C316_CONTROL2);

	rs5c316_chip_disable();
}

static int rs5c316_rtc_read_time(struct device *dev, struct rtc_time *tm)
{
	int i = 0;

//	printk("%s\n", __func__);

	rs5c316_chip_enable();

	rs5c316_write(0x0, RS5C316_CONTROL1);
	while (rs5c316_read(RS5C316_CONTROL1) & 0x1) {
		rs5c316_chip_disable();
		if (i++ > 8)
			return -EIO;
		udelay(1 << i);
		rs5c316_chip_enable();
	}

	tm->tm_sec = bcd2bin(rs5c316_read(RS5C316_SEC1) |
			     (rs5c316_read(RS5C316_SEC10) << 4));
	tm->tm_min = bcd2bin(rs5c316_read(RS5C316_MIN1) |
			     (rs5c316_read(RS5C316_MIN10) << 4));
	tm->tm_hour = bcd2bin(rs5c316_read(RS5C316_HOUR1) |
			      (rs5c316_read(RS5C316_HOUR10) << 4));
	tm->tm_mday = bcd2bin(rs5c316_read(RS5C316_DAY1) |
			      (rs5c316_read(RS5C316_DAY10) << 4));
	tm->tm_mon = bcd2bin(rs5c316_read(RS5C316_MON1) |
			     (rs5c316_read(RS5C316_MON10) << 4));
	tm->tm_year = bcd2bin(rs5c316_read(RS5C316_YEAR1) |
			      (rs5c316_read(RS5C316_YEAR10) << 4));
	tm->tm_wday = rs5c316_read(RS5C316_WDAY);

	rs5c316_chip_disable();

	if (tm->tm_year < 70)
		tm->tm_year += 100;
	tm->tm_mon -= 1;

//	printk("%d/%d/%d %02d:%02d:%02d\n", tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);

	return 0;
}

static int rs5c316_rtc_set_time(struct device *dev, struct rtc_time *tm)
{
	int i = 0;
	unsigned char data;
#if 0
	printk("%s: %02d:%02d:%02d %d/%d/%d %d %d %d\n", __func__,
	       tm->tm_hour, tm->tm_min, tm->tm_sec, 
	       tm->tm_year, tm->tm_mon, tm->tm_mday, 
	       tm->tm_wday, tm->tm_yday, tm->tm_isdst);
#endif
	rs5c316_chip_enable();

	rs5c316_write(0x0, RS5C316_CONTROL1);
	while (rs5c316_read(RS5C316_CONTROL1) & 0x1) {
		rs5c316_chip_disable();
		if (i++ > 8)
			return -EIO;
		udelay(1 << i);
		rs5c316_chip_enable();
	}

	data = bin2bcd(tm->tm_sec);
	rs5c316_write(data, RS5C316_SEC1);
	rs5c316_write((data >> 4), RS5C316_SEC10);

	data = bin2bcd(tm->tm_min);
	rs5c316_write(data, RS5C316_MIN1);
	rs5c316_write((data >> 4), RS5C316_MIN10);

	data = bin2bcd(tm->tm_hour);
	rs5c316_write(data, RS5C316_HOUR1);
	rs5c316_write((data >> 4), RS5C316_HOUR10);

	data = bin2bcd(tm->tm_mday);
	rs5c316_write(data, RS5C316_DAY1);
	rs5c316_write((data>> 4), RS5C316_DAY10);

	data = bin2bcd(tm->tm_mon + 1);
	rs5c316_write(data, RS5C316_MON1);
	rs5c316_write((data >> 4), RS5C316_MON10);

	data = bin2bcd(tm->tm_year % 100);
	rs5c316_write(data, RS5C316_YEAR1);
	rs5c316_write((data >> 4), RS5C316_YEAR10);

	data = bin2bcd(tm->tm_wday);
	rs5c316_write(data, RS5C316_WDAY);

	rs5c316_chip_disable();
	
	return 0;
}

static const struct rtc_class_ops rs5c316_rtc_ops = {
	.read_time = rs5c316_rtc_read_time,
	.set_time = rs5c316_rtc_set_time,
};

static int rs5c316_rtc_probe(struct platform_device *pdev)
{
	struct rtc_device *rtc = 
		rtc_device_register("rs5c316", &pdev->dev,
				    &rs5c316_rtc_ops, THIS_MODULE);

	if (IS_ERR(rtc))
		return PTR_ERR(rtc);

	platform_set_drvdata(pdev, rtc);

	rs5c316_enable();

	return 0;
}

static int __devexit rs5c316_rtc_remove(struct platform_device *pdev)
{
	struct rtc_device *rtc = platform_get_drvdata(pdev);

	rtc_device_unregister(rtc);

	return 0;
}

static struct platform_driver rs5c316_rtc_platform_driver = {
	.driver         = {
		.name   = DRV_NAME,
		.owner  = THIS_MODULE,
	},
	.probe 	= rs5c316_rtc_probe,
	.remove = __devexit_p(rs5c316_rtc_remove),
};

static int __init rs5c316_rtc_init(void)
{
	int err;

	err = platform_driver_register(&rs5c316_rtc_platform_driver);
	if (err)
		return err;

//	rs5c316_init_port();
//	rs5c316_check_xstp_bit();

	return 0;
}

static void __exit rs5c316_rtc_exit(void)
{
	platform_driver_unregister(&rs5c316_rtc_platform_driver);
}

module_init(rs5c316_rtc_init);
module_exit(rs5c316_rtc_exit);

MODULE_VERSION(DRV_VERSION);
MODULE_AUTHOR("ALPHAPROJECT");
MODULE_DESCRIPTION("Ricoh RS5C316 RTC device driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:" DRV_NAME);
