/*
 * Seiko s35190a RTC device/driver
 *  Copyright (C) 2008 ALPHAPROJECT
 *
 * Based on the drivers/rtc/rtc-rs5c313.c
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/rtc.h>
#include <linux/bcd.h>
#include <linux/delay.h>
#include <linux/io.h>

#define DRV_NAME	"rtc-s35190a"
#define DRV_VERSION	"0.0.1"

#define PTIO_A	0xFFF10000
#define PTDAT_A	0xFFF10040


#define S35190A_SR1 0
#define S35190A_SR2 1
#define S35190A_DR1 2

#if 0
#define S35190A_SR1_POC   0x01
#define S35190A_SR1_BLD   0x02
#define S35190A_SR1_1224  0x40
#define S35190A_SR1_RESET 0x80
#else
#define S35190A_SR1_POC   0x80
#define S35190A_SR1_BLD   0x40
#define S35190A_SR1_1224  0x02
#define S35190A_SR1_RESET 0x01
#endif

static void sck_h (void)
{
	udelay(20);
	//	debug("sck H\n");
	ctrl_outw(ctrl_inw(PTDAT_A) | 0x0010, PTDAT_A);
	udelay(20);
}

static void sck_l (void)
{
	udelay(20);
	//	debug("sck L\n");
	ctrl_outw(ctrl_inw(PTDAT_A) & ~0x0010, PTDAT_A);
	udelay(20);
}

static void sio_h (void)
{
  //	debug("sio: 1\n");
	ctrl_outw(ctrl_inw(PTDAT_A) | 0x0020, PTDAT_A);
}

static void sio_l (void)
{
  //	debug("sio: 0\n");
	ctrl_outw(ctrl_inw(PTDAT_A) & ~0x0020, PTDAT_A);
}


static void sio_write_bit (int bit)
{
	if (bit)
		sio_h();
	else
		sio_l();
}

static void sio_write (int bit)
{
	sck_l();

	sio_write_bit(bit);
	udelay(50);
	sck_h();
	udelay(50);
}

static int sio_read_bit (void)
{
	u16 val = ctrl_inw(PTDAT_A);

	return !!(val & 0x0020);
}

static int sio_read (void)
{
	int bit;

	sck_l();
	
	bit = sio_read_bit();
	udelay(50);
	sck_h();
	udelay(50);

	return bit;
}

static void sio_mode_in (void)
{
	ctrl_outw((ctrl_inw(PTIO_A) & ~0x0C00) | 0x0800, PTIO_A);
}

static void sio_mode_out (void)
{
	ctrl_outw((ctrl_inw(PTIO_A) & ~0x0C00) | 0x0400, PTIO_A);
}

static void cs_assert (void)
{
	sck_h();
	udelay(50);
	ctrl_outw(ctrl_inw(PTDAT_A) | 0x0008, PTDAT_A);
	udelay(50);
}

static void cs_negate (void)
{
	ctrl_outw((ctrl_inw(PTDAT_A) & 0xFF) & ~0x0008, PTDAT_A);
}

static void s35190a_send_cmd (int cmd)
{
	sio_write(0);
	sio_write(1);
	sio_write(1);
	sio_write(0);

	sio_write(cmd & 4);
	sio_write(cmd & 2);
	sio_write(cmd & 1);
}

static inline void s35190a_send_cmd_read (int cmd)
{
	s35190a_send_cmd(cmd);
	sio_write(1); // READ
}

static inline void s35190a_send_cmd_write (int cmd)
{
	s35190a_send_cmd(cmd);
	sio_write(0); // WRITE	
}

static void s35190a_read (int cmd, int ndata, u8 *data)
{
	int val, i, j;

	cs_assert();

	sio_mode_out();

	s35190a_send_cmd_read(cmd);

	sio_mode_in();

	for (i = 0; i < ndata; i++) {
		val = 0;
		for (j = 0; j < 8; j++) {
			val |= sio_read() << j;
		}
		data[i] = val;
	}
	
	cs_negate();
}

static u8 s35190a_read1 (int cmd)
{
	u8 data;

	s35190a_read(cmd, 1, &data);

	return data;
}

static void s35190a_write (int cmd, int ndata, u8 *data)
{
	int i, j;

//	debug("WRITE %d byte\n", ndata);

	cs_assert();

	sio_mode_out();

	s35190a_send_cmd_write(cmd);

	for (i = 0; i < ndata; i++) {
		int d = data[i];

//		debug("  w -> %02x\n", d);
		for (j = 0; j < 8; j++, d >>= 1)
			sio_write(d & 1);
	}

	cs_negate();
}

static void s35190a_write1 (int cmd, u8 data)
{
	s35190a_write(cmd, 1, &data);
}

static int s35190a_rtc_read_time (struct device *dev, struct rtc_time *tm)
{
	const int ndata = 7;
	u8 data[ndata];

	s35190a_read(S35190A_DR1, ndata, data);

	tm->tm_sec = bcd2bin(data[6]);
	tm->tm_min = bcd2bin(data[5]);
	tm->tm_hour = bcd2bin(data[4] & 0x3F);
	tm->tm_wday = bcd2bin(data[3]);
	tm->tm_mday = bcd2bin(data[2]);
	tm->tm_mon = bcd2bin(data[1]) - 1;
	tm->tm_year = bcd2bin(data[0]);

	if (tm->tm_year < 70)
		tm->tm_year += 100;

	dev_dbg(dev, "%s: tm is secs=%d, mins=%d, hours=%d, mday=%d, "
		"mon=%d, year=%d, wday=%d\n", __func__, tm->tm_sec,
		tm->tm_min, tm->tm_hour, tm->tm_mday, tm->tm_mon, tm->tm_year,
		tm->tm_wday);

	return rtc_valid_tm(tm);
}

static int s35190a_rtc_set_time (struct device *dev, struct rtc_time *tm)
{
	u8 data[7];

	dev_dbg(dev, "%s: tm is secs=%d, mins=%d, hours=%d mday=%d, "
		"mon=%d, year=%d, wday=%d\n", __func__, tm->tm_sec,
		tm->tm_min, tm->tm_hour, tm->tm_mday, tm->tm_mon, tm->tm_year,
		tm->tm_wday);

	data[0] = bin2bcd(tm->tm_year % 100);
	data[1] = bin2bcd(tm->tm_mon + 1);
	data[2] = bin2bcd(tm->tm_mday);
	data[3] = bin2bcd(tm->tm_wday);
	data[4] = bin2bcd(tm->tm_hour);
	data[5] = bin2bcd(tm->tm_min);
	data[6] = bin2bcd(tm->tm_sec);

	s35190a_write(S35190A_DR1, 7, data);

	return 0;
}

static const struct rtc_class_ops s35190a_rtc_ops = {
	.read_time	= s35190a_rtc_read_time,
	.set_time	= s35190a_rtc_set_time,
};

static int s35190a_rtc_probe (struct platform_device *pdev)
{
	struct rtc_device *rtc = 
		rtc_device_register(DRV_NAME, &pdev->dev,
				    &s35190a_rtc_ops, THIS_MODULE);

	if (IS_ERR(rtc))
		return PTR_ERR(rtc);

	platform_set_drvdata(pdev, rtc);

	return 0;
}

static int __devexit s35190a_rtc_remove (struct platform_device *pdev)
{
	struct rtc_device *rtc = platform_get_drvdata(pdev);

	rtc_device_unregister(rtc);

	return 0;
}

static struct platform_driver s35190a_rtc_platform_driver = {
	.driver = {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
	},
	.probe	= s35190a_rtc_probe,
	.remove	= __devexit_p(s35190a_rtc_remove)
};

static int s35190a_reset (void)
{
	u8 v, v2;

	ctrl_outw((ctrl_inw(PTIO_A) & ~0x0FC0) | 0x0540, PTIO_A);

	mdelay(500);

	cs_negate();

//	s35190a_write1(S35190A_SR1, S35190A_SR1_RESET);
//redo:

	v = s35190a_read1(S35190A_SR1);

	//	debug("v=%02x\n", v);
	s35190a_write1(S35190A_SR1, S35190A_SR1_1224);
	v2 = s35190a_read1(S35190A_SR1);
	//	debug("%02x == %02x\n", v, v2);
	return 0;
}

static int __init s35190a_rtc_init (void)
{
	int err;

	err = platform_driver_register(&s35190a_rtc_platform_driver);
	if (err)
		return err;

	s35190a_reset();

	return 0;
}

static void __exit s35190a_rtc_exit (void)
{
	platform_driver_unregister(&s35190a_rtc_platform_driver);
}

module_init(s35190a_rtc_init);
module_exit(s35190a_rtc_exit);

MODULE_VERSION(DRV_VERSION);
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:" DRV_NAME);
