#include <linux/init.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/bitrev.h>
//#include <liu
#include <asm/spi.h>

#define SCSMR1  0xFFE10000
#define SCBRR1  0xFFE10004
#define SCSCR1  0xFFE10008
#define SCFTDR1	0xFFE1000C
#define SCFSR1	0xFFE10010
#define SCFRDR1	0xFFE10014
#define SCFCR1	0xFFE10018
#define SCFDR1	0xFFE1001C
#define SCSPTR1 0xFFE10020
#define SCLSR1	0xFFE10024

#define FSR_ER		0x0080
#define FSR_TEND	0x0040
#define FSR_TDFE	0x0020
#define FSR_BRK		0x0010
#define FSR_RDF		0x0002
#define FSR_DR		0x0001

#define FCR_RTRG(x)	((x&3)<<6)
#define FCR_TTRG(x)	((x&3)<<4)
#define FCR_TFRRST	0x0004
#define FCR_RFRST	0x0002

#define LSR_ORER	0x0001

struct spi_sh7764 {
	struct workqueue_struct *workqueue;
	struct work_struct work;
	spinlock_t lock;
	struct list_head queue;
	wait_queue_head_t waitq;
	struct sh_spi_info *info;
	int irq;
};

static inline void sh7764_spi_delay(void)
{
	udelay(300);
}

static irqreturn_t sh7764_spi_eri(int irq, void *ptr)
{
	pr_info("%s\n", __func__);
	return IRQ_HANDLED;
}

static irqreturn_t sh7764_spi_rxi(int irq, void *ptr)
{
	pr_info("%s\n", __func__);
	return IRQ_HANDLED;
}

static irqreturn_t sh7764_spi_bri(int irq, void *ptr)
{
	pr_info("%s\n", __func__);
	return IRQ_HANDLED;
}

static irqreturn_t sh7764_spi_txi(int irq, void *ptr)
{
	pr_info("%s\n", __func__);
	return IRQ_HANDLED;
}

static void sh7764_spi_transfer(struct spi_transfer *t)
{
	const u8  *tx_buf = t->tx_buf;
	u8 *rx_buf= t->rx_buf;
	u8 tx, rx;
	unsigned int len = t->len;
#if 0
	int do_debug = 0;
#endif
	/* send | recv */
	while (len > 0) {
		/* wait TDFE=1 */
		while (!(inw(SCFSR1) & FSR_TDFE))
			;
		if (tx_buf)
			tx = bitrev8(*tx_buf++);
		else
			tx = 0;
#if 0
		if (!do_debug)
			do_debug = bitrev8(tx) == 0x46;
#endif
		outb(tx, SCFTDR1);

		/* clear TDFE */
		outw(inw(SCFSR1) & ~FSR_TDFE, SCFSR1);
#if 0
		/* wait TEND=1 */
		while (!(inw(SCFSR1) & FSR_TEND))
			;
#endif
		do {
			u16 lsr = inw(SCLSR1);
			
			if (lsr & LSR_ORER) {
				pr_err("scif LSR_ORER=1\n");
				outw(lsr & ~LSR_ORER, SCLSR1);
			}
		} while (!(inw(SCFSR1) & FSR_RDF));

		rx = inb(SCFRDR1);
		if (rx_buf)
			*rx_buf++ = bitrev8(rx);

		/* clear RDF */
		outw(inw(SCFSR1) & ~FSR_RDF, SCFSR1);
#if 0
		if (do_debug)
			pr_debug("%02x %02x\n", bitrev8(tx), bitrev8(rx));
#endif
		len--;
	}
//	pr_debug("  %04x %04x\n", inw(SCFSR1), inw(SCFDR1));

	t->len -= len;

//	outw(inw(SCFSR1) & ~FSR_TEND, SCFSR1);
}

static void sh7764_spi_work_one(struct spi_sh7764 *sh7764,
				struct spi_message *msg)
{
	struct spi_device *spi = msg->spi;
	struct spi_transfer *t;
	unsigned int cs_change = 1;

	list_for_each_entry(t, &msg->transfers, transfer_list) {
		u32 speed_hz = t->speed_hz ? : spi->max_speed_hz;

		if (cs_change)
			sh7764->info->chip_select(sh7764->info,
						  spi->chip_select, 1);
		sh7764_spi_delay();

		cs_change = t->cs_change;

		sh7764_spi_transfer(t);

		msg->actual_length += t->len;

		if (!cs_change)
			continue;
		if (t->transfer_list.next == &msg->transfers)
			break;

		sh7764->info->chip_select(sh7764->info, spi->chip_select, 0);
		sh7764_spi_delay();
	}

	msg->complete(msg->context);
}

static void sh7764_spi_work(struct work_struct *work)
{
	struct spi_sh7764 *sh7764 = container_of(work, struct spi_sh7764, work);
	unsigned long flags;

	spin_lock_irqsave(&sh7764->lock, flags);
	while (! list_empty(&sh7764->queue)) {
		struct spi_message *msg = container_of(sh7764->queue.next,
						       struct spi_message,
						       queue);

		list_del_init(&msg->queue);
		spin_unlock_irqrestore(&sh7764->lock, flags);

		sh7764_spi_work_one(sh7764, msg);

		spin_lock_irqsave(&sh7764->lock, flags);
	}
	spin_unlock_irqrestore(&sh7764->lock, flags);
}

static int sh7764_spi_master_setup(struct spi_device *spi)
{
	struct spi_master *master = spi->master;
	struct spi_sh7764 *sh7764 = spi_master_get_devdata(master);

//	pr_debug("%s:\n", __func__);

	sh7764->info->chip_select(sh7764->info, spi->chip_select, 0);
	sh7764_spi_delay();

	return 0;
}

static int sh7764_spi_master_transfer(struct spi_device *spi,
				      struct spi_message *msg)
{
	struct spi_master *master = spi->master;
	struct spi_sh7764 *sh7764 = spi_master_get_devdata(master);
	struct spi_transfer *t;
	unsigned long flags;

	msg->actual_length = 0;

	list_for_each_entry(t, &msg->transfers, transfer_list) {
		u32 speed_hz = t->speed_hz ? : spi->max_speed_hz;
		u8 bits_per_word = t->bits_per_word ? : spi->bits_per_word;

//		pr_debug(" t{sp=%d bpw=%d} spi{max_sp=%d bpw=%d}\n", t->speed_hz, t->bits_per_word, spi->max_speed_hz, spi->bits_per_word);

		if (! bits_per_word)
			bits_per_word = 8;

		if (!t->tx_buf && !t->rx_buf && t->len)
			return -EINVAL;

//		pr_debug("  speed_hz=%d bpw=%d\n", speed_hz, bits_per_word);
	}

	spin_lock_irqsave(&sh7764->lock, flags);
	list_add_tail(&msg->queue, &sh7764->queue);
	queue_work(sh7764->workqueue, &sh7764->work);
	spin_unlock_irqrestore(&sh7764->lock, flags);

	return 0;
}

static void sh7764_spi_master_cleanup(struct spi_device *spi)
{
	pr_info("%s:\n", __func__);
}

static int sh7764_spi_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct spi_master *master;
	struct spi_sh7764 *sh7764;
	struct sh_spi_info *pdata = dev->platform_data;
	int irq;
	int err;

	master = spi_alloc_master(&pdev->dev, sizeof *sh7764);

	master->bus_num = pdev->id;
	master->num_chipselect = pdata->num_chipselect;
	master->setup = sh7764_spi_master_setup;
	master->transfer = sh7764_spi_master_transfer;
	master->cleanup = sh7764_spi_master_cleanup;

	platform_set_drvdata(pdev, master);

	sh7764 = spi_master_get_devdata(master);
	sh7764->info = pdata;
	INIT_WORK(&sh7764->work, sh7764_spi_work);
	spin_lock_init(&sh7764->lock);
	INIT_LIST_HEAD(&sh7764->queue);
	init_waitqueue_head(&sh7764->waitq);

	outw(0x0006, SCFCR1);

	inw(SCFSR1);
	outw(0x0000, SCFSR1);
	inw(SCLSR1);
	outw(0x0000, SCLSR1);
	outw(0x0080, SCSMR1);
	outw(0x0000, SCSCR1);
	outb(0x06, SCBRR1);
	outw(0x0030, SCFCR1);
	outw(inw(SCSCR1) | 0x00F8, SCSCR1);

	if ((irq = platform_get_irq(pdev, 0)) < 0) {
		err = irq;
		dev_err(dev, "get_irq failed\n");
		goto get_irq_fail;
	}

	sh7764->irq = irq;

#if 0
	err = request_irq(irq,   sh7764_spi_eri, 0, "sh7764 spi ERI", sh7764);
	if (err)
		goto request_irq_fail0;
	err = request_irq(irq+1, sh7764_spi_rxi, 0, "sh7764 spi RXI", sh7764);
	if (err)
		goto request_irq_fail1;
	err = request_irq(irq+2, sh7764_spi_bri, 0, "sh7764 spi BRI", sh7764);
	if (err)
		goto request_irq_fail2;
	err = request_irq(irq+3, sh7764_spi_txi, 0, "sh7764 spi TXI", sh7764);
	if (err)
		goto request_irq_fail3;
#endif
	if (!(sh7764->workqueue =
	      create_singlethread_workqueue(dev->bus_id))) {
		goto workqueue_fail;
	}
	if ( (err = spi_register_master(master)) ) {
		dev_err(dev, "registering spi master failed\n");
		goto spi_register_master_fail;
	}

	return 0;

spi_register_master_fail:
	destroy_workqueue(sh7764->workqueue);
workqueue_fail:
#if 0
	free_irq(irq+3, sh7764);
request_irq_fail3:
	free_irq(irq+2, sh7764);
request_irq_fail2:
	free_irq(irq+1, sh7764);
request_irq_fail1:
	free_irq(irq, sh7764);
request_irq_fail0:
#endif
get_irq_fail:
	spi_master_put(master);
	dev_err(dev, "spi_sh7764 add failed");
	return err;
}

static int sh7764_spi_remove(struct platform_device *pdev)
{
	struct spi_master *master = platform_get_drvdata(pdev);
	struct spi_sh7764 *sh7764 = spi_master_get_devdata(master);
	int irq = sh7764->irq;

	pr_debug("%s:\n", __func__);

	destroy_workqueue(sh7764->workqueue);

	spi_unregister_master(master);
#if 0
	free_irq(irq, sh7764);
	free_irq(irq+1, sh7764);
	free_irq(irq+2, sh7764);
	free_irq(irq+3, sh7764);
#endif
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static struct platform_driver sh7764_spi_driver = {
	.probe	= sh7764_spi_probe,
	.remove	= __devexit_p(sh7764_spi_remove),
	.driver = {
		.name	= "sh7764_spi",
		.owner	= THIS_MODULE,
	},
};

static __init int spi_sh7764_init(void)
{
	return platform_driver_register(&sh7764_spi_driver);
}

static __exit void spi_sh7764_exit(void)
{
	platform_driver_unregister(&sh7764_spi_driver);
}

module_init(spi_sh7764_init);
module_exit(spi_sh7764_exit);

MODULE_LICENSE("GPL");
//MODULE_ALIAS("platform:sh7764_spi");
