#include "sh7764fb.h"

#include <asm/uaccess.h>
#include <linux/kthread.h>

static bool first_displist = 1;

static struct task_struct *g2d_task;
static int vsync_flag;
static int dl_num = 256;
static int dl_size = 32;
static int todo_idx = 0;
static int done_idx = 0;
static int full_flag = 0;
static int dma_done;
static u32 *dl_buf;
static int *dl_sizes;
static struct semaphore dl_sem;
static wait_queue_head_t todo_wait;
static wait_queue_head_t done_wait;
static wait_queue_head_t dma_wait;
static wait_queue_head_t idle_wait;

static irqreturn_t g2d_intr(int irq, void *data)
{
	u32 sr = ctrl_inl(SR);

//	printk("%s: %08x\n", __func__, sr);

	if (sr & 0x00000004) {
		printk(KERN_ERR "Command error SR=%08x\n", sr);
//		sh7764fb_hw_restart();
		ctrl_outl(0x00000004, SRCR);
	} else if (sr & 0x00000001) {
//		printk("Trap end\n");
		ctrl_outl(0x00000001, SRCR);

		dma_done = 1;
		wake_up(&dma_wait);
	} else {
		printk("%s: ?\n", __func__);
	}

	return IRQ_HANDLED;
}

static inline int displist_is_empty(void)
{
	return (!full_flag) && (todo_idx == done_idx);
}

static int sh7764fb_wait_idle(void)
{
	down(&dl_sem);
	while (!displist_is_empty()) {
		int ret;

 		up(&dl_sem);
		
		ret = wait_event_interruptible_timeout(idle_wait, 
						       displist_is_empty(),
						       3*HZ);
		if (ret == -ERESTARTSYS) {
			printk("%s: idle_wait interrupted\n", __func__);
			return ret;
		}
		if (ret == 0) {
			printk("%s: idle_wait timeout\n", __func__);
			return -ETIMEDOUT;
		}
		
		down(&dl_sem);
	}
	up(&dl_sem);

	return 0;
}

static inline void start_command(void)
{
#if 0
	printk("%s: >> todo=%d done=%d full=%d %08lx\n",
	       __func__, todo_idx, done_idx, full_flag, 
	       dl_buf[dl_size * done_idx]);
#endif
#if 0
	dma_cache_sync(0, &dl_buf[dl_size * done_idx], 
		       dl_sizes[done_idx] * sizeof(u32), DMA_TO_DEVICE);
#endif
	wmb();

	ctrl_outl((unsigned long)&dl_buf[dl_size * done_idx], DLSAR);
	ctrl_outl(0x00000005, SRCR);
	ctrl_outl(0x00000005, IER);
	ctrl_outl(0x00000001, SCLR);
//	printk("%s: <<\n", __func__);
}

static int g2d_thread(void *num)
{
	int ret;

	printk("%s: >>\n", __func__);

	for ( ; ; ) {
//		printk("%s: go...\n", __func__);
		
		ret = wait_event_interruptible(todo_wait,
					       !displist_is_empty() ||
					       kthread_should_stop());

		if (ret)
			printk("%s: ret=%d\n", __func__, ret);

		if (kthread_should_stop())
			break;

		dma_done = 0;
		start_command();

		ret = wait_event_timeout(dma_wait, dma_done, 2*HZ);
		if (ret == 0) {
			printk(KERN_ERR "irq lost...\n");
//			sh7764fb_hw_restart();
			continue;
		}

		down(&dl_sem);
		done_idx = (done_idx + 1) % dl_num;
		full_flag = 0;
		up(&dl_sem);
#if 0
		printk("%s: * todo=%d done=%d full=%d\n", 
		       __func__, todo_idx, done_idx, full_flag);
#endif
		if (todo_idx == done_idx)
			wake_up(&idle_wait);
		
		wake_up(&done_wait);
	}

	printk("%s: << todo=%d done=%d full=%d\n", 
	       __func__, todo_idx, done_idx, full_flag);

	return 0;
}

static int set_displist(struct fb_info *info, u32 *data, int size)
{
//	void *vp;
	int i = 0;
	u32 *dp;

	down(&dl_sem);
	while (full_flag) {
		up(&dl_sem);

		if (wait_event_interruptible(done_wait, !full_flag))
			return -ERESTARTSYS;

		down(&dl_sem);
	}
//	up(&dl_sem);

	dp = &dl_buf[dl_size * todo_idx];

	while (size--)
		dp[i++] = *data++;
//	dp[i++] = G2D_DL_VBKEM;
	dp[i++] = G2D_DL_TRAP;

//	dl_sizes[todo_idx] = i;

//	down(&dl_sem);
	todo_idx = (todo_idx + 1) % dl_num;
	full_flag = todo_idx == done_idx;
	up(&dl_sem);

	if (full_flag)
		printk("%s: buffer full\n", __func__);
#if 0
	printk("%s: * todo=%d done=%d full=%d\n", 
	       __func__, todo_idx, done_idx, full_flag);
#endif

#if 1
	dma_cache_sync(0, dp, i * sizeof(u32), DMA_TO_DEVICE);
	//wmb();
#endif
	wake_up(&todo_wait);

	return 0;
}

static int set_ini_displist(struct fb_info *info, u32 *data, int size)
{
	__u32 xres = info->var.xres;
	__u32 yres = info->var.yres;

	printk("%s: f=%d size=%d\n", __func__, first_displist, size);

	if (first_displist) {
		u32 dp[8];
		int i = 0;
		int ret;

		first_displist = 0;
		dp[i++] = G2D_DL_WPR;
		dp[i++] = 0x0D0;
		dp[i++] = (xres - 1) << 16 | (yres - 1);
		dp[i++] = G2D_DL_LCOFS;
		dp[i++] = 0;	/* x=0, y=0 */
		if ( (ret = set_displist(info, dp, i)) )
			return ret;
	}
	return set_displist(info, data, size);
}

static int sh7764fb_ioctl(struct fb_info *info,
			  unsigned int cmd, unsigned long arg)
{
	const void __user *argp = (void *)arg;

	switch (cmd) {
#if 1
	case FBIO_WAITFORVSYNC:
		printk("NI WAITFORVSYNC\n");
//		sh7764fb_wait_for_vsync();
		break;
#endif
	case G2DIOC_DL_RAW:
	{
		struct g2d_dl_raw g;

		if (copy_from_user(&g, argp, 
				   sizeof(g)/* + dl_size * sizeof(u32)*/)) {
			printk(KERN_WARNING "copy_from_user failed");
			return -EINVAL;
		}

		{
			u32 data[g.size];
			
			if (copy_from_user(&data[0], 
					   argp + sizeof(g), sizeof(data))) {
				printk(KERN_WARNING "copy_from_user failed");
				return -EINVAL;
			}
			set_ini_displist(info, data, g.size);
		}
		break;
	}
	case G2DIOC_WAIT_IDLE:
		sh7764fb_wait_idle();
		break;
	default:
		//printk("%s: cmd=%08x\n", __func__, cmd);
		printk(KERN_WARNING "%s: cmd=%08x  not handled.\n", 
		       __func__, cmd);
	}

	return 0;
}

static int sh7764g2d_set_par(struct fb_info *info)
{
	struct fb_fix_screeninfo *fix = &info->fix;

	iowrite32(fix->line_length, SSTRR);
	iowrite32(fix->line_length, DSTRR);
	fix->accel = FB_ACCEL_SH7764_G2D;
	fix->mmio_start = 0xffea0000;
	fix->mmio_len = 0x00050000;
	fix->xpanstep = 0;
	fix->ypanstep = 1;
	fix->ywrapstep = 0;

	return 0;
}

static int sh7764g2d_init(struct fb_info *info)
{
	int retval = 0;

	if (request_irq(G2D_IRQ, g2d_intr, 0, "G2D", info) < 0) {
		printk(KERN_ERR "unable to request IRQ %d\n", G2D_IRQ);
		retval = -EINVAL;
		goto err;
	}

	dl_buf = kmalloc(sizeof(u32) * dl_size * dl_num, GFP_KERNEL);

	if (!dl_buf) {
		printk(KERN_ERR "unable to kmalloc\n");
		retval = -ENOMEM;
		goto err;
	}

	dl_sizes = kmalloc(sizeof(int) * dl_num, GFP_KERNEL);

	if (!dl_sizes) {
		printk(KERN_ERR "unable to kmalloc\n");
		retval = -ENOMEM;
		goto err;
	}
	printk("%s: ln = %d\n", __func__, info->fix.line_length);

	init_MUTEX(&dl_sem);
	init_waitqueue_head(&todo_wait);
	init_waitqueue_head(&done_wait);
	init_waitqueue_head(&dma_wait);
	init_waitqueue_head(&idle_wait);

	g2d_task = kthread_create(g2d_thread, NULL, "G2Dd");
	if (IS_ERR(g2d_task)) {
		printk(KERN_ERR "Unable to kthread_create\n");
		retval = PTR_ERR(g2d_task);
		goto err;
	}

	wake_up_process(g2d_task);


	return 0;
err:
	return retval;
}

static void sh7764g2d_exit(struct fb_info *info)
{
	kthread_stop(g2d_task);

	free_irq(G2D_IRQ, info);
	iowrite32(0x00000000, SCLR);

	kfree(dl_sizes);
	kfree(dl_buf);
}
