#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/fb.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#undef CONFIG_FB_MEMSIZE
#define CONFIG_FB_MEMSIZE 783360
#include "sh7764fb.h"

#define DYN_MALLOC

#ifdef DYN_MALLOC
static char *vram;
#define VRAM ((u32)vram)
#else
static char vram[CONFIG_FB_MEMSIZE] __attribute__((aligned(16)));
#define VRAM (P1SEGADDR((u32)&vram[0]))
#endif

static int fb_memsize	= CONFIG_FB_MEMSIZE;

static int disp_width	= XRES;
static int disp_height	= YRES;

static int vsync_start	= 1;
static int vsync_end	= 0;
static int hsync_start	= 0; 	/* Must be ZERO */
static int hsync_end	= 0x0A;

static int syn_width	= SYN_WIDTH;
static int syn_height	= SYN_HEIGHT;
static int sgde_width	= SGDE_WIDTH;
static int sgde_height	= SGDE_HEIGHT;
static int sgde_start_h	= 16;
static int sgde_start_v	= 1;

static struct fb_info sh7764fb;

static bool first_displist = 1;

static struct task_struct *g2d_task;
static int vsync_flag;
static int dl_num = 256;
static int dl_size = 32;
static int todo_idx = 0;
static int done_idx = 0;
static int full_flag = 0;
static int dma_done;
static u32 *dl_buf;
static int *dl_sizes;
static struct semaphore dl_sem;
//static struct semaphore buffer_sem;
static wait_queue_head_t todo_wait;
static wait_queue_head_t done_wait;
static wait_queue_head_t dma_wait;
static wait_queue_head_t idle_wait;
static wait_queue_head_t vsync_wait;


static void reset_dl_idx(void)
{
	todo_idx = 0;
	done_idx = 0;
	full_flag = 0;
}

static inline void sgmode_we(bool enable)
{
	unsigned long v = ctrl_inl(SGMODE);

	if (enable)
		v |= 0x80000000;
	else
		v &= ~0x80000000;

	ctrl_outl(v, SGMODE);
}

static inline void grcmen_we(int n, bool enable)
{
	unsigned long v = ctrl_inl(GRCMEN(n));
    
	if (enable)
		v |= 0x80000000;
	else
		v &= ~0x80000000;
	
	ctrl_outl(v, GRCMEN(n));
}

static int sh7764fb_hw_restart(void)
{
	volatile unsigned long dummy;

	ctrl_outl(0x80000000, SCLR);
	dummy = ctrl_inl(0xA4000000);
	dummy = ctrl_inl(0xA4000000);
	dummy = ctrl_inl(0xA4000000);
	dummy = ctrl_inl(0xA4000000);
	reset_dl_idx();
	ctrl_outl(0x00000000, SCLR);
	ctrl_outl(0x00040000, RCRL); /* GBM=1 16bit */
	ctrl_outl(0x00050000, RCL2R);

	return 0;
}

static irqreturn_t vdc2_intr(int irq, void *data)
{
	ctrl_outl(0x00000010, SGINTCNT);

	vsync_flag = 1;
	wake_up_interruptible(&vsync_wait);

	return IRQ_HANDLED;
}

static irqreturn_t g2d_intr(int irq, void *data)
{
	u32 sr = ctrl_inl(SR);

//	printk("%s: %08x\n", __func__, sr);

	if (sr & 0x00000004) {
		printk(KERN_ERR "Command error SR=%08x\n", sr);
		sh7764fb_hw_restart();
		ctrl_outl(0x00000004, SRCR);
	} else if (sr & 0x00000001) {
//		printk("Trap end\n");
		ctrl_outl(0x00000001, SRCR);

		dma_done = 1;
		wake_up(&dma_wait);
	} else {
		printk("%s: ?\n", __func__);
	}

	return IRQ_HANDLED;
}

static void sh7764fb_wait_for_vsync(void)
{
	vsync_flag = 0;
	ctrl_outl(0x00000001, SGINTCNT);
	wait_event_interruptible_timeout(vsync_wait, vsync_flag, HZ);	
}

static inline int displist_is_empty(void)
{
	return (!full_flag) && (todo_idx == done_idx);
}

static int sh7764fb_check_var(struct fb_var_screeninfo *var,
			       struct fb_info *info)
{
	var->activate = FB_ACTIVATE_NOW;
#if 0
	printk("%s:\n", __func__);
	printk(" xres=%d\n", var->xres);
	printk(" yres=%d\n", var->yres);
	printk(" xres_virtual=%d\n", var->xres_virtual);
	printk(" yres_virtual=%d\n", var->yres_virtual);
#endif
#if 0
	printk(" xoffset=%d\n", var->xoffset);
	printk(" yoffset=%d\n", var->yoffset);
	printk(" bits_per_pixel=%d\n", var->bits_per_pixel);
	printk(" grayscale=%d\n", var->grayscale);
	printk(" nonstd=%08x\n", var->nonstd);
	printk(" activate=%08x\n", var->activate);
	printk(" height=%d\n", var->height);
	printk(" width=%d\n", var->width);
	printk(" accel_flags=%08x\n",  var->accel_flags);
	printk(" pixclock=%d\n", var->pixclock);
	printk(" left_margin=%d\n", var->left_margin);
	printk(" right_margin=%d\n", var->right_margin);
	printk(" upper_margin=%d\n", var->upper_margin);
	printk(" lower_margin=%d\n", var->lower_margin);
	printk(" hsync_len=%d\n", var->hsync_len);
	printk(" vsync_len=%d\n", var->vsync_len);
	printk(" sync=%d\n", var->sync);
	printk(" vmode=%d\n", var->vmode);
	printk(" rotate=%d\n", var->rotate);
#endif

	if (var->yres_virtual > info->fix.smem_len / info->fix.line_length) {
		printk(" EINVAL\n");
		return -EINVAL;
	}

	return 0;
}
#if 0
static int sh7764fb_set_par(struct fb_info *info)
{
	printk("%s:\n", __func__);

	return 0;
}
#endif
static int sh7764fb_wait_idle(void)
{
	down(&dl_sem);
	while (!displist_is_empty()) {
		int ret;

 		up(&dl_sem);
		
		ret = wait_event_interruptible_timeout(idle_wait, 
						       displist_is_empty(),
						       3*HZ);
		if (ret == -ERESTARTSYS) {
			printk("%s: idle_wait interrupted\n", __func__);
			return ret;
		}
		if (ret == 0) {
			printk("%s: idle_wait timeout\n", __func__);
			return -ETIMEDOUT;
		}
		
		down(&dl_sem);
	}
	up(&dl_sem);

	return 0;
}

static int prev_layer = 0;

static int sh7764fb_pan_display(struct fb_var_screeninfo *var,
				struct fb_info *info)
{
#define USE_LAYER 0

#if USE_LAYER
	int layer = var->yoffset / var->yres;

//	printk("%d %d\n", layer, prev_layer);
#endif

#if 0
	int ret;

	printk("+------- PAN DISPLAY -------+\n");
	printk("+ xoffset: %d\n", var->xoffset);
	printk("+ yoffset: %d\n", var->yoffset);
	printk("+   xresv: %d\n", var->xres_virtual);
	printk("+   yresv: %d\n", var->yres_virtual);
#endif


#if USE_LAYER
	outl(inl(GRCMEN(layer)) | 0x00000002, GRCMEN(layer));
	outl(inl(GRCMEN(prev_layer)) & ~0x00000002, GRCMEN(prev_layer));
#else
	outl(info->fix.smem_start + var->yoffset * info->fix.line_length,
	     GROPSADR(0));
#endif
	vsync_flag = 0;
#if 0
	sh7764fb_wait_idle();
#endif
#if USE_LAYER
	grcmen_we(layer, 1);
	grcmen_we(prev_layer, 1);
	prev_layer = layer;
#else
	grcmen_we(0, 1);
#endif
#if 0
	outl(0x00000001, SGINTCNT);
	wait_event_interruptible_timeout(vsync_wait, vsync_flag, HZ);	
#endif
#if 0
	printk("%s: xres=%d yres=%d xoff=%d yoff=%d xresv=%d yresv=%d mem=%08lx line_length=%d\n", 
	       __func__, var->xres, var->yres, var->xoffset, var->yoffset,
	       var->xres_virtual, var->yres_virtual,
	       info->fix.smem_start, info->fix.line_length);
#endif
	return 0;	/* success */
}

#if 0
static void sh7764fb_fillrect(struct fb_info *p,
			       const struct fb_fillrect *rect)
{
	printk("%s: dx=%d dy=%d width=%d height=%d color=%08x rop=%d\n", 
	       __func__, rect->dx, rect->dy, rect->width, rect->height,
	       rect->color, rect->rop);
}

static void sh7764fb_copyarea(struct fb_info *info,
			       const struct fb_copyarea *region)
{
	printk("%s:\n", __func__);
}

#endif
static int sh7764fb_sync(struct fb_info *info)
{
	printk("%s:\n", __func__);

	return 0;
}

static inline void start_command(void)
{
#if 0
	printk("%s: >> todo=%d done=%d full=%d %08lx\n",
	       __func__, todo_idx, done_idx, full_flag, 
	       dl_buf[dl_size * done_idx]);
#endif
#if 0
	dma_cache_sync(0, &dl_buf[dl_size * done_idx], 
		       dl_sizes[done_idx] * sizeof(u32), DMA_TO_DEVICE);
#endif
	wmb();

	ctrl_outl((unsigned long)&dl_buf[dl_size * done_idx], DLSAR);
	ctrl_outl(0x00000005, SRCR);
	ctrl_outl(0x00000005, IER);
	ctrl_outl(0x00000001, SCLR);
//	printk("%s: <<\n", __func__);
}

static int g2d_thread(void *num)
{
	int ret;

	printk("%s: >>\n", __func__);

	for ( ; ; ) {
//		printk("%s: go...\n", __func__);
		
		ret = wait_event_interruptible(todo_wait,
					       !displist_is_empty() ||
					       kthread_should_stop());

		if (ret)
			printk("%s: ret=%d\n", __func__, ret);

		if (kthread_should_stop())
			break;

		dma_done = 0;
		start_command();

		ret = wait_event_timeout(dma_wait, dma_done, 2*HZ);
		if (ret == 0) {
			printk(KERN_ERR "irq lost...\n");
			sh7764fb_hw_restart();
			continue;
		}

		down(&dl_sem);
		done_idx = (done_idx + 1) % dl_num;
		full_flag = 0;
		up(&dl_sem);
#if 0
		printk("%s: * todo=%d done=%d full=%d\n", 
		       __func__, todo_idx, done_idx, full_flag);
#endif
		if (todo_idx == done_idx)
			wake_up(&idle_wait);
		
		wake_up(&done_wait);
	}

	printk("%s: << todo=%d done=%d full=%d\n", 
	       __func__, todo_idx, done_idx, full_flag);

	return 0;
}

static int set_displist(struct fb_info *info, u32 *data, int size)
{
//	void *vp;
	int i = 0;
	u32 *dp;

	down(&dl_sem);
	while (full_flag) {
		up(&dl_sem);

		if (wait_event_interruptible(done_wait, !full_flag))
			return -ERESTARTSYS;

		down(&dl_sem);
	}
//	up(&dl_sem);

	dp = &dl_buf[dl_size * todo_idx];

	while (size--)
		dp[i++] = *data++;
//	dp[i++] = G2D_DL_VBKEM;
	dp[i++] = G2D_DL_TRAP;

//	dl_sizes[todo_idx] = i;

//	down(&dl_sem);
	todo_idx = (todo_idx + 1) % dl_num;
	full_flag = todo_idx == done_idx;
	up(&dl_sem);

	if (full_flag)
		printk("%s: buffer full\n", __func__);
#if 0
	printk("%s: * todo=%d done=%d full=%d\n", 
	       __func__, todo_idx, done_idx, full_flag);
#endif

#if 1
	dma_cache_sync(0, dp, i * sizeof(u32), DMA_TO_DEVICE);
	//wmb();
#endif
	wake_up(&todo_wait);

	return 0;
}

static int set_ini_displist(struct fb_info *info, u32 *data, int size)
{
	if (first_displist) {
		u32 dp[8];
		int i = 0;
		int ret;

		first_displist = 0;
		dp[i++] = G2D_DL_WPR;
		dp[i++] = 0x0D0;
		dp[i++] = (disp_width - 1) << 16 | (disp_height - 1);
		dp[i++] = G2D_DL_LCOFS;
		dp[i++] = 0;	/* x=0, y=0 */
		if ( (ret = set_displist(info, dp, i)) )
			return ret;
	}
	return set_displist(info, data, size);
}

static int sh7764fb_ioctl(struct fb_info *info,
			  unsigned int cmd, unsigned long arg)
{
	const void __user *argp = (void *)arg;

	switch (cmd) {
	case FBIO_WAITFORVSYNC:
	  //		printk("WAITFORVSYNC\n");
		sh7764fb_wait_for_vsync();
		break;
	case G2DIOC_DL_RAW:
	{
		struct g2d_dl_raw g;

		if (copy_from_user(&g, argp, 
				   sizeof(g)/* + dl_size * sizeof(u32)*/)) {
			printk(KERN_WARNING "copy_from_user failed");
			return -EINVAL;
		}

		{
			u32 data[g.size];
			
			if (copy_from_user(&data[0], 
					   argp + sizeof(g), sizeof(data))) {
				printk(KERN_WARNING "copy_from_user failed");
				return -EINVAL;
			}
			set_ini_displist(info, data, g.size);
		}
		break;
	}
	case G2DIOC_WAIT_IDLE:
		sh7764fb_wait_idle();
		break;
	default:
		//printk("%s: cmd=%08x\n", __func__, cmd);
		printk(KERN_WARNING "%s: cmd=%08x  not handled.\n", 
		       __func__, cmd);
	}

	return 0;
}

static struct fb_ops sh7764fb_ops = {
	.owner		= THIS_MODULE,
	.fb_check_var	= sh7764fb_check_var,
//	.fb_set_par	= sh7764fb_set_par,
	.fb_pan_display	= sh7764fb_pan_display,
	//	.fb_fillrect	= cfb_fillrect,
	//	.fb_copyarea	= cfb_copyarea,
	//	.fb_imageblit	= cfb_imageblit,
	.fb_sync	= sh7764fb_sync,
	.fb_ioctl	= sh7764fb_ioctl
};
#if 0
static struct device_driver sh7764fb_driver = {
	.name		= "sh7764fb",
	.bus		= &platform_bus_type,
	.probe		= sh7764fb_probe,
	.remove		= sh7764fb_remove,
	.suspend	= NULL,
	.resume		= NULL
};
#endif

static int sh7764fb_probe(struct platform_device *pdev)
{
	int retval = 0;
	int i;
	struct fb_fix_screeninfo *fix = &sh7764fb.fix;
	struct fb_var_screeninfo *var = &sh7764fb.var;
	int bpp = 16;
	struct fb_bitfield red = {11, 5, 0};
	struct fb_bitfield green = {5, 6, 0};
	struct fb_bitfield blue = {0, 5, 0};
	struct fb_bitfield transp = {0, 0, 0};

#ifdef DYN_MALLOC
	vram = kmalloc(fb_memsize, GFP_KERNEL);
	if (!vram)
	  return -ENOMEM;
#endif
	printk("VRAM=%08x-%08x\n", VRAM, VRAM+fb_memsize-1);
	sh7764fb.fbops = &sh7764fb_ops;
	strcpy(fix->id, "VDC2");
	fix->smem_start = VRAM;
	fix->smem_len = fb_memsize;
	fix->type = FB_TYPE_PACKED_PIXELS;
	fix->type_aux = 0;
	fix->visual = FB_VISUAL_TRUECOLOR;
	fix->xpanstep = 0;
	fix->ypanstep = 1;
	fix->ywrapstep = 0;
	fix->line_length = XRES * (bpp / 8);
	fix->mmio_start = 0xffea0000;
	fix->mmio_len = 0x00050000;
	fix->accel = FB_ACCEL_SH7764_G2D;

	var->xres = XRES;
	var->yres = YRES;
	var->xres_virtual = var->xres;
	var->yres_virtual = var->yres * 3;
	var->xoffset = 0;
	var->yoffset = 0;
	var->bits_per_pixel = bpp;
	var->grayscale = 0;
	var->red = red;
	var->green = green;
	var->blue = blue;
	var->transp = transp;

	//	var->activate = FB_ACTIVATE_NOW;

	if (request_irq(VDC2_IRQ, vdc2_intr, 0, "VDC2", vdc2_intr) < 0) {
		printk(KERN_ERR "unable to request IRQ %d\n", VDC2_IRQ);
		retval = -EINVAL;
		goto err0;
	}

	if (request_irq(G2D_IRQ, g2d_intr, 0, "G2D", g2d_intr) < 0) {
		printk(KERN_ERR "unable to request IRQ %d\n", G2D_IRQ);
		retval = -EINVAL;
		goto err1;
	}

	ctrl_outl(ctrl_inl(VDC2CLKCR) & ~0x00000080, VDC2CLKCR);

	sh7764fb_hw_restart();

	ctrl_outl(VRAM & 0x1FFFFFF0, RSAR);
	ctrl_outl(VRAM & 0x1FFFFFF0, SSAR);

	ctrl_outw(0x5555, PTSEL_G);

	/* PH3:out */
	ctrl_outw(0x0040, PTIO_H);
	/* PH3:PH3, PH2:DCLKOUT, PH1:DR4, DR0:DR5 */
	ctrl_outw((ctrl_inw(PTSEL_H) & ~0x00FF) | 0x0095, PTSEL_H);
	ctrl_outw((ctrl_inw(PTDAT_H) & ~0x0008) | 0x0008, PTDAT_H);

	/* PI7:DB3, PI6:DB2, PI5:DB1, PI4:DG1, 
	   PI3:DG0, PI2:DB5, PI1:DB4, PI0:PI0 */
	ctrl_outw(0x5554, PTSEL_I);

	/* PI{7..1}:other, PIO0:out */
	ctrl_outw(0x0001, PTIO_I); 

	ctrl_outw(ctrl_inw(PTDAT_I) | 0x0001, PTDAT_I);

	/* PK4:DB0, 
	   PK3:HSYNC/SPL, PK2:DCLKIN, PK1:VSYNC/SPS, PK0:DE_C/CD_E */
	ctrl_outw((ctrl_inw(PTSEL_K) & ~0x03FF) | 0x0155, PTSEL_K);

	sgmode_we(0);
	outl((sgde_start_v << 16 & 0x03FF0000) | (sgde_start_h & 0x07FF),
		  SGDESTART);
	outl((sgde_height << 16 & 0x03FF0000) | (sgde_width & 0x07FF),
		  SGDESIZE);
	ctrl_outl(0x00000010, SGINTCNT);
	ctrl_outl(0x00000020, SGMODE);
	ctrl_outl(0x00000010, SGINTCNT);
	ctrl_outl(0x00000000, SYNCNT);
	ctrl_outl(0x00000000, EXTSYNCNT);
	ctrl_outl(0x00000000, SYNCNT);
	ctrl_outl(0x00000000, EXTSYNCNT);
	ctrl_outl((syn_height << 16 & 0x03FF0000) | (syn_width & 0x07FF),
		  SYNSIZE);
	ctrl_outl(vsync_start << 16 | (vsync_end & 0x03FF), VSYNCTIM);
	ctrl_outl(hsync_start << 16 | (hsync_end & 0x07FF), HSYNCTIM);
	ctrl_outl(0x00000001, CLSTIM);
	ctrl_outl(0x00000001, SPLTIM);
	ctrl_outl(0x00000000, COMTIM);
	ctrl_outl(0x00000000, T1004CNT);
	ctrl_outl(0x00000000, T1004OFFSET);
	sgmode_we(1);
    
	for (i = 0; i < 4; i++) {
		grcmen_we(i, 0);
		outl((fix->line_length & 0x1FFFFFF0), GROPSOFST(i));
		outl((var->yres & 0xFFFE) << 16 | (var->xres & 0xFFFE),
		     GROPSWH(i));
		outl(0x00000000, GROPDPHV(i));
		outl((fix->smem_start & 0x1FFFFFFE) +
		     (var->yres * fix->line_length * i),
		     GROPSADR(i));
		outl(0x0000AAAA, GROPBASERGB(i));
		if (i == 0)
			ctrl_outl(0x00000003, GRCMEN(i)); /* GEN enable */
		else
			ctrl_outl(0x00000001, GRCMEN(i));
		grcmen_we(i, 1);
	}
	
	if (register_framebuffer(&sh7764fb) < 0) {
		printk(KERN_ERR "unable to register framebuffer\n");
		retval = -EINVAL;
		goto err2;
	}

	dl_buf = kmalloc(sizeof(u32) * dl_size * dl_num, GFP_KERNEL);
	
	if (! dl_buf) {
		printk(KERN_ERR "unable to kmalloc\n");
		retval = -ENOMEM;
		goto err3;
	}

	dl_sizes = kmalloc(sizeof(int) * dl_num, GFP_KERNEL);
	
	if (! dl_sizes) {
		printk(KERN_ERR "unable to kmalloc\n");
		retval = -ENOMEM;
		goto err4;
	}

	memset((void *)VRAM, 0, fb_memsize);
//	printk("VRAM=%08lx\n", VRAM);

	ctrl_outl((fix->line_length / 2) & 0x1FFF, SSTRR);
	ctrl_outl((fix->line_length / 2) & 0x1FFF, DSTRR);

	init_MUTEX(&dl_sem);
//	init_MUTEX(&buffer_sem);
	init_waitqueue_head(&todo_wait);
	init_waitqueue_head(&done_wait);
	init_waitqueue_head(&dma_wait);
	init_waitqueue_head(&idle_wait);
	init_waitqueue_head(&vsync_wait);

	g2d_task = kthread_create(g2d_thread, NULL, "G2Dd");
	if (IS_ERR(g2d_task)) {
		printk("unable to kthread_create\n");
		retval = PTR_ERR(g2d_task);
		goto err5;
	}

	wake_up_process(g2d_task);

	printk("sh7764fb loaded\n");
	printk(" SYN_WIDTH:   %4d\n", syn_width);
	printk(" SYN_HEIGHT:  %4d\n", syn_height);	
	printk(" SGDE_START_H:%4d\n", sgde_start_h);
	printk(" SGDE_START_V:%4d\n", sgde_start_v);
	printk(" HSYNC_START: %4d\n", hsync_start);
	printk(" HSYNC_END:   %4d\n", hsync_end);
	printk(" VSYNC_START: %4d\n", vsync_start);
	printk(" VSYNC_END:   %4d\n", vsync_end);

	return 0;
err5:
	kfree(dl_sizes);
err4:
	kfree(dl_buf);
err3:
	unregister_framebuffer(&sh7764fb);	
err2:
	free_irq(G2D_IRQ, g2d_intr);	
err1:
	free_irq(VDC2_IRQ, vdc2_intr);
err0:
	return retval;
}

#ifdef MODULE
module_param(fb_memsize, int, S_IRUGO);
module_param(dl_num, int, S_IRUGO);
module_param(dl_size, int, S_IRUGO);
module_param(vsync_start, int, S_IRUGO | S_IWUSR);
module_param(vsync_end, int, S_IRUGO | S_IWUSR);
module_param(hsync_start, int, S_IRUGO | S_IWUSR);
module_param(hsync_end, int, S_IRUGO | S_IWUSR);
module_param(syn_width, int, S_IRUGO | S_IWUSR);
module_param(syn_height, int, S_IRUGO | S_IWUSR);
module_param(sgde_start_h, int, S_IRUGO | S_IWUSR);
module_param(sgde_start_v, int, S_IRUGO | S_IWUSR);
//MODULE_PARM_DESC(vram, "System RAM to allocate to framebuffer in MiB" " (default=4)");
#endif

static int sh7764fb_remove(struct platform_device *pdev)
{
	int ret;

	ret = kthread_stop(g2d_task);
//	wake_up(&todo_wait);
	printk("kthread stop %d\n", ret);
	unregister_framebuffer(&sh7764fb);
	free_irq(G2D_IRQ, g2d_intr);
	free_irq(VDC2_IRQ, vdc2_intr);
	ctrl_outl(0x00000000, SCLR);

	kfree(dl_sizes);
	kfree(dl_buf);
#if 1
	kfree(vram);
#endif
	printk("sh7764fb unloaded\n");

	return 0;
}

static struct platform_driver sh7764_vdc2_driver = {
	.driver = {
		   .name = "sh7764-vdc2",
		   .owner = THIS_MODULE,
	},
	.probe = sh7764fb_probe,
	.remove = __devexit_p(sh7764fb_remove),
};

static int __init sh7764fb_init(void)
{
	return platform_driver_register(&sh7764_vdc2_driver);
}

static void __exit sh7764fb_exit(void)
{
	platform_driver_unregister(&sh7764_vdc2_driver);
}

module_init(sh7764fb_init);
module_exit(sh7764fb_exit);

MODULE_AUTHOR("ALPHAPROJECT");
MODULE_DESCRIPTION("FBdev for SH7764 VDC2 Controller");
MODULE_LICENSE("GPL");

