/*
 * ms104sh4ag.c  --  SoC audio for MS104SH4AG 5912
 *
 * Copyright (C) 2009 ALPHAPROJECT
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/clk.h>
#include <linux/platform_device.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <linux/i2c.h>

#include "../codecs/tlv320aic23.h"

extern struct snd_soc_dai sh7764_ssi_dai[2];
extern struct snd_soc_platform sh7764_soc_platform;

#define CODEC_CLOCK 12288000

static struct i2c_client *ad5246_i2c;

static int ad5246_value = 0;

static void ad5246_read_regs(void)
{
	int ret;
	char value;

	if ((ret = i2c_master_recv(ad5246_i2c, &value, 1)) != 1)
		printk(KERN_ERR "ad5246: i2c read failed\n");
	else
		ad5246_value = value;
}

static void ad5246_write_regs(void)
{
	int ret;
	const char value = ad5246_value;

	if ((ret = i2c_master_send(ad5246_i2c, &value, 1)) != 1) {
		printk(KERN_ERR "ad5246: i2c write failed\n");
	}
}

static int ad5246_i2c_probe(struct i2c_client *client)
{
	dev_dbg(&client->dev, "%s: client{name=%s flags=%04x addr=%04x}\n",
		__func__, client->name, client->flags, client->addr);

	ad5246_i2c = client;

	ad5246_read_regs();

	return 0;
}

static int ad5246_i2c_remove(struct i2c_client *client)
{
	dev_dbg(&client->dev, "%s: client{name=%s flags=%04x addr=%04x}\n",
		__func__, client->name, client->flags, client->addr);

	return 0;
}

static struct i2c_device_id ad5246_idtable[] = {
	{"ad5246", 0},
	{}
};

MODULE_DEVICE_TABLE(i2c, ad5246_idtable);

static struct i2c_driver ad5246_i2c_driver = {
	.driver = {
		.name	= "ad5246",
		.owner	= THIS_MODULE,
	},
	.id_table	= ad5246_idtable,
	.probe		= ad5246_i2c_probe,
	.remove		= ad5246_i2c_remove,
};

static int ad5246_get_reg(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = 127 - ad5246_value;

	return 0;
}

static int ad5246_set_reg(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	if (127 - ad5246_value == ucontrol->value.integer.value[0])
		return 0;

	ad5246_value = 127 - ucontrol->value.integer.value[0];
	ad5246_write_regs();
	return 1;
}

static const struct snd_kcontrol_new ms104sh4ag_controls[] = {
#if 1
	SOC_SINGLE_EXT("Master Playback Volume", 0, 0, 127, 1,
		       ad5246_get_reg, ad5246_set_reg),
#else
	{
		.iface	= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name	= "Master Playback Volume",
		.index	= 0,
		.info	= ms104sh4ag_info_volume,
		.get	= ms104sh4ag_get_volume,
		.put	= ms104sh4ag_put_volume,
	}
#endif
};

static int ms104sh4ag_startup(struct snd_pcm_substream *substream)
{
	return 0;
}

static void ms104sh4ag_shutdown(struct snd_pcm_substream *substream)
{
}

static int ms104sh4ag_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->dai->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->dai->cpu_dai;
	int err;

	/* Set codec DAI configuration */
	err = snd_soc_dai_set_fmt(codec_dai,
				  SND_SOC_DAIFMT_LEFT_J |
				  SND_SOC_DAIFMT_NB_IF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (err < 0) {
		printk(KERN_ERR "can't set codec DAI configuration\n");
		return err;
	}

	/* Set cpu DAI configuration */
	err = snd_soc_dai_set_fmt(cpu_dai,
				  SND_SOC_DAIFMT_LEFT_J	|
				  SND_SOC_DAIFMT_IB_IF	|
				  SND_SOC_DAIFMT_CBM_CFM);
	if (err < 0) {
		printk(KERN_ERR "can't set cpu DAI configuration\n");
		return err;
	}

	/* Set the codec system clock for DAC and ADC */
	err = snd_soc_dai_set_sysclk(codec_dai, 0,
				     CODEC_CLOCK, SND_SOC_CLOCK_IN);

	if (err < 0) {
		printk(KERN_ERR "can't set codec system clock\n");
		return err;
	}

	return err;
}

static struct snd_soc_ops ms104sh4ag_ops = {
	.startup = ms104sh4ag_startup,
	.hw_params = ms104sh4ag_hw_params,
	.shutdown = ms104sh4ag_shutdown,
};

static const struct snd_soc_dapm_widget tlv320aic23_dapm_widgets[] = {
	SND_SOC_DAPM_HP("Headphone Jack", NULL),
	SND_SOC_DAPM_LINE("Line In", NULL),
	SND_SOC_DAPM_MIC("Mic Jack", NULL),
};

static const struct snd_soc_dapm_route audio_map[] = {
	{"Headphone Jack", NULL, "LHPOUT"},
	{"Headphone Jack", NULL, "RHPOUT"},

	{"LLINEIN", NULL, "Line In"},
	{"RLINEIN", NULL, "Line In"},

	{"MICIN", NULL, "Mic Jack"},
};

static int ms104sh4ag_tlv320aic23_init(struct snd_soc_codec *codec)
{
	int err, i;

	/* Add ms104sh4ag specific widgets */
	snd_soc_dapm_new_controls(codec, tlv320aic23_dapm_widgets,
				  ARRAY_SIZE(tlv320aic23_dapm_widgets));

	/* Set up ms104sh4ag specific audio path audio_map */
	snd_soc_dapm_add_routes(codec, audio_map, ARRAY_SIZE(audio_map));

	snd_soc_dapm_enable_pin(codec, "Headphone Jack");
	snd_soc_dapm_enable_pin(codec, "Line In");
	snd_soc_dapm_enable_pin(codec, "Mic Jack");

	snd_soc_dapm_sync(codec);

	for (i = 0; i < ARRAY_SIZE(ms104sh4ag_controls); i++) {
		if ((err = snd_ctl_add(codec->card, 
				       snd_soc_cnew(&ms104sh4ag_controls[i],
						    codec, NULL))) < 0) {
			printk(KERN_ERR "snd_ctl_add error %d\n", err);
			return err;
		}
	}	
	return 0;
}

static struct snd_soc_dai_link ms104sh4ag_dais[] = {
	{
		.name = "TLV320AIC23-cap",
		.stream_name = "AIC23",
		.cpu_dai = &sh7764_ssi_dai[0],
		.codec_dai = &tlv320aic23_dai,
		.init = ms104sh4ag_tlv320aic23_init,
		.ops = &ms104sh4ag_ops,
	}, {
		.name = "TLV320AIC23-play",
		.stream_name = "AIC23",
		.cpu_dai = &sh7764_ssi_dai[1],
		.codec_dai = &tlv320aic23_dai,
		/* .init = ms104sh4ag_tlv320aic23_init, */
		.ops = &ms104sh4ag_ops,
	}
};

/* Audio machine driver */
static struct snd_soc_card snd_soc_card_ms104sh4ag = {
	.name = "MS104-SH4AG",
	.platform = &sh7764_soc_platform,
	.dai_link = ms104sh4ag_dais,
	.num_links = 2,
};

/* Audio subsystem */
static struct snd_soc_device ms104sh4ag_snd_devdata = {
	.card = &snd_soc_card_ms104sh4ag,
	.codec_dev = &soc_codec_dev_tlv320aic23,
};

static struct platform_device *ms104sh4ag_snd_device;

static int __init ms104sh4ag_soc_init(void)
{
	int err;

	ms104sh4ag_snd_device = platform_device_alloc("soc-audio", -1);
	if (!ms104sh4ag_snd_device)
		return -ENOMEM;

	platform_set_drvdata(ms104sh4ag_snd_device, &ms104sh4ag_snd_devdata);
	ms104sh4ag_snd_devdata.dev = &ms104sh4ag_snd_device->dev;

	if ((err = platform_device_add(ms104sh4ag_snd_device)))
		goto err1;

	if ((err = i2c_add_driver(&ad5246_i2c_driver)))
		goto err1;

	return 0;
err1:
	platform_device_del(ms104sh4ag_snd_device);
	platform_device_put(ms104sh4ag_snd_device);

	return err;

}

static void __exit ms104sh4ag_soc_exit(void)
{
	i2c_del_driver(&ad5246_i2c_driver);
	platform_device_unregister(ms104sh4ag_snd_device);
}

module_init(ms104sh4ag_soc_init);
module_exit(ms104sh4ag_soc_exit);

MODULE_AUTHOR("ALPHAPROJECT");
MODULE_DESCRIPTION("ALSA SoC MS104-SH4AG");
MODULE_LICENSE("GPL");
