#define DEBUG
/*
 * SH7764 ("sh7764") DMABRG audio DMA unit support
 *
 * Copyright (C) 2007 Manuel Lauss <mano@roarinelk.homelinux.net>
 *  licensed under the terms outlined in the file COPYING at the root
 *  of the linux kernel sources.
 *
 * The SH7764 DMABRG provides 4 dma channels (2x rec, 2x play), which
 * trigger an interrupt when one half of the programmed transfer size
 * has been xmitted.
 *
 * FIXME: little-endian only for now
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
//#include <asm/dmabrg.h>

#define SSIDMA0_IRQ	64
#define SSICH0_IRQ	65
#define SSICH1_IRQ 	66
#define SSICH2_IRQ	67
#define SSIDMA1_IRQ	69
#define SSICH3_IRQ	70
#define SSICH4_IRQ	71
#define SSICH5_IRQ	72

#define SSIDMMR		0x00
#define SSIRDMADR	0x08
#define SSIRDMCNTR	0x10
#define SSIWDMADR	0x18
#define SSIWDMCNTR	0x20
#define SSIDMCOR	0x28
#define SSISTPBLCNT	0x30
#define SSISTPDR	0x38
#define SSIBLCNTSR	0x40
#define SSIBLCNT	0x48
#define SSIBLNCNTSR	0x50
#define SSIBLNCNT	0x58

#define SSIDMAOR	0x180
#define SSIDMINTSR	0x188
#define SSIDMINTMR	0x190

#define SSIDMMR_RDMBSZ(v)	(((v) & 3) << 6)
#define SSIDMMR_WDMBSZ(v)	(((v) & 3) << 4)

#define SSIDMCOR_DMRST	0x80000000
#define SSIDMCOR_TXRST	0x40000000
#define SSIDMCOR_RXRST	0x20000000
#define SSIDMCOR_RPTMD	0x00000004
#define SSIDMCOR_TRMD	0x00000002
#define SSIDMCOR_DMEN	0x00000001

#define SSIDMINTSR_BLKEND	0x10
#define SSIDMINTSR_BLKNEND	0x08
#define SSIDMINTSR_DMEND	0x04
#define SSIDMINTSR_TXFIFOFUL	0x02
#define SSIDMINTSR_RXFIFOEMP	0x01

#define SSIDMINTMR_BLKENDM	0x10
#define SSIDMINTMR_BLKNENDM	0x08
#define SSIDMINTMR_DMENDM	0x04
#define SSIDMINTMR_TXFIFOFULM	0x02
#define SSIDMINTMR_RXFIFOEMPM	0x01

struct ssi_dmac {
	int irq;
	int ref;
	char *name;
} ssi_dmacs[2] = {
	{
		.irq = SSIDMA0_IRQ,
		.name = "SSIDMA0"
	}, {
		.irq = SSIDMA1_IRQ,
		.name = "SSIDMA1"
	}
};

struct sh7764_ssidma {
	unsigned long mmio[2];
	unsigned int intr_offset;
	struct ssi_dmac *dmac;

	struct snd_pcm_substream *tx_ss;
	unsigned long tx_period_size;
	unsigned int  tx_period;

	struct snd_pcm_substream *rx_ss;
	unsigned long rx_period_size;
	unsigned int  rx_period;
	unsigned long buffer_size;
	unsigned long period_size;
	unsigned long vperiod;

} sh7764_ssidma_data[6] = {
	{
		.mmio		=	{0xff401000, 0xff401000},
		.intr_offset	=	0,
		.dmac		=	&ssi_dmacs[0],
	}, {
		.mmio		=	{0xff401060, 0xff401000},
		.intr_offset	=	8,
		.dmac		=	&ssi_dmacs[0],
	}, {
		.mmio		=	{0xff4010c0, 0xff401000},
		.intr_offset	=	16,
		.dmac		=	&ssi_dmacs[0],
	},
};

inline u32 ssidma_reg(struct sh7764_ssidma *sd, u32 reg)
{
	int n = 0;

	switch (reg) {
	case SSIDMMR:
	case SSIRDMADR:
	case SSIRDMCNTR:
	case SSIWDMADR:
	case SSIWDMCNTR:
	case SSIDMCOR:
	case SSISTPBLCNT:
	case SSISTPDR:
	case SSIBLCNTSR:
	case SSIBLCNT:
	case SSIBLNCNTSR:
	case SSIBLNCNT:
		break;
	case SSIDMAOR:
	case SSIDMINTSR:
	case SSIDMINTMR:
		n = 1;
		break;
	default:
		BUG();
	}
	return sd->mmio[n] + reg;
}

inline u32 ssidma_inl(struct sh7764_ssidma *sd, u32 reg)
{
	return ctrl_inl(ssidma_reg(sd, reg));
}

inline void ssidma_outl(struct sh7764_ssidma *sd, u32 val, u32 reg)
{
	ctrl_outl(val, ssidma_reg(sd, reg));
}

static struct snd_pcm_hardware sh7764_ssidma_hardware = {
	.info		= (SNDRV_PCM_INFO_MMAP | 
			   SNDRV_PCM_INFO_INTERLEAVED |
			   SNDRV_PCM_INFO_BLOCK_TRANSFER |
			   SNDRV_PCM_INFO_RESUME |
			   SNDRV_PCM_INFO_MMAP_VALID),
	.formats	= SNDRV_PCM_FMTBIT_S16_LE,
	.rates		= (SNDRV_PCM_RATE_8000 |
			   SNDRV_PCM_RATE_22050 |
			   SNDRV_PCM_RATE_32000 |
			   SNDRV_PCM_RATE_44100 |
			   SNDRV_PCM_RATE_48000 |
			   SNDRV_PCM_RATE_96000),
	.rate_min	= 8000,
	.rate_max	= 96000,
	.channels_min	= 2,
	.channels_max	= 2,
	.buffer_bytes_max	= 65536,
	.period_bytes_min	= 1024, /* or 4096 */
	.period_bytes_max	= 4096,
	.periods_min		= 1,
	.periods_max		= 1024,
};

static inline void ssidma_dump_regs(struct sh7764_ssidma *sd)
{
#ifdef DEBUG
	int i;
	struct {char *name; u32 reg;} pairs[] = {
#define ENTRY(x) {#x, sd->mmio[0] + SSI##x}
		ENTRY(DMMR),
		ENTRY(RDMADR),
		ENTRY(RDMCNTR),
		ENTRY(WDMADR),
		ENTRY(WDMCNTR),
		ENTRY(DMCOR),
		ENTRY(STPBLCNT),
		ENTRY(STPDR),
		ENTRY(BLCNTSR),
		ENTRY(BLCNT),
		ENTRY(BLNCNTSR),
		ENTRY(BLNCNT),
#undef ENTRY
#define ENTRY(x) {#x, sd->mmio[1] + SSI##x}
		ENTRY(DMAOR),
		ENTRY(DMINTSR),
		ENTRY(DMINTMR),
#undef ENTRY
#if 0
#define ENTRY(x) {#x, ssi_base + x}
		ENTRY(SSICR),
		ENTRY(SSISR),
		ENTRY(SSITDR),
		ENTRY(SSIRDR),
#undef ENTRY
#endif
	};
	for (i = 0; i < ARRAY_SIZE(pairs); i++) {
		char *name = pairs[i].name;
		u32 reg = pairs[i].reg;
		int n;

		n = printk("%08x:%s", reg, name);
		n = 24 - n;
		while (n--)
		  printk(" ");
		printk("%08x\n", ctrl_inl(reg));
	}
#endif
}

static void ssidma_intr1(struct sh7764_ssidma *sd, u32 intsr)
{
	printk("INT: %02x {,N}CNT=%08x %08x\n", intsr, ssidma_inl(sd, SSIBLCNT), ssidma_inl(sd, SSIBLNCNT));

	if (intsr & SSIDMINTSR_DMEND) {
		ssidma_outl(sd, ssidma_inl(sd, SSIDMCOR) | SSIDMCOR_DMEN,
			    SSIDMCOR);
	}
	
	if (intsr & SSIDMINTSR_BLKNEND) {
		if (ssidma_inl(sd, SSIDMCOR) & SSIDMCOR_TRMD) {
			if (sd->tx_ss)
				snd_pcm_period_elapsed(sd->tx_ss);
		} else {
			if (sd->rx_ss)
				snd_pcm_period_elapsed(sd->rx_ss);
		}
	}
}

static irqreturn_t sh7764_ssidma_intr(int irq, void *ptr)
{
	int i, index, ret = IRQ_NONE;
	u32 dmintsr, dmintmr;
	struct sh7764_ssidma *sd;

	if (irq == SSIDMA0_IRQ)
		index = 0;
	else {
		index = 3;
		printk(KERN_ERR "%s:%d not implemented\n", __func__, __LINE__);
		BUG();
	}
	
	sd = &sh7764_ssidma_data[index];
	dmintsr = ssidma_inl(sd, SSIDMINTSR);
	dmintmr = ssidma_inl(sd, SSIDMINTMR);
#if 0
	printk("%s: %08x/%08x => %08x\n",
	       __func__, dmintsr, dmintmr, dmintsr & ~dmintmr);
#endif
	dmintsr &= ~dmintmr;
	for (i = 0; i < 3; i++, sd++) {
		u32 intsr = 0x1f & (dmintsr >> sd->intr_offset);

		if (!intsr)
			continue;

		ret = IRQ_HANDLED;
		ssidma_intr1(sd, intsr);
		ssidma_outl(sd, intsr << sd->intr_offset, SSIDMINTSR);
	}
	return ret;
}

static int sh7764_ssidma_open(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_pcm_runtime *runtime = substream->runtime;
	unsigned int id = rtd->dai->cpu_dai->id;
	struct sh7764_ssidma *sd = &sh7764_ssidma_data[id];
	struct ssi_dmac *dmac = sd->dmac;
	int recv = substream->stream == SNDRV_PCM_STREAM_PLAYBACK ? 0:1;
	int err;
	u32 tmp;
	char *devname;

//	printk("%s:", __func__);

	snd_soc_set_runtime_hwparams(substream, &sh7764_ssidma_hardware);

	ssidma_outl(sd, 0, SSIDMMR);
	ssidma_outl(sd, 0, SSISTPDR);
	ssidma_outl(sd, SSIDMCOR_DMRST, SSIDMCOR);
	udelay(100);
	ssidma_outl(sd, SSIDMCOR_DMRST | SSIDMCOR_TXRST | SSIDMCOR_RXRST,
		    SSIDMCOR);

	udelay(100);

	ssidma_outl(sd, SSIDMCOR_RPTMD, SSIDMCOR);
	/* DMA Register  */
	ssidma_outl(sd, SSIDMMR_RDMBSZ(2) | SSIDMMR_WDMBSZ(2), SSIDMMR);
	ssidma_outl(sd, 0, SSIRDMADR);
	ssidma_outl(sd, 0, SSIRDMCNTR);
	ssidma_outl(sd, 0, SSIWDMADR);
	ssidma_outl(sd, 0, SSIWDMCNTR);
	ssidma_outl(sd, ssidma_inl(sd, SSIDMAOR) | 3, SSIDMAOR);
	/* Interrupt enable */
	tmp = ssidma_inl(sd, SSIDMINTMR) | (0x1F << sd->intr_offset);
	tmp &= ~((SSIDMINTMR_BLKNENDM|SSIDMINTMR_DMENDM) << sd->intr_offset);
	ssidma_outl(sd, tmp, SSIDMINTMR);


	if (dmac->ref == 0) {
		if ((err = request_irq(dmac->irq, sh7764_ssidma_intr,
				       IRQF_DISABLED, dmac->name, 0)) ) {
			printk(KERN_ERR "unable to request IRQ %d\n", dmac->irq);
			goto err;
		}
	}
	dmac->ref++;

	if (recv) {
		sd->rx_ss = substream;
		ssidma_outl(sd, ssidma_inl(sd, SSIDMCOR) & ~SSIDMCOR_TRMD,
			    SSIDMCOR);
	} else {
		sd->tx_ss = substream;
		ssidma_outl(sd, ssidma_inl(sd, SSIDMCOR) | SSIDMCOR_TRMD,
			    SSIDMCOR);
	}
	if ((err = snd_pcm_hw_constraint_integer(
		     runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0 ) {
		goto err;
	}
//	printk(" 0\n");
	return 0;
#if 0
	return snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					  &hw_constraints_rates);
#endif
err:
//	printk(" %d\n", err);
	return err;
}

static int sh7764_ssidma_close(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	unsigned int id = rtd->dai->cpu_dai->id;
	struct sh7764_ssidma *sd = &sh7764_ssidma_data[id];
	struct ssi_dmac *dmac = sd->dmac;
	int recv = substream->stream == SNDRV_PCM_STREAM_PLAYBACK ? 0:1;

	ssidma_outl(sd,
		    ssidma_inl(sd, SSIDMINTMR) | (0x1f << sd->intr_offset),
		    SSIDMINTMR);

	if (recv)
		sd->rx_ss = NULL;
	else
		sd->tx_ss = NULL;

	if (--dmac->ref == 0)
		free_irq(dmac->irq, NULL);

//	printk("%s:\n", __func__);

	return 0;
}

static int sh7764_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *hw_params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssidma *cam = &sh7764_ssidma_data[rtd->dai->cpu_dai->id];
	int recv = substream->stream == SNDRV_PCM_STREAM_PLAYBACK ? 0:1;
	int ret;

	ret = snd_pcm_lib_malloc_pages(substream,
				       params_buffer_bytes(hw_params));
	if (ret < 0)
		return ret;

	return 0;
}

static int sh7764_hw_free(struct snd_pcm_substream *substream)
{
	return snd_pcm_lib_free_pages(substream);
}

static int sh7764_prepare(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssidma *sd = &sh7764_ssidma_data[rtd->dai->cpu_dai->id];
	const int blcntsr = 128;

	sd->buffer_size = snd_pcm_lib_buffer_bytes(substream);
	sd->period_size = snd_pcm_lib_period_bytes(substream);
	sd->vperiod = sd->buffer_size / sd->period_size;

	printk("%s: p_sz=%ld b_sz=%ld vp=%ld\n", __func__, sd->period_size, sd->buffer_size, sd->vperiod);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		ssidma_outl(sd, (u32)runtime->dma_area, SSIRDMADR);
		ssidma_outl(sd, sd->buffer_size, SSIRDMCNTR);
		ssidma_outl(sd, blcntsr, SSIBLCNTSR);
		ssidma_outl(sd, sd->period_size / blcntsr, SSIBLNCNTSR);
	} else {
		ssidma_outl(sd, (u32)runtime->dma_area, SSIWDMADR);
		ssidma_outl(sd, sd->buffer_size, SSIWDMCNTR);
		ssidma_outl(sd, blcntsr, SSIBLCNTSR);
		ssidma_outl(sd, sd->period_size / blcntsr, SSIBLNCNTSR);
	}
	ssidma_dump_regs(sd);
	return 0;
}

static void sh7764_start_dma(struct sh7764_ssidma *sd)
{
	u32 ssidmcor = ssidma_inl(sd, SSIDMCOR);

	if (ssidmcor & SSIDMCOR_DMEN)
		return;

	ssidma_outl(sd, ssidmcor | SSIDMCOR_DMEN, SSIDMCOR);
}

static void sh7764_stop_dma(struct sh7764_ssidma *sd)
{
	/*
	  Disable DMA module
	  * Renesas technical update *
	 */
#if 1
	u32 dmcor = ssidma_inl(sd, SSIDMCOR);

//	printk("%s: DMCOR=%08x >>\n", __func__, dmcor);
	
	ssidma_outl(sd, dmcor & ~(SSIDMCOR_RPTMD | SSIDMCOR_DMEN), SSIDMCOR);
#endif
}

static int sh7764_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssidma *sd = &sh7764_ssidma_data[rtd->dai->cpu_dai->id];

//	printk("%s; cmd=%d\n", __func__, cmd);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		sh7764_start_dma(sd);
		break;
	case SNDRV_PCM_TRIGGER_RESUME:
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		sh7764_stop_dma(sd);
		break;
	case SNDRV_PCM_TRIGGER_SUSPEND:
		printk("suspend DMA\n");
		break;
	default:
		pr_warning("?\n");
		return -EINVAL;
	}
	return 0;
}

static snd_pcm_uframes_t sh7764_pos(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssidma *sd = &sh7764_ssidma_data[rtd->dai->cpu_dai->id];
	int recv = substream->stream == SNDRV_PCM_STREAM_PLAYBACK ? 0:1;
	u32 count;
 	u32 blcntsr = ssidma_inl(sd, SSIBLCNTSR);
 	u32 blcnt = ssidma_inl(sd, SSIBLCNT);
 	u32 blncntsr = ssidma_inl(sd, SSIBLNCNTSR);
 	u32 blncnt = ssidma_inl(sd, SSIBLNCNT);
	snd_pcm_sframes_t frames;

	count = blncnt;
	count %= sd->vperiod;
	count *= blncntsr;
	count *= blcntsr;

	frames = bytes_to_frames(runtime, count);

	printk("POS: {,N}CNT=%08x %08x F=%ld c=%d\n", blcnt, blncnt, frames, count);

	return frames;
}

static struct snd_pcm_ops sh7764_ssidma_ops = {
	.open		= sh7764_ssidma_open,
	.close		= sh7764_ssidma_close,
	.ioctl		= snd_pcm_lib_ioctl,
	.hw_params	= sh7764_hw_params,
	.hw_free	= sh7764_hw_free,
	.prepare	= sh7764_prepare,
	.trigger	= sh7764_trigger,
	.pointer	= sh7764_pos,
};

static void sh7764_ssidma_free(struct snd_pcm *pcm)
{
	snd_pcm_lib_preallocate_free_for_all(pcm);
}

static int sh7764_ssidma_new(struct snd_card *card,
			     struct snd_soc_dai *dai,
			     struct snd_pcm *pcm)
{
	/* dont use SNDRV_DMA_TYPE_DEV, since it will oops the SH kernel
	 * in MMAP mode (i.e. aplay -M)
	 */
	snd_pcm_lib_preallocate_pages_for_all(pcm,
					      SNDRV_DMA_TYPE_CONTINUOUS,
					      snd_dma_continuous_data(GFP_KERNEL),
					      64*1024, 64*1024);

	return 0;
}

struct snd_soc_platform sh7764_soc_platform = {
	.name		= "sh7764-pcm",
	.pcm_ops 	= &sh7764_ssidma_ops,
	.pcm_new	= sh7764_ssidma_new,
	.pcm_free	= sh7764_ssidma_free,
};
EXPORT_SYMBOL_GPL(sh7764_soc_platform);

static int __init sh7764_soc_platform_init(void)
{
	return snd_soc_register_platform(&sh7764_soc_platform);
}
module_init(sh7764_soc_platform_init);

static void __exit sh7764_soc_platform_exit(void)
{
	snd_soc_unregister_platform(&sh7764_soc_platform);
}
module_exit(sh7764_soc_platform_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SH7764 Audio SSI-DMA driver");
MODULE_AUTHOR("ALPHAPROJECT");
